/**
 * XmlTransformer.java
 * @author Thomas Rowland
 * @version 02-08-03
 */
package xml.musicCatalog;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.BufferedWriter;
import java.io.File;


/*
 * This class demonstrates how to use XSLT to perform transformations,
 * for example XML to HTML, or writing an XML document back out to file.
 * This program uses the DOM Level 2 api and JAXP 1.2.
 * You need to have jaxp-api.jar, xalan.jar in your classpath.
 */

public class XmlTransformer {

    /*
     *	Private constructor - no instances of this class should be created.
     *	All public methods are decared as static.
     */
    private XmlTransformer() {
    }


    /*
     * Writes a DOM Document to an OutputStream. Demonstrates how easy it
     * is to update an XML file or create a new XML file after modification.
     * @args document a DOM Document
     * @args uri output to write the XML to
     */
    public static void writeXml(Document document, String uri)
            throws Exception {
        try {
            File out = new File(uri);

            // Obtain a Transformer
            TransformerFactory tf = TransformerFactory.newInstance();
            Transformer t = tf.newTransformer();

            // Set some properties if desired
            t.setOutputProperty(OutputKeys.INDENT, "yes");

            // Create a source object
            DOMSource source = new DOMSource(document.getDocumentElement());

            // Create a result object and perform the transform
            StreamResult result = new StreamResult(out);
            t.transform(source, result);
        } catch (TransformerConfigurationException e) {
            // wrapped exception generated by the TransformerFactory
            Throwable ex = e.getException();
            String msg = null;
            if (ex != null) {
                ex.printStackTrace();
                msg = ex.getMessage();
            }
            throw new Exception(
                    "** TransformerConfigurationException:\n"
                    + e.getMessage());
        } catch (TransformerException e) {
            // wrapped exception generated by the Transformer
            Throwable ex = e.getException();
            String msg = null;
            if (ex != null) {
                ex.printStackTrace();
                msg = ex.getMessage();
            }
            throw new Exception(
                    "** TransformerException:\n"
                    + e.getMessage());
        }
    }


    /*
     * Transforms a DOM Document into another form, specified by its stylesheet.
     * @args node the top level DOM node reference
     * @args stylesheet the file containing the transform definitions
     * @args out the output destination
     */
    public static void transform(Node node, File stylesheet, BufferedWriter out) {
        try {
            // Instantiate the TransformerFactory
            TransformerFactory tf = TransformerFactory.newInstance();

            // Create a source object from the stylesheet
            StreamSource ss = new StreamSource(stylesheet);

            // Obtain a Transformer
            Transformer t = tf.newTransformer(ss);

            // Create a source object from the DOM document
            DOMSource ds = new DOMSource(node);

            // Create a result object from the BufferedWriter
            StreamResult result = new StreamResult(out);

            // Perform the transform
            t.transform(ds, result);
        } catch (TransformerConfigurationException e) {
            // Exception generated by the TransformerFactory
            System.out.println("\n** Transformer Factory error: "
                    + e.getMessage());
            //get the wrapped exception
            Throwable th = e;
            if (e.getException() != null)
                th = e.getException();
            th.printStackTrace();
        } catch (TransformerException e) {
            // Exception generated by the transformer
            System.out.println("\n** Transformation error: "
                    + e.getMessage());
            //get the wrapped exception
            Throwable th = e;
            if (e.getException() != null)
                th = e.getException();
            th.printStackTrace();
        }
    }

}//