/**
 * MusicCatalog.java
 * @author Thomas Rowland
 * @version 02-08-03
 */
package xml.musicCatalog;

import futils.Futil;

import java.io.File;

/*
 * This class acts as a driver for demonstrating different types of
 * parsing XML files, based on the MusicCatalog examples.
 *
 * The following files are used with these examples:
 * MusicCatalog.xml  - an XML file.
 * MusicCatalog1.xml - same XML file but containing a reference to a DTD.
 * MusicCatalog2.xml - same XML file but containing a reference to an XMLSchema.
 * MusicCatalog.dtd  - an XML DTD containing validation constraints.
 * MusicCatalog.xsd  - an XMLSchema validation constraints.
 * MusicCatalog.xsl  - a Stylesheet containing definitions for transformation to HTML.
 *
 */

public class MusicCatalog {

    public static void main(String argv []) {
        MusicCatalog music = new MusicCatalog();

        //Test (1) -- parse an xml file using the Xerces parser api directly
        music.parseTest();

        //Test (2) -- parse an xml file using JAXP
        music.parseTest2();

        //Test (3) -- parse an xml file with DTD validation using JAXP
        music.dtdTest();

        //Test (4) -- parse an xml file with XMLSchema validation using JAXP
        music.xsdTest();

        //Test (5) -- add a new entry to the MusicCatalog and update the XML file
        music.modTest();

        //Test (6) -- add a new entry to the MusicCatalog and update the XML file
        music.htmlTest();

        System.out.println("\nEnd.");
    }


    /*
     * Shows how to parse an XML file using the Xerces api.
     * No Validation. Does not use JAXP.
     */
    public void parseTest() {
        try {
            System.out.println("\nParsing XML file using Xerces implementation api, No Validation...");
            String file = selectFile("Select MusicCatalog.xml for No Validation.");
            System.out.println("Selected:\n" + file);
            MyDomParser.parse(file, System.out);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
    }

    /*
     * Shows how to parse an XML file using JAXP. No Validation.
     */
    private MyJaxpDomParser parseTest2() {
        MyJaxpDomParser dp = new MyJaxpDomParser();
        ;
        try {
            System.out.println("\nParsing XML file using JAXP api, No Validation...");
            String file = selectFile("Select MusicCatalog.xml for No Validation.");
            System.out.println("Selected:\n" + file);
            dp.parse(file, System.out);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
        return dp;

    }

    /*
     * Shows how to parse an XML file with DTD validation, using JAXP.
     */
    private MyJaxpDomParser dtdTest() {
        MyJaxpDomParser dp = new MyJaxpDomParser();
        ;
        try {
            System.out.println("\nParsing XML file with DTD validation, using JAXP...");
            String file = selectFile("Select MusicCatalog1.xml for DTD Validation.");
            System.out.println("Selected:\n" + file);
            dp.dtdParse(file, System.out);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
        return dp;
    }

    /*
     * Shows how to parse an XML file with XMLSchema validation, using JAXP.
     */
    private MyJaxpDomParser xsdTest() {
        MyJaxpDomParser dp = new MyJaxpDomParser();
        ;
        try {
            System.out.println("\nParsing XML file with XSD validation, using JAXP...");
            String file = selectFile("Select MusicCatalog2.xml for XSD Validation.");
            System.out.println("Selected:\n" + file);
            dp.xsdParse(file, System.out);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
        return dp;
    }

    /*
     * Shows how to modify content of an XML file by adding a new
     * <Item> entry to the MusicCatalog, and updating the XML file
     * using XSLT transforms.
     */
    private void modTest() {
        try {
            MyJaxpDomParser dp;
            // you can use any of these for testing...
            //dp = parseTest2();
            //dp = dtdTest();
            dp = xsdTest();

            Item item = dp.createItem();
            item.setMedia("cd");
            item.setArtist("Aaron Copeland");
            item.setTitle("Fanfare");
            item.setYear("1970");
            item.setMember("Aaron Copeland");
            item.setMember("the rest of the band");
            dp.addItem(item);

            // write the xml back out to the file
            dp.toXml();
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
    }

    /**
     * Shows how to transform an XML file into HTML
     * using XSLT transforms.
     */
    private void htmlTest() {
        try {
            MyJaxpDomParser dp;
            // you can use any of these for testing...
            dp = parseTest2();
            //dp = dtdTest();
            //dp = xsdTest();

            File stylesheet = new File(selectFile(
                    "Select a stylesheet."));

            dp.toHtml(stylesheet);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }
    }

    /**
     * Utility function - Prompts for an XML file to parse
     */
    private String selectFile(String prompt) {
        return Futil.getReadFile(prompt).getAbsolutePath();
    }

}//