/**
 * XsltXml2Html.java
 * @author Thomas Rowland
 * @version 02-08-03
 */
package xml.classInfo;

import futils.Futil;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

/*
 * This program demonstrates how to use XSLT Transforms
 * to transform an XML document into HTML (or other format).
 */

public class XsltXml2Html {

    public static void main(String[] args) {
        try {
            // Select an XML file and associated Stylesheet as inputs
            File xmlFile = Futil.getReadFile("select an XML file");
            File stylesheet = Futil.getReadFile("select a stylesheet");

            // Define the output to write to
            String fn = xmlFile.getName();
            fn = fn.substring(0, fn.lastIndexOf("."));
            File htmlFile = new File(
                    xmlFile.getParent() + "\\" + fn + ".html");
            BufferedWriter out = new BufferedWriter(
                    new FileWriter(htmlFile));

            // Parse the XML and pass the DOM Document for transformation
            Document document = parse(xmlFile);
            transform(document, stylesheet, out);
            out.flush();
            out.close();
            System.out.println("HTML output written to:\n" + htmlFile.toString());
        } catch (IOException ioe) {
            ioe.printStackTrace();
        }
    }


    /*
     * Parses an XML file and returns the DOM Document
     */
    private static Document parse(File xmlFile) throws IOException {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            //factory.setNamespaceAware(true);
            //factory.setValidating(true);

            DocumentBuilder db = factory.newDocumentBuilder();
            Document document = db.parse(xmlFile);
            return document;
        } catch (SAXException e) {
            // Parsing error. Get the wrapped exception
            Exception x = e;
            if (e.getException() != null)
                x = e.getException();
            x.printStackTrace();
        } catch (ParserConfigurationException e) {
            // Factory unable to create parser
            e.printStackTrace();
        }
        return null;
    }


    /*
     * Transforms a DOM Document into the form specified by the stylesheet,
     * and writes the result to the specified output.
     */
    public static void transform(Node node, File stylesheet, BufferedWriter out) {
        try {
            // Instantiate the TransformerFactory
            TransformerFactory tf = TransformerFactory.newInstance();

            // Create a source object from the stylesheet
            StreamSource ss = new StreamSource(stylesheet);

            // Obtain a Transformer
            Transformer t = tf.newTransformer(ss);

            // Create a source object from the DOM document
            DOMSource ds = new DOMSource(node);

            // Create a result object from the BufferedWriter
            StreamResult result = new StreamResult(out);

            // Perform the transform
            t.transform(ds, result);
        } catch (TransformerConfigurationException tce) {
            // Exception generated by the TransformerFactory
            System.out.println("\n** Transformer Factory error");
            System.out.println(" " + tce.getMessage());
            // Get the wrapped Throwable exception
            Throwable x = tce;
            if (tce.getException() != null)
                x = tce.getException();
            x.printStackTrace();
        } catch (TransformerException te) {
            // Exception generated by the transformer
            System.out.println("\n** Transformation error");
            System.out.println(" " + te.getMessage());
            // Get the wrapped Throwable exception
            Throwable x = te;
            if (te.getException() != null)
                x = te.getException();
            x.printStackTrace();
        }
    }

}//