package sound.spectrogram;

import gui.layouts.VerticalLayout;
import gui.run.RunButton;
import gui.run.RunRadio;
import gui.run.RunSpinner;
import gui.run.RunSpinnerSlider;
import math.fourierTransforms.r2.FFT1dDouble;
import sound.Oscillator;
import sound.OscopePanel;
import sound.SoundUtils;
import sound.ulaw.UlawCodec;

import javax.swing.*;
import java.awt.*;

/**
 * Created by Robert Distinti.
 * Pitch shifter code reused for final
 * User: default
 * Date: Jul 23, 2005
 * Time: 11 aM
 */
public class MyPlayer extends JDialog {
    private OscopePanel op = new OscopePanel();
    private RunSpinnerSlider slider;
    private double[] originalData = new Oscillator(440, 1000).getSineWave();  // set default data
    private int originalLength = originalData.length;
    private double[] filteredData;
    private UlawCodec playData;
    private RunRadio isPSD;
    private RunRadio isReal;
    private RunRadio isImag;
    private ButtonGroup displayGroup = new ButtonGroup();
    private RunRadio isDougs;
    private RunRadio isBobs;
    private ButtonGroup shiftGroup = new ButtonGroup();
    private int sampleRate = 8000;        //todo needs to be wired into load

    private double[] copy(double []a) {
        double out[] = new double[a.length];
        System.arraycopy(a, 0, out, 0, a.length);
        return out;
    }

    private double logBase2(int length) {
        return (Math.log(length) / Math.log(2));
    }

    private double[] makePowerOfTwo(double[] in) {
        int n = (int) Math.ceil(logBase2(in.length));
        double out[] = new double[1 << n];
        System.arraycopy(in, 0, out, 0, originalLength);
        return out;
    }

    private void scaleData(double scale_) {
        double out[] = new double[originalData.length];
        for (int i = 0; i < originalData.length; i++) {
            out[i] = originalData[i] * scale_;
        }
        originalData = out;
    }

    private void record() {
        JDialog f = new JDialog(this, "Capture/Playback", true);
        SimplifiedCapturePlayBackPanel capturePlayback = new SimplifiedCapturePlayBackPanel(f);
        capturePlayback.open();
        f.getContentPane().add("Center", capturePlayback);
        f.pack();
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int w = 720;
        int h = 340;
        f.setLocation(screenSize.width / 2 - w / 2,
                screenSize.height / 2 - h / 2);
        f.setSize(w, h);

        f.setVisible(true);
        int data[] = capturePlayback.getData();
        if (data != null && data.length > 32) {
            sampleRate = (int) capturePlayback.getAudioInputStream().getFormat().getSampleRate();
            originalData = new double[data.length];
            int max = 0;
            // first find max value
            for (int i = 0; i < data.length; i++) {
                if (Math.abs(data[i]) > max) max = Math.abs(data[i]);
            }
            double scale = 0.48 / max;
            for (int i = 0; i < data.length; i++) {
                originalData[i] = data[i] * scale;
            }
            dataChanged();
        }

    }

    private void open() {
        UlawCodec ulc = new UlawCodec();
        sampleRate = 8000;
        originalData = copy(ulc.getDoubleArray());
        System.out.println("number of samples " + originalData.length);
        dataChanged();

    }

    private void play() {
        if (playData != null) {
            playData.play();
        }
    }

    public void addNoise(double d[], double amp) {
        for (int i = 0; i < originalLength; i++) {
            d[i] = d[i] + amp * (Math.random() - 0.5);
        }
    }

    private void dataChanged() {
        originalLength = originalData.length;
        originalData = makePowerOfTwo(originalData);
        shiftChanged();
    }


    private void shiftChanged() {

        double imag[] = new double[originalData.length];
        double real2[] = new double[originalData.length];
        double imag2[] = new double[originalData.length];
        double real[] = copy(originalData);

        // transform
        FFT1dDouble f = new FFT1dDouble();
        f.computeForwardFFT(real, imag);

        // perform scaling
        double r[] = f.getRealData();
        double im[] = f.getImaginaryData();
        for (int i = 0; i < r.length; i++) {
            r[i] /= (double) originalLength;
            im[i] /= (double) originalLength;
        }

        final int lengthMinusOne = (r.length - 1);
        if (isDougs.isSelected()) {
            // this method assumes all data is symetical about center
            // it simply uses the data from 0 to N/2 for both halves
            int len = r.length / 2;
            int shift = slider.getValue();
            int destIdxL = 0;
            int destIdxH = lengthMinusOne;
            int srcIdx = (len - 1) * shift / 100;
            int cnt = len - srcIdx;
            for (int i = 0; i < cnt; i++) {
                real2[destIdxH] = real2[destIdxL] = r[srcIdx];
                imag2[destIdxH] = imag2[destIdxL] = im[srcIdx];
                // test to see if inverting the reflected imag data would solve problem
                //imag2[destIdxH]=-im[srcIdx];
                //imag2[destIdxL]=im[srcIdx];
                // end test code 
                destIdxH--;
                destIdxL++;
                srcIdx++;
            }
        } else if (isBobs.isSelected()) {
            System.out.println("using bobs algorithm to shift)");
            // this does not assume data symetry
            int len = r.length / 2;
            int shift = slider.getValue();
            int destIdxL = 0;
            int destIdxH = lengthMinusOne;
            final int shiftAmount = (len - 1) * shift / 100;
            int srcIdxL = shiftAmount;
            int srcIdxH = lengthMinusOne - shiftAmount;
            int cnt = len - srcIdxL;
            for (int i = 0; i < cnt; i++) {
                if (destIdxL < 0 || srcIdxL < 0 || srcIdxH < 0 || destIdxH < 0) {
                    continue;
                }
                real2[destIdxL] = r[srcIdxL];
                real2[destIdxH] = r[srcIdxH];
                imag2[destIdxH] = im[srcIdxH];
                imag2[destIdxL] = im[srcIdxL];
                System.out.println("destIdxL=" + destIdxL + " srcIdxL=" + srcIdxL);
                System.out.println("destIdxH=" + destIdxH + " srcIdxH=" + srcIdxH);
                destIdxH--;
                destIdxL++;
                srcIdxL++;
                srcIdxH--;
            }
        } else {
            // just pass through
            real2 = r;
            imag2 = im;
        }
        // convert back
        f.computeBackwardFFT(real2, imag2);
        filteredData = f.getRealData();
        displayChanged();
    }


    private void displayChanged() {
        double out[] = new double[originalLength];
        double imag[] = new double[filteredData.length];
        System.arraycopy(filteredData, 0, out, 0, originalLength);
        double[] displayData = out;
        if (isPSD.isSelected()) {
            out = copy(filteredData);
            FFT1dDouble f = new FFT1dDouble();
            f.computeForwardFFT(out, imag);
            double psd[] = f.getPSD();
            op.setData(psd);
        } else if (isImag.isSelected()) {
            out = copy(filteredData);
            FFT1dDouble f = new FFT1dDouble();
            f.computeForwardFFT(out, imag);
            double d[] = f.getImaginaryData();
            // need to scale it
            for (int i = 0; i < d.length; i++) {
                d[i] /= (double) originalLength;
            }
            op.setData(d);
        } else if (isReal.isSelected()) {
            out = copy(filteredData);
            FFT1dDouble f = new FFT1dDouble();
            f.computeForwardFFT(out, imag);
            double d[] = f.getRealData();
            // need to scale it
            for (int i = 0; i < d.length; i++) {
                d[i] /= (double) originalLength;
            }
            op.setData(d);
        } else {
            op.setData(displayData);
        }
        playData = new UlawCodec(displayData);
    }


    public MyPlayer() {
        setModal(false);
        setTitle("Sound Thingy -- by Robert Distinti");
        JPanel jp = new JPanel();
        Container c = getContentPane();
        c.setLayout(new BorderLayout());
        c.add(jp, BorderLayout.NORTH);
        c.add(op, BorderLayout.CENTER);

        JPanel jp0 = new JPanel();
        jp.add(jp0);
        jp0.setLayout(new VerticalLayout());
        jp0.add(new RunButton("[Open") {
            public void run() {
                open();
            }
        });
        jp0.add(new RunButton("[Record") {
            public void run() {
                record();
            }
        });

        jp0.add(new RunButton("[Play") {
            public void run() {
                play();
            }
        });


        JPanel jp2 = new JPanel();
        jp.add(jp2);
        jp2.setBorder(BorderFactory.createLoweredBevelBorder());
        jp2.setLayout(new VerticalLayout());
        jp2.add(new JLabel("Display Control"));
        RunRadio normal;
        jp2.add(normal = new RunRadio("[Normal") {
            public void run() {
                displayChanged();
            }
        });
        jp2.add(isPSD = new RunRadio("PS[D") {
            public void run() {
                displayChanged();
            }
        });
        jp2.add(isImag = new RunRadio("[Imag") {
            public void run() {
                displayChanged();
            }
        });
        jp2.add(isReal = new RunRadio("[Real") {
            public void run() {
                displayChanged();
            }
        });

        jp.add(new SpectrogramControlPanel());


        JPanel jp3 = new JPanel();
//        jp.add(jp3);             // IF you want to select the pitch algorith then uncomment this line
        jp3.setLayout(new VerticalLayout());
        jp3.setBorder(BorderFactory.createEtchedBorder());
        jp3.add(new JLabel("Pitch Algorthm"));

        RunRadio none;
        jp3.add(none = new RunRadio("No Shift") {
            public void run() {
                shiftChanged();
            }
        });
        jp3.add(isDougs = new RunRadio("Use Doug's") {
            public void run() {
                shiftChanged();
            }
        });
        jp3.add(isBobs = new RunRadio("Use Bob's") {
            public void run() {
                shiftChanged();
            }
        });


        jp3 = new JPanel();
        jp.add(jp3);
        jp3.setLayout(new VerticalLayout());
        jp3.setBorder(BorderFactory.createEtchedBorder());
        jp3.add(new JLabel("Pitch Shift"));
        jp3.add(slider = new RunSpinnerSlider(new SpinnerNumberModel(0, 0, 100, 10)) {
            public void run() {
                shiftChanged();
            }
        });

        originalData = new Oscillator(440, 2048).getSineWave();
        scaleData(0.5);
        dataChanged();

        displayGroup.add(normal);
        displayGroup.add(isPSD);
        displayGroup.add(isImag);
        displayGroup.add(isReal);
        normal.setSelected(true);

        shiftGroup.add(none);
        shiftGroup.add(isDougs);
        shiftGroup.add(isBobs);
        isBobs.setSelected(true);


        setSize(800, 600);
        setVisible(true);
    }


    public static void main(String[] args) {
        new MyPlayer();
    }

    private class SpectrogramControlPanel extends JPanel {
        private JRadioButton isRect = new JRadioButton("Rect");
        private JRadioButton isHanning = new JRadioButton("Hanning");
        private JRadioButton isBartlett = new JRadioButton("Bartlett");
        private JRadioButton isLyon = new JRadioButton("Lyon");
        private JCheckBox isLog = new JCheckBox("log scale");
        private ButtonGroup windowGroup = new ButtonGroup();
        private RunSpinner2 windowSize = new RunSpinner2(new SpinnerNumberModelPow2(256, 16, 1 << 16, 1)) {
            public void run() {
            }

            ;};
        private RunSpinner overLap = new RunSpinner(new SpinnerNumberModel(50, 10, 90, 5)) {
            public void run() {
            }

            ;};

        public SpectrogramControlPanel() {
            this.setBorder(BorderFactory.createLoweredBevelBorder());
            this.setLayout(new BorderLayout());
            this.add(new JLabel(" Spectrogram Control"), BorderLayout.NORTH);

            JPanel jp0 = new JPanel();
            this.add(jp0, BorderLayout.WEST);
            jp0.setLayout(new VerticalLayout());
            jp0.add(isRect);
            windowGroup.add(isRect);
            jp0.add(isHanning);
            windowGroup.add(isHanning);
            jp0.add(isBartlett);
            windowGroup.add(isBartlett);
            jp0.add(isLyon);
            windowGroup.add(isLyon);
            isRect.setSelected(true);

            JPanel jp1 = new JPanel();
            jp1.add(windowSize);
            jp1.add(new JLabel(" Window Size"));

            JPanel jp2 = new JPanel();
            jp2.add(overLap);
            jp2.add(new JLabel(" OverLap (%)"));

            jp0 = new JPanel();
            this.add(jp0, BorderLayout.EAST);
            jp0.setLayout(new VerticalLayout());
            jp0.add(jp1);
            jp0.add(jp2);
            jp0.add(isLog);
            jp0.add(new RunButton("Create") {
                public void run() {
                    createSpectrogram();
                }
            });

        }

        String windowName = "";

        private double[] getWindow() {
            int size;
            Object v = windowSize.getValue();
            if (v instanceof Integer) {
                size = ((Integer) v).intValue();
            } else {
                size = 128;
            }
            if (isBartlett.isSelected()) {
                windowName = "Bartlett";
                return SoundUtils.makeBartlett(size);
            } else if (isHanning.isSelected()) {
                windowName = "Hanning";
                return SoundUtils.makeHanning(size);
            } else if (isLyon.isSelected()) {
                windowName = "Lyon";
                return SoundUtils.makeLyon(size);
            } else if (isHanning.isSelected()) {
                windowName = "Hanning";
                return SoundUtils.makeHanning(size);
            } else if (isHanning.isSelected()) {
                windowName = "Hanning";
                return SoundUtils.makeHanning(size);
            } else if (isHanning.isSelected()) {
                windowName = "Hanning";
                return SoundUtils.makeHanning(size);
            }
            windowName = "Rectangular";
            double w[] = new double[size];
            for (int i = 0; i < size; i++) {
                w[i] = 1.0;
            }
            return w;
        }

        private int getOverlap() {
            int size;
            Object v = overLap.getValue();
            if (v instanceof Integer) {
                size = ((Integer) v).intValue();
            } else {
                size = 50;    // default to 50 %
            }
            return size;
        }

        private void createSpectrogram() {
            // first make the window
            double[] window = getWindow();
            double data[] = copy(filteredData); // make copy of filtered data
            int wSize = window.length;
            int dSize = originalLength;           // the length of the original data
            int olap = getOverlap();

            String info = "Spectrogram: " + windowName + "(" + wSize + ") Overlap(" + olap + "%) Samples(" + dSize + ") Rate(" + sampleRate + ") PitchShift(" + slider.getValue() + ")";

            // Compute horizontal values
            int windowAdvance = wSize - wSize * olap / 100;
            if (windowAdvance < 1) windowAdvance = 1;    // just to make sure
            int numWin = (dSize - wSize) / windowAdvance + 1;  // number of windows
            int numPixH = numWin;
            int hScale = 1;
            while (numPixH < 512) {
                hScale = hScale * 2;
                numPixH = numWin * hScale;
            }
            double periodPerPixH = (double) dSize / sampleRate / numPixH;

            // compute vertical values
            int numDisplayBins = wSize / 2; // the number that will be displayed
            int numPixV = numDisplayBins;
            int vScale = 1;
            while (numPixV < 256) {
                vScale = vScale * 2;
                numPixV = numDisplayBins * vScale;
            }
            double freqPerPixV = (double) sampleRate / 2 / numPixV;

            // make the array based on above
            double b[][] = new double[numPixH][numPixV];  //width, hight
            int idxH = 0;

            // Perform the Spectrogram (with expand)
            for (int n = 0; n < numWin; n++) {
                // isolate the data
                double windowData[] = new double[wSize];
                double windowImag[] = new double[wSize];
                int sidx = n * windowAdvance;
                for (int i = 0; i < wSize; i++, sidx++) {
                    windowData[i] = data[sidx] * window[i] / wSize;
                }
                FFT1dDouble f = new FFT1dDouble();
                f.computeForwardFFT(windowData, windowImag);
                double psd[] = f.getPSD();
                for (int i = 0; i < numPixV; i++) {
                    int ii = numPixV - i - 1; // invert the vertical data
                    double Z = psd[i / vScale];
                    for (int q = 0; q < hScale; q++) {
                        b[idxH + q][ii] = Z;
                    }
                }
                idxH += hScale;
            }

            SpectroGramContainer scont = new SpectroGramContainer(b, periodPerPixH, freqPerPixV);
            new mySpectrogramViewer(info, scont, isLog.isSelected());

        }


    }// end subclass


} // end class


