package net.stocks;

import utils.StringUtils;

import java.beans.XMLEncoder;
import java.io.*;
import java.net.URL;
import java.util.Calendar;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.Vector;


public class Quote {
    private String symbol;
    private float price;
    private Date date;
    private float change;
    private float open;
    private float bid;
    private float ask;
    private long volume;

    public Quote(String s) {
        // assume that the quote is of the form
        // symbol	price	date	time  (20 min. Delay)	  change	  open	  bid	  ask	  volume
        // "AAPL",55.31,"11/3/2004","4:00pm",+1.81,54.44,56.11,53.99,21504764
        StringTokenizer st = new StringTokenizer(s, ",");
        symbol = getSymbol(st);
        price = getPrice(st);
        date = getDate(getSymbol(st), getSymbol(st));
        change = getPrice(st);
        open = getPrice(st);
        bid = getPrice(st);
        ask = getPrice(st);
        volume = getLong(st);
    }

    private static String getSymbol(StringTokenizer st) {
        return trimQuotes(st.nextToken());
    }

    private float getPrice(StringTokenizer st) {
        final String s = getSymbol(st);
        if (s.indexOf("N/A") == -1)
            return Float.valueOf(s).floatValue();
        return Float.NaN;
    }

    private static long getLong(StringTokenizer st) {
        final String s = getSymbol(st);
        return Long.valueOf(s).longValue();
    }

    private static int getInt(StringTokenizer st) {
        final String s = getSymbol(st);
        return Integer.valueOf(s).intValue();
    }

    private static int getInt(String s) {
        int i = 0;
        try {
            i = Integer.valueOf(s).intValue();
        } catch (NumberFormatException e) {
            System.out.println("err:" + s + " not an int");
            e.printStackTrace();

        }
        return i;
    }

    /*
    return a string with no quotes
    */
    private static String trimQuotes(String s) {
        //System.out.println("trim:"+s);
        return StringUtils.replaceAll(s, "\"", "");
    }
    public void print(Object o){
        System.out.println(o);
    }
    /*
    Parse two strings of the form:
    "11/3/2004","4:00pm"
    */
    private static Date getDate(String dateString, String time) {
        //System.out.println(dateString);
        StringTokenizer st = new StringTokenizer(dateString, "/");
        String symbol = getSymbol(st);
        //System.out.println("symbol=" + symbol);
        int month = getInt(symbol);
        symbol = getSymbol(st);
        int date = getInt(symbol);
        int year = getInt(getSymbol(st).substring(0, 4));
        st = new StringTokenizer(time, ":");

        int hour = getInt(st);
        String minuteString = st.nextToken("");
        boolean isPM = minuteString.indexOf("AM") == -1;
        if (isPM) hour = hour + 12; //use a 24 hour clock
        int minute = getMin(minuteString);
        Calendar c = Calendar.getInstance();
        // the month is zero based
        // i.e., Jan = 0.
        c.set(year, month - 1, date, hour, minute);
        return c.getTime();
    }

    private static int getMin(String minuteString) {
        String ms = StringUtils.replaceAll(minuteString.toUpperCase(), ":", "");
        ms = StringUtils.replaceAll(ms, "PM", "");
        ms = StringUtils.replaceAll(ms, "AM", "");
        ms = StringUtils.replaceAll(ms, "\"", "");
        return Integer.valueOf("00").intValue();
    }

    public static void main(String args[]) {
        testQuoteSystem();
    }

    public static void print(Object o[]) {
        for (int i = 0; i < o.length; i++)
            System.out.println(o[i]);
    }

    public String toXml() {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        XMLEncoder e = new XMLEncoder(baos);
        e.writeObject(this);
        e.flush();
        return baos.toString();
    }

    public String toString() {
        return symbol + " " + date + " " + price + " " + volume;
    }

    private static void testQuoteSystem() {
        Vector s = new Vector();
        s.addElement("aapl");
        s.addElement("xlnx");
        s.addElement("altr");
        s.addElement("mot");
        s.addElement("cy");
        s.addElement("crus");
        s.addElement("sfa");
        s.addElement("adbe");
        s.addElement("msft");
        s.addElement("sunw");
        String urlString = makeQuoteURLString(s);
        //System.out.println(urlString);
        Vector url = getUrl(urlString);
        //printVector(url);
        print(getQuotes(url));
    }

    public static Quote makeQuote(String symbol) {
        Vector s = new Vector();
        s.addElement(symbol);
        String urlString = makeQuoteURLString(s);
        Vector url = getUrl(urlString);
        Quote[] quotes = getQuotes(url);
        for (int i = 0; i < quotes.length; i++)
            if (quotes[i] != null) return quotes[i];
        return null;
    }

    public static Quote[] getQuotes(Vector v) {
        Quote qa [] = new Quote[v.size()];
        for (int i = 0; i < qa.length; i++) {
            String s = v.elementAt(i).toString();
            if (s.indexOf(",") == -1) continue;
            System.out.println(s);
            qa[i] = new Quote(s);
        }
        return qa;
    }

    public static void printVector(Vector v) {
        for (int i = 0; i < v.size(); i++) {
            System.out.println(v.elementAt(i));
        }
    }

    public static Vector getUrl(String _URLString) {
        Vector contents = new Vector();
        BufferedReader in = null;
        try {
            URL URLString = new URL(_URLString);
            InputStream is = URLString.openStream();
            in = new BufferedReader(new
                    InputStreamReader(is));
            String line;
            contents.addElement("Copyright 1999 by Doug Lyon http://www.docjava.com");
            contents.addElement("symbol\tprice\tdate\ttime" +
                    "  (20 min. Delay)\t  change\t  open\t  bid\t  ask\t  volume");
            while (null != (line = in.readLine())) contents.addElement(line);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (in != null) in.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return contents;
    }

    public static String makeQuoteURLString(Vector symbols) {
        String symbolsString = "";
        for (int i = 0; i < symbols.size(); i++) {
            String symbol = (String) symbols.elementAt(i);
            symbolsString += ((i != 0) ? "," : "") + symbol.toUpperCase();
        }
        return
                "http://216.109.124.128/download/javasoft.beans?SYMBOLS=" +
                        symbolsString +
                        "&format=sl";
    }

    public String getSymbol() {
        return symbol;
    }

    public float getPrice() {
        return price;
    }

    public Date getDate() {
        return date;
    }

    public float getChange() {
        return change;
    }

    public float getOpen() {
        return open;
    }

    public float getBid() {
        return bid;
    }

    public float getAsk() {
        return ask;
    }

    public long getVolume() {
        return volume;
    }
}
