/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * A utility class for numerical analysis.
 * This class cannot be subclassed or instantiated because all methods are static.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public final class Util {
  /** The default precision for numerical analysis. */
  public static final double defaultNumericalPrecision = Math.sqrt(Double.MIN_VALUE);

  private static SuryonoParser parser= new SuryonoParser(0);

  private Util() {} // prohibit instantiation because all methods are static

  /**
   * Method relativePrecision
   *
   * @param epsilon the absolute error
   * @param result  the result
   *
   * @return the relative error
   */
  public static double relativePrecision(final double epsilon,
                                         final double result) {
    return (result > defaultNumericalPrecision)
        ? epsilon / result
        : epsilon;
  }

  /**
   * Checks if an array is sorted.
   * Returns:
   *   Positive integer if array is sorted in increasing value.
   *   Negative integer if array is sorted in decreasing value.
   *   Zero if array is not sorted.
   *
   * @param array double[]
   * @return int 1,0,-1 based on sorting
   */
  public static int checkSorting(double[] array) {
    int sign = (array[0] <= array[array.length - 1]) ? 1 : -1;
    for (int i = 1, n = array.length; i < n; i++) {
      switch (sign) {
        case -1:
          if (array[i - 1] < array[i])return 0; // not sorted
          break;
        case 1:
          if (array[i - 1] > array[i])return 0; // not sorted
      }
    }
    return sign;
  }

  public static double[] getDomain(Function f, double a, double b, int n){
    double min=f.evaluate(a);
    double max=f.evaluate(a);
    double x=a, dx=(b-a)/(n-1);
    for(int i=1; i<n; i++){
      double y=f.evaluate(x);
      min=Math.min(min,y);
      max=Math.max(max,y);
      x+=dx;
    }
    return new double[] {min,max};
  }

  /**
   * Fills an array with f(x) values.
   *
   * @param f Function
   * @param start double
   * @param stop double
   * @param n int
   * @return double[][]
   */
  static public double[][] functionFill(Function f, double start, double stop,
                                        int n) {
    double[][] data = new double[2][n];
    return functionFill(f, start, stop, data);
  }

  /**
   * Fills an array with f(x) values.
   *
   * @param f Function
   * @param start double
   * @param stop double
   * @param data double[][]
   * @return double[][]
   */
  static public double[][] functionFill(Function f, double start, double stop, double[][] data) {
    double dx = 1;
    int n = data[0].length;
    if (n > 1) dx = (stop - start) / (n - 1);
    double x = start;
    for (int i = 0; i < n; i++) {
      data[0][i] = x;
      data[1][i] = f.evaluate(x);
      x += dx;
    }
    return data;
  }

  /**
   * Computes the average value of a subset of an array.
   *
   * @param array the data to be averaged
   * @param start the index of the first point to be averaged
   * @param num   the total number of points to be averaged
   * @return
   */
  public static double computeAverage(double[] array, int start, int num) {
    double sum = 0;
    for (int i = start, stop = start + num; i < stop; i++) {
      sum += array[i];
    }
    return sum / num;
  }


  /**
   * Creates a function having a constant value.
   * @param _c
   * @return
   */
  public static Function constantFunction(double _c) {
    final double c = _c;
    return new Function() {
      public double evaluate(double x) {
        return c;
      }
    };
  }

  /**
   * Evalautes a mathematical expression without variables.
   * @param str String
   * @return double
   */
  public static synchronized double evalMath(String str){
    try {
      parser.parse(str);
      return parser.evaluate();
    }
    catch (ParserException ex) {
    }
    return Double.NaN;
  }

}
