
package math.numerics;

/**
 * StateEvent defines an event consisting in that a given numeric value
 * changes sign from positive (or zero) to strict negative.
 * This numeric value usually consists in an expression of the
 * variables of an ODE.
 * Every valid model state provides a value f(t)>=0 at every instant of time.
 * Finding the moment when the event takes place corresponds to finding the
 * root of f(t). More precisely, and because of numerical approximation,
 * the root is considered to be found when
 *   -tolerance<f(t)<=0
 * where tolerance is a (small positive) number prescribed by
 * the user.
 * When the event takes place an action is triggered.
 *
 * @author Francisco Esquembre (Feb 2004)
 */

public interface StateEvent {

  /**
   * Returns the tolerance for the event.
   * @return the tolerance
   */
  public double getTolerance();

  /**
   * Returns the value f(t) for a given state of the model.
   * The values returned by this method will be used by the solver
   * to find the exact time at which the event took place.
   * @param state The current state of the ODE
   * @return the value for this state
   */
  public double getValue(double[] state);

  /**
   * What to do when the event has taken place.
   * The return value tells the solver wether it should stop the
   * computation at the exact moment of the event or continue
   * solving the ODE for the rest of the prescribed dt.
   * @return true if the solver should return at this instant of time,
   * false otherwise.
   */
  public boolean action ();
}
