/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.linearAlgebra;


/**
 * Title:        VectorMath
 * Description:  A utility class for vector math.
 * Define static methods for dot products, cross products, etc.
 * Copyright:    Copyright (c) 2003  Gould, Christian, and Tobochnik
 * @author       Wolfgang Christian
 * @version 1.0
 */
public final class VectorMath {


    /**
     * Add a vector times a scalar to a vector. The first vector contains the sum.
     * @param a  the first  vector
     * @param b  the second vector
     * @param c  the scalar multiplier
     */
    static public void plus (double[] a, double[] b, double c) {
        int aLength = a.length;
        if (aLength != b.length) {
            System.out.println("ERROR: Vectors must be of equal length to add.");
            return;
        }
        for (int i = 0; i < aLength; i++) {
            a[i] += c*b[i];
        }
    }

    /**
     * Add two vectors. The first vector contains the sum.
     * @param a  the first  vector
     * @param b  the second vector
     */
    static public void plus (double[] a, double[] b) {
        int aLength = a.length;
        if (aLength != b.length) {
            System.out.println("ERROR: Vectors must be of equal length to add.");
            return;
        }
        for (int i = 0; i < aLength; i++) {
            a[i] += b[i];
        }
    }

    /**
     * Calculate the dot product of two vectors.
     * @param a  the first  vector
     * @param b  the second vector
     * @return the dot product
     */
    static public double dot (double[] a, double[] b) {
        int aLength = a.length;
        if (aLength != b.length) {
            System.out.println("ERROR: Vectors must be of equal length in dot product.");
            return  0;
        }
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i]*b[i];
        }
        return  sum;
    }


    /**
     * Calculate the dot product of a vector with itself.
     * @param a  the  vector
     * @return the dot product
     */
    static public double dot (double[] a) {
        int aLength = a.length;
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i]*a[i];
        }
        return  sum;
    }

    /**
     * Calculate the magnitude a vector.
     * @param a  the  vector
     * @return the magnitude
     */
    static public double magnitude(double[] a) {
        int aLength = a.length;
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i]*a[i];
        }
        return  Math.sqrt(sum);
    }

    /**
     * Calculate the cross product of two vectors that lie within a plane.
     * The result is the component perpendicular to the plane.
     *
     * @param a  the first  vector
     * @param b  the second vector
     * @return the cross product.
     */
    static public double cross2D (double[] a, double[] b) {
        if (a.length != 2 || b.length != 2 ) {
            System.out.println("ERROR: Cross2D product requires 2 component array.");
            return  0;
        }
        return  a[0]*b[1]-a[1]*b[0];
    }

    /**
     * Calculate the cross product of a vector within a plane and a vector perpendicular to a plane.
     * The operation transforms the vector in the plane into the new vector.
     *
     * @param a  the vector in the plane
     * @param b  the vector perpendicular to the plane
     */
    static public double[] cross2D (double[] a, double b) {
        if (a.length != 2  ) {
            System.out.println("ERROR: Cross2D product requires 2 component array.");
        }
        double temp=a[0];
        a[0]=a[1]*b;
        a[1]=-temp*b;
        return a;
    }
}



