/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package math.fourierTransforms.r2;

import j2d.ImageUtils;
import j2d.ShortImageBean;
import math.complex.ComplexFloat1d;

import java.awt.*;

public class FFT2dComplex {
    ShortImageBean sib;
    ComplexFloat1d r;
    ComplexFloat1d g;
    ComplexFloat1d b;

    public static void main(String[] args) {
        final Image image1 = ImageUtils.getImage();
        final Image image2 = ImageUtils.getImage();

        FFT2dComplex f1 = new FFT2dComplex(image1);
        FFT2dComplex f2 = new FFT2dComplex(image2);

        f1.normalize();
        f2.normalize();
        //make f2 unitary.
        f2.scale(1f / 255f);

        ImageUtils.displayImage(image1, "image1");
        ImageUtils.displayImage(image2, "image2");

        f1.fft();
        f2.fft();

        f1.mult(f2);

        f1.ifft();
        Image filteredImage = ImageUtils.unahe(ImageUtils.quadSwap(f1.getImage()));
        ImageUtils.displayImage(filteredImage, "complex plane");
    }

    private void scale(float s) {
        r.scale(s);
        g.scale(s);
        b.scale(s);
    }

    FFT2dComplex(Image img) {
        sib = new ShortImageBean(img);
        r = new ComplexFloat1d(sib.getR());
        g = new ComplexFloat1d(sib.getG());
        b = new ComplexFloat1d(sib.getB());
    }

    public void normalize() {
        r.scale(1f / r.getLength());
        g.scale(1f / g.getLength());
        b.scale(1f / b.getLength());
    }

    public void center() {
        r.center();
        g.center();
        b.center();
    }

    /**
     * Get the real part of the complex plane in red, green and blue,
     * combine into an image.
     *
     * @return java.awt.Image of real number plane
     */
    public Image getImage() {
        int w = sib.getWidth();
        int h = sib.getHeight();
        r.scale(255);
        g.scale(255);
        b.scale(255);
        sib.setR(r.getRealAs2dShort(w, h));
        sib.setG(g.getRealAs2dShort(w, h));
        sib.setB(b.getRealAs2dShort(w, h));
        return sib.getImage();
    }


    private void makeUnitary() {
        float rmax = r.getMaxReal();
        float gmax = g.getMaxReal();
        float bmax = b.getMaxReal();
        float max = rmax;
        if (max < gmax) max = gmax;
        if (max < bmax) max = bmax;
        float sf = 1f / max;
        r.scale(sf);
        g.scale(sf);
        b.scale(sf);
    }

    public static Image filter(Image img1, Image img2) {
        FFT2dComplex f1 = new FFT2dComplex(img1);
        FFT2dComplex f2 = new FFT2dComplex(img2);
        f1.normalize();
        f2.normalize();
        //make unitary.
        f1.scale(1f / 255f);
        f2.scale(1f / 255f);

        f1.fft();
        f2.fft();

        f1.mult(f2);
        f1.ifft();
        f1.scale(255);
        return ImageUtils.unahe(ImageUtils.quadSwap(f1.getImage()));
    }

    public static void testAndDisplay(Image image1, Image image2) {
        FFT2dComplex f = new FFT2dComplex(image1);
        f.fft();
        f.normalize();
        FFT2dComplex f2 = new FFT2dComplex(image2);
        ImageUtils.displayImage(image1, "image1");
        ImageUtils.displayImage(image2, "image2");
        f2.fft();
        f2.normalize();
        f.mult(f2);
        f.ifft();
        Image filteredImage = ImageUtils.quadSwap(f.getImage());
        ImageUtils.displayImage(
                ImageUtils.unahe(
                        filteredImage), "complex plane");
    }

    public Image getPsdImage() {
        float rf[] = r.getPSD();
        float gf[] = g.getPSD();
        float bf[] = b.getPSD();
        ShortImageBean psd = new ShortImageBean(rf, gf, bf, sib.getWidth(), sib.getHeight());
        return psd.getImage();
    }

    /**
     * Multiply r,g,b in the complex frequency domain
     *
     * @param f2
     */
    public void mult(FFT2dComplex f2) {
        r.mult(f2.r);
        g.mult(f2.g);
        b.mult(f2.b);
    }

    public void ifft() {
        FFT1dComplex fftr = new FFT1dComplex();
        FFT1dComplex fftg = new FFT1dComplex();
        FFT1dComplex fftb = new FFT1dComplex();
        fftr.computeBackwardFFT(r);
        fftg.computeBackwardFFT(g);
        fftb.computeBackwardFFT(b);
    }

    public void fft() {
        FFT1dComplex fftr = new FFT1dComplex();
        FFT1dComplex fftg = new FFT1dComplex();
        FFT1dComplex fftb = new FFT1dComplex();
        fftr.computeForwardFFT(r);
        fftg.computeForwardFFT(g);
        fftb.computeForwardFFT(b);
    }
}
