/***************************************************************************
 // @(#)vsFFT1D.java
 //
 // Perform a non recursive in place FFT using doubles.
 //
 // @version 1.0, 4 JULY 1997
 // @author  D. Lyon, V. Silva
 ****************************************************************************/
package math.fourierTransforms.r2;

import math.Mat1;
import math.MathUtils;
import sound.OscopePanel;

import javax.swing.*;
import java.awt.*;

/**
 * Works. Automatically scales
 * ouput in ifft.
 * A 1D FFT for integral power of two FFTs
 */
public class FFT1dDouble {
    private final int FORWARD_FFT = -1;
    private final int REVERSE_FFT = 1;
    private int direction = FORWARD_FFT;
    private static final float twoPI = (float) (2 * Math.PI);
    // These arrays hold the complex 1D FFT data.
    private double realData[] = null;
    private double imaginaryData[] = null;

    /**
     * A way to visually test the 1D FFT on a small amount of data.
     */
    public static void showSpectrumAnalyzer() {
        sound.Oscillator o = new sound.Oscillator(440, 1024);
        javax.swing.JTabbedPane jtp = new JTabbedPane();
        addPanel(jtp, "getSquareWave", o.getSquareWave());
        addPanel(jtp, "getSawWave", o.getSawWave());
        addPanel(jtp, "getSineWave", o.getSineWave());
        addPanel(jtp, "getTriangleWave", o.getTriangleWave());
        gui.ClosableJFrame cf = new gui.ClosableJFrame("psd test");
        Container c = cf.getContentPane();
        c.setLayout(new BorderLayout());
        c.add(jtp, BorderLayout.CENTER);
        cf.setSize(400, 400);
        cf.setVisible(true);
    }

    private static void addPanel(JTabbedPane jtp, String title, double d[]) {
        jtp.add(title, getSpectrumPanel(d));
    }

    public static OscopePanel getSpectrumPanel(double audioWaveForm[]) {
        return new OscopePanel(getPsd(audioWaveForm));
    }

    public static double[] getPsd(double audioWaveForm[]) {
        double i[] = new double[audioWaveForm.length];
        FFT1dDouble f = new FFT1dDouble();
        f.computeForwardFFT(audioWaveForm, i);
        return f.getPSD();
    }

    public void visuallyTest() {
        System.out.println("Testing FFT engine in 1D class...");
        int N = 8;
        double[] in_r = new double[N];
        double[] in_i = new double[N];
        double[] holder_r = new double[N];
        double[] holder_i = new double[N];

        // Create a test ramp signal.
        for (int i = 0; i < N; i++) {
            in_r[i] = (double) i / N;
            in_i[i] = (double) 0;
        }
        computeForwardFFT(in_r, in_i);

        // Copy to new array because IFFT will destroy the FFT results.
        for (int i = 0; i < N; i++) {
            holder_r[i] = in_r[i];
            holder_i[i] = in_i[i];
        }
        for (int i = 0; i < N; i++) {
            in_r[i] = in_r[i];
            in_i[i] = in_i[i];
        }
        computeBackwardFFT(in_r, in_i);
        double[] mag = getPSD(in_r, in_i);
        System.out.println("i    x1[i]    re[i]             im[i]           tv[i]");
        for (int i = 0; i < N; i++) {
            System.out.println(i +
                    "\t" + i +
                    "\t" + holder_r[i] +
                    "\t\t" +
                    holder_i[i] +
                    "\t\t" + mag[i]);
        }
    }

    public void timeFFT(int n) {
        // Time a 256K FFT
        double realPart[] = new double[n];
        double imaginaryPart[] = new double[n];

        // Create a test ramp signal.

        synthesizeRamp(realPart, imaginaryPart);
        //MathUtils.print(realPart);

        utils.StopWatch t1 = new utils.StopWatch();
        runExperiment(t1, realPart, imaginaryPart);
    }

    public static void print(Object o[]) {
        for (int i = 0; i < o.length; i++)
            System.out.println(o[i]);
    }

    /**
     * Destroy the input data with a linear ramp.
     *
     * @param realPart      input datas real component
     * @param imaginaryPart input datas' imaginary component.
     */
    public static void synthesizeRamp(double[] realPart, double[] imaginaryPart) {
        int n = realPart.length;
        for (int i = 0; i < n; i++) {
            realPart[i] = (double) i / n;
            imaginaryPart[i] = (double) 0;
        }
    }

    private void runExperiment(utils.StopWatch t1,
                               double[] realPart,
                               double[] imaginaryPart) {
        t1.start();
        //double w[] = MathUtils.getGauss(realPart.length);

        //window(realPart,w);
        computeForwardFFT(realPart, imaginaryPart);
        //getMaxPSDLocation(realPart, imaginaryPart);
        //System.out.println("....real Part...");
        //MathUtils.print(realPart);
        //System.out.println("---- Imaginary Part");
        //MathUtils.print(imaginaryPart);
        //System.out.println("ifft");
        //direction = REVERSE_FFT;
        //MathUtils.print(realPart);

        // Stop the timer and report.
        t1.stop();
        t1.print("did an fft-radix 2 of length" + realPart.length +
                "using double in:");
    }

    public static void window(double r[], double w[]) {
        for (int i = 0; i < r.length; i++)
            r[i] = r[i] * w[i];
    }

    /**
     * 1D FFT utility functions.
     */
    public void swap(int i, int numBits) {
        double tempr;
        int j = bitr(i, numBits);
        tempr = realData[j];
        realData[j] = realData[i];
        realData[i] = tempr;
        tempr = imaginaryData[j];
        imaginaryData[j] = imaginaryData[i];
        imaginaryData[i] = tempr;
    }

    // swap Zi with Zj
    private void swapInt(int i, int j) {
        double tempr;
        int ti;
        int tj;
        ti = i - 1;
        tj = j - 1;
        // System.out.print("swapInt " + ti+","+tj+"\t");
        // System.out.println(Integer.toBinaryString(ti)+","+Integer.toBinaryString(tj));
        // System.out.println(Integer.toBinaryString(ti)+"bitr="+Integer.toBinaryString(bitr(ti)));
        tempr = realData[tj];
        realData[tj] = realData[ti];
        realData[ti] = tempr;
        tempr = imaginaryData[tj];
        imaginaryData[tj] = imaginaryData[ti];
        imaginaryData[ti] = tempr;
    }

    /**
     * get the location of the maximum partial
     *
     * @param realPart
     * @param imaginaryPart
     * @return location of max(realPart[i],imaginaryPart[i])
     */
    public static int getMaxPSDLocation(double realPart[],
                                              double imaginaryPart[]) {
        double max;
        max = -0.99e30;
        int location = 0;
        for (int i = 0; i < realPart.length; i++)
            if (getMagnitude(realPart[i], imaginaryPart[i]) > max) {
                max = realPart[i];
                location = i;
            }
        return location;
    }

    /**
     * Compute the sum of the squares of a complex number
     *
     * @param r    real part
     * @param imag imaginary part
     * @return r*r + imag * imag.
     */
    public static double getMagnitude(double r, double imag) {
        return r * r + imag * imag;
    }

    double getMaxValue(double in[]) {
        double max;
        max = -0.99e30;
        for (int i = 0; i < in.length; i++)
            if (in[i] > max)
                max = in[i];
        return max;
    }

    private void bitReverse2() {
        /* bit reversal */
        int n = realData.length;
        int j = 1;
        int k;
        for (int i = 1; i < n; i++) {
            if (i < j) {
                swapInt(i, j);
            } //if
            k = n / 2;
            while (k >= 1 && k < j) {
                j = j - k;
                k = k / 2;
            }
            j = j + k;
        } // for
    }

    private int bitr(int j, int numBits) {
        int ans = 0;
        for (int i = 0; i < numBits; i++) {
            ans = (ans << 1) + (j & 1);
            j = j >> 1;
        }
        return ans;
    }

    public void computeForwardFFT(double[] in_r, double[] in_i) {
        direction = FORWARD_FFT;
        fft(in_r, in_i);
    }

    public void computeBackwardFFT(double[] in_r, double[] in_i) {
        direction = REVERSE_FFT;
        fft(in_r, in_i);
    }

    /**
     * FFT engine.
     */
    public void fft(double _realPart[], double _imaginaryPart[]) {
        int id;
        // radix 2 number if sample, 1D of course.
        int localN;
        double wtemp, Wjk_r, Wjk_i, Wj_r, Wj_i;
        double theta, tempr, tempi;
        int numBits = MathUtils.getLogBase2(_realPart.length);
        // Truncate input data to a power of two
        int length = 1 << numBits; // length = 2**nu

        // Copy passed references to variables to be used within
        // transforms.fft routines & utilities
        realData = _realPart;
        imaginaryData = _imaginaryPart;
        bitReverse2();
        for (int m = 1; m <= numBits; m++) {
            // localN = 2^m;
            localN = 1 << m;
            Wjk_r = 1;
            Wjk_i = 0;
            theta = twoPI / localN;
            Wj_r = Math.cos(theta);
            Wj_i = direction * Math.sin(theta);
            int nby2 = localN / 2;
            for (int j = 0; j < nby2; j++) {
                // This is the FFT innermost loop
                // Any optimizations that can be made here will yield
                // great rewards.
                for (int k = j; k < length; k += localN) {
                    id = k + nby2;
                    tempr = Wjk_r * realData[id] - Wjk_i * imaginaryData[id];
                    tempi = Wjk_r * imaginaryData[id] + Wjk_i * realData[id];

                    // Zid = Zi -C
                    realData[id] = realData[k] - tempr;
                    imaginaryData[id] = imaginaryData[k] - tempi;
                    realData[k] += tempr;
                    imaginaryData[k] += tempi;
                }

                // (eq 6.23) and (eq 6.24)

                wtemp = Wjk_r;
                Wjk_r = Wj_r * Wjk_r - Wj_i * Wjk_i;
                Wjk_i = Wj_r * Wjk_i + Wj_i * wtemp;
            }
        }
    }

    /**
     * @return gets the real component from FFT.
     */
    public double[] getRealData() {
        return realData;
    }

    /**
     * @return gets the imaginary component from FFT.
     */
    public double[] getImaginaryData() {
        return imaginaryData;
    }

    /**
     * @return Power Spectral Density of the internal data
     */
    public double[] getPSD() {
        double psd[] =
                getPSD(realData, imaginaryData);
        Mat1.normalize(psd);
        return psd;
    }

    /**
     * Compute the power spectral density of the input
     * arrays
     *
     * @param in_r real part of an fft
     * @param in_i imaginary part of an fft
     * @return the psd.
     */
    public static double[] getPSD(double in_r[], double in_i[]) {
        int N = in_r.length;
        double[] mag = new double[N];
        for (int i = 0; i < in_r.length; i++)
            mag[i] = Math.sqrt(in_r[i] * in_r[i] + in_i[i] * in_i[i]);
        return (mag);
    }

    public static void testFFT2() {
        FFT1dDouble f = new FFT1dDouble();
        //f.visuallyTest();
        for (int i = 0; i < 10; i++)
            f.timeFFT(1024 * 1024);
    }

    /**
     * Test the DFT using a small number of samples.
     * This is an <i>O(N**2)</i> algorithm.
     */
    public static void testFFT() {
        int N = 8;
        FFT1dDouble f = new FFT1dDouble();

        double x1[] = Mat1.getRampData(N);
        // take dft
        double in_im[] = new double[x1.length];
        double in_r[] = Mat1.arrayCopy(x1);
        f.computeForwardFFT(in_r, in_im);
        Mat1.scale(in_r,1f/in_r.length);
        Mat1.scale(in_im,1f/in_im.length);
        f.computeBackwardFFT(in_r, in_im);

        System.out.println("j\tx1[j]\tre[j]\tim[j]");
        for (int j = 0; j < N; j++) {

            System.out.println(j + "\t" +
                    x1[j] + "\t" +
                    in_r[j] + "\t" +
                    in_im[j] );
        }
    }

    public static void main(String args[]) {
        testFFT();
    }
}
