/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package math.fourierTransforms.r2;

import math.MathUtils;
import math.Mat1;
import math.complex.ComplexFloat1d;
/**
 * A 1D FFT for integral power of two FFTs
 */
public class FFT1dComplex {
    private final int FORWARD_FFT = -1;
    private final int REVERSE_FFT = 1;
    private int direction = FORWARD_FFT;
    private static final float twoPI = (float) (2 * Math.PI);
    private final ComplexFloat1d data = new ComplexFloat1d();


    private void bitReverse2() {
        /* bit reversal */
        int n = data.getLength();
        int j = 1;
        int k;
        for (int i = 1; i < n; i++) {
            if (i < j) {
                data.swapInt(i, j);
            } //if
            k = n / 2;
            while (k >= 1 && k < j) {
                j = j - k;
                k = k / 2;
            }
            j = j + k;
        } // for
    }

    public void computeForwardFFT(float[] in_r, float[] in_i) {
        direction = FORWARD_FFT;
        fft(in_r, in_i);
    }

    public void computeBackwardFFT(float[] in_r, float[] in_i) {
        direction = REVERSE_FFT;
        fft(in_r, in_i);
    }

    /**
     * FFT engine.
     */
    public void fft(float _realPart[], float _imaginaryPart[]) {
        int id;
        // radix 2 number if sample, 1D of course.
        int localN;
        float wtemp, Wjk_r, Wjk_i, Wj_r, Wj_i;
        float theta, tempr, tempi;
        int numBits = MathUtils.getLogBase2(_realPart.length);
        // Truncate input data to a power of two
        int length = 1 << numBits; // length = 2**nu

        // Copy passed references to variables to be used within
        // transforms.fft routines & utilities
        data.re = _realPart;
        data.im = _imaginaryPart;
        bitReverse2();
        for (int m = 1; m <= numBits; m++) {
            // localN = 2^m;
            localN = 1 << m;
            Wjk_r = 1;
            Wjk_i = 0;
            theta = twoPI / localN;
            Wj_r = (float) Math.cos(theta);
            Wj_i = direction * (float) Math.sin(theta);
            int nby2 = localN / 2;
            for (int j = 0; j < nby2; j++) {
                // This is the FFT innermost loop
                // Any optimizations that can be made here will yield
                // great rewards.
                for (int k = j; k < length; k += localN) {
                    id = k + nby2;
                    tempr = Wjk_r * data.re[id] - Wjk_i * data.im[id];
                    tempi = Wjk_r * data.im[id] + Wjk_i * data.re[id];

                    // Zid = Zi -C
                    data.re[id] = data.re[k] - tempr;
                    data.im[id] = data.im[k] - tempi;
                    data.re[k] += tempr;
                    data.im[k] += tempi;
                }

                // (eq 6.23) and (eq 6.24)

                wtemp = Wjk_r;
                Wjk_r = Wj_r * Wjk_r - Wj_i * Wjk_i;
                Wjk_i = Wj_r * Wjk_i + Wj_i * wtemp;
            }
        }
    }


    public ComplexFloat1d getData() {
        return data;
    }

    public static void main(String[] args) {
               int N = 8;
        FFT1dComplex f = new FFT1dComplex();

        float x1[] = Mat1.getRampDataFloat(N);
        // take dft
        float in_im[] = new float[x1.length];
        float in_r[] = Mat1.arrayCopy(x1);
        ComplexFloat1d c = new ComplexFloat1d(in_r,in_im);
        c.scale(1f/in_r.length);
        //Mat1.scale(in_r,1f/in_r.length);
        f.computeForwardFFT(c);
        f.computeBackwardFFT(c);

        for (int j = 0; j < N; j++) {
            System.out.println(Math.round(x1[j])+"\t"+Math.round(in_r[j]));
        }
    }

    public void computeBackwardFFT(ComplexFloat1d c) {
        computeBackwardFFT(c.getRe(),c.getIm());
    }

    public void computeForwardFFT(ComplexFloat1d c) {
        computeForwardFFT(c.getRe(),c.getIm());
    }
}
