/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package math.complex;

import j2d.ImageUtils;
import j2d.ShortImageBean;

import java.awt.*;

import math.Mat1;

/**
 * User: lyon
 * Date: Nov 13, 2005
 * Time: 8:00:08 AM
 * Copyright DocJava, Inc. 2005.
 */
public class ComplexFloat1d {
    public float re[];
    public float im[];

    public ComplexFloat1d() {
    }

    /**
     * Copy the s input into the real number
     * plane and set the imaginary plane to zero.
     *
     * @param s
     */
    public ComplexFloat1d(short s[][]) {
        int w = s.length;
        int h = s[0].length;
        int n = w*h;
        re = new float[n];
        im = new float[n];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                re[x+y*w] = s[x][y];
            }
    }

    public short[][] getRealAs2dShort(int w, int h) {
        short s[][] = new short[w][h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                s[x][y] = (short) re[x+y*w];
            }
        return s;
    }

    public static void main(String[] args) {
        Image img = ImageUtils.getImage();
        ShortImageBean sib = new ShortImageBean(img);
        ImageUtils.displayImage(sib.getImage(), " input image");
        int w = sib.getWidth();
        int h = sib.getHeight();
        System.out.println("dim wxh=" + w + "\t" + h);
        ComplexFloat1d c = new ComplexFloat1d(sib.getR());
        final short[][] r = c.getRealAs2dShort(w, h);
        ShortImageBean sib2 = new ShortImageBean(r, r, r);
        ImageUtils.displayImage(sib2.getImage(), "complex out");
    }

    public ComplexFloat1d(int n) {
        re = new float[n];
        im = new float[n];
    }

    // swap Zi with Zj
    public void swapInt(int i, int j) {
        float tempr;
        int ti;
        int tj;
        ti = i - 1;
        tj = j - 1;
        // System.out.print("swapInt " + ti+","+tj+"\t");
        // System.out.println(Integer.toBinaryString(ti)+","+Integer.toBinaryString(tj));
        // System.out.println(Integer.toBinaryString(ti)+"bitr="+Integer.toBinaryString(bitr(ti)));
        tempr = re[tj];
        re[tj] = re[ti];
        re[ti] = tempr;
        tempr = im[tj];
        im[tj] = im[ti];
        im[ti] = tempr;
    }

    public ComplexFloat1d(float[] realPart, float[] imaginaryPart) {
        re = realPart;
        im = imaginaryPart;
    }

    // a = a + b, a=a.plus(b)
    public void plus(ComplexFloat1d b) {
        for (int i = 0; i < re.length; i++) {
            re[i] += b.getRe(i);
        }
        for (int i = 0; i < re.length; i++) {
            im[i] += b.getIm(i);
        }
    }

    //todo must test the operations! All are untested!
    public ComplexFloat1d getCopy() {
        ComplexFloat1d c = new ComplexFloat1d();
        c.setRe(Mat1.arrayCopy(re));
        c.setIm(Mat1.arrayCopy(im));
        return c;
    }

    /**
     * Mask the image kernal with the real and imaginary numbers.
     * Assume the image kernal ranges from 0..255. Make it a unitary
     * filter by dividing by 255.
     *
     * @param r
     */
    public void scaleAndMask(short r[][]) {
        int w = r.length;
        int h = r[0].length;
        for (int x = 0; x < w; x++) {
            for (int y = 0; y < h; y++) {
                int i = x + y * w;
                re[i] *= r[x][y] / 255f;
                im[i] *= r[x][y] / 255f;
            }
        }
    }

    public void scale(float sc) {
        for (int i = 0; i < re.length; i++)
            re[i] = re[i] * sc;
        for (int i = 0; i < re.length; i++)
            im[i] = im[i] * sc;
    }

    //todo test the below code...
    // a = a * b, a=a.times(b);
    //(x1 + iy1)(x2 + iy2) =
    //      x1        *  x2    - y1        * y2 +
    //    i(y1        *  x2    + x1        * y2)
    public void mult(ComplexFloat1d b) {
        float realT;
        for (int i = 0; i < re.length; i++) {

            float x1 = re[i];
            float x2 = b.getRe(i);
            float y1 = im[i];
            float y2 = b.getIm(i);
            realT = x1 * x2 - y1 * y2;
            im[i] = y1 * x2 + x1 * y2;
            re[i] = realT;
        }
    }


    // a = a - b, a=a.minus(b)
    public void minus(ComplexFloat1d b) {
        for (int i = 0; i < re.length; i++) {
            re[i] -= b.getRe(i);
        }
        for (int i = 0; i < re.length; i++) {
            im[i] -= b.getIm(i);
        }
    }

    public float[] logScaleMagnitude(float magScale) {
        int N = re.length;
        float result[] = new float[N];
        for (int i = 0; i < N; i++) {
            result[i] = (float) (magScale *
                    Math.log(1 + Math.sqrt(re[i] * re[i] + im[i] * im[i])));
            if (result[i] > 255)
                result[i] = 255;
        }
        return result;
    } // End of function magnitude().

    public float[] getIm() {
        return im;
    }

    public float getIm(int i) {
        return im[i];
    }

    public void setIm(float[] im) {
        this.im = im;
    }

    public float getRe(int i) {
        return re[i];
    }

    public float[] getRe() {
        return re;
    }

    public int getLength() {
        return re.length;
    }

    public void setRe(float[] re) {
        this.re = re;
    }

    public void swap(int i, int j) {
        float tempr = re[j];
        re[j] = re[i];
        re[i] = tempr;
        tempr = im[j];
        im[j] = im[i];
        im[i] = tempr;
    }

    public float[] getPSD() {
        return getPSD(re, im);
    }

    /**
     * Compute the power spectral density of the input
     * arrays
     *
     * @param in_r real part of an fft
     * @param in_i imaginary part of an fft
     * @return the psd.
     */
    public static float[] getPSD(float in_r[], float in_i[]) {
        int N = in_r.length;
        float[] mag = new float[N];
        for (int i = 0; i < in_r.length; i++)
            mag[i] = (float) Math.sqrt(in_r[i] * in_r[i] + in_i[i] * in_i[i]);
        return (mag);
    }

    public void center() {
        float alternateSign;
        int n = re.length;
        for (int i = 0; i < n; i++) {

            // Calculate (-1)**(i)
            alternateSign = (i % 2 == 0) ? -1 : 1;
            // 1.
            // (-1)**(i) is to put the zero frequency in the
            // center of the image when it is displayed.
            re[i] = (re[i] * alternateSign);
            im[i] = (im[i] * alternateSign);
        }
    }

    public void setRe(short re[]) {
        this.re = new float[re.length];
        for (int i=0; i < re.length; i++){
            this.re[i] = re[i];
        }

    }

    public void print() {
        System.out.println("re[i],im[i]");
        for (int i=0; i < re.length; i++){
            System.out.println(re[i]+","+im[i]);
        }
    }

    public void normalize() {
        scale(1f/re.length);
    }

    /**
     * Search the real part of the complex array
     * @return   the max value
     */
    public float getMaxReal() {
        return Mat1.getMax(re);
    }

    public float getRealSum() {
        return Mat1.sum(re);
    }


}
