/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.shapes;
import j3d.examples.appearance.texture.noise.ImprovedNoise;

import javax.media.j3d.*;
import javax.vecmath.*;

// TODO: Document the Rock class
public class Rock extends Shape3D {
	protected static float NOISE_ZOOM = 0.7f; // change to taste
	protected int divisions;
	protected int vertexCount;
	protected float radius;
	protected float radiansPerDivisionHorizontal;
	protected float radiansPerDivisionVertical;

	public Rock() {
		this(2, 16);
	}
	
	public Rock(float aRadius) {
		this(aRadius, 16);
	}

	public Rock(float aRadius, int divisions) {
		initialize(aRadius, divisions);
		ImprovedNoise.reinitialize();
		create();
	}
	
	protected void initialize(float aRadius, int divisions){
		this.divisions = divisions;		
		radius = aRadius;
		radiansPerDivisionHorizontal = (float) (2* Math.PI/(float)divisions);
		radiansPerDivisionVertical = (float) (Math.PI/(float)(divisions));
		vertexCount = (divisions + 1) * (divisions + 1);
	}
	protected void create(){
		setGeometry(createGeometry());
		setAppearance(createAppearance());
	}
	
	protected Geometry createGeometry() {
		// This implementation 'wastes' half of the triangles
		// on the top and bottom strips.  Optimally, the top
		// and bottom strips should be handled as a sawtooth
		// formation.
		// The number of indices is based on the 
		// number of horizontal strips (height - 1) times the 
		// number of vertices per strip (width * 2).
		int indexCount = divisions * (divisions + 1) * 2;
		int[] stripCounts = new int[divisions];
		for (int strip = 0; strip < divisions; strip++) {
			stripCounts[strip] = (divisions + 1) * 2;
		}

		IndexedTriangleStripArray geometry =
			new IndexedTriangleStripArray(
				vertexCount,
				getVertexFormat(),
				indexCount,
				stripCounts);

		float[] coordinates = new float[vertexCount * 3];
		// The rock will spin based on the center of mass.
		// Nudge the center of mass (local origin) a little 
		// bit to make it wobble.
		float cmX = 0;//(float) Math.random() * 0.3f;
		float cmY = 0;//(float) Math.random() * 0.3f;
		float cmZ = 0;//(float) Math.random() * 0.3f;
		int middle = (divisions + 1)/2;
		for (int row = 0; row < (divisions + 1); row++) {
			int width = row * (divisions + 1);
			float verticalAngle = (row - middle) * radiansPerDivisionVertical;
			//System.out.println("vertical angle: " + verticalAngle + " y: " + radius * Math.sin(verticalAngle));
			for (int col = 0; col < (divisions + 1); col++) {
				// coordinate index is the column plus
				// the row times the width of a row times
				// three (one for each x, y, and z).
				
				int ci = (col + width) * 3;
				float horizontalAngle = (col - middle) * radiansPerDivisionHorizontal;
				float x = (float)(Math.cos(horizontalAngle) * Math.cos(verticalAngle));
				float y = (float)Math.sin(verticalAngle);
				float z = -(float)(Math.sin(horizontalAngle) * Math.cos(verticalAngle));
				float noise = (2 - (float)ImprovedNoise.noise(x/NOISE_ZOOM, y/NOISE_ZOOM, z/NOISE_ZOOM))/2;
				float weight = 0.1f;
				float aRadius = radius * weight + ( 1 - weight) * noise * radius;
				coordinates[ci + 0] = aRadius * (x + cmX);
				coordinates[ci + 1] = aRadius * (y + cmY);
				coordinates[ci + 2] = aRadius * (z + cmZ);
			}
		}
		geometry.setCoordRefFloat(coordinates);

		int[] indices = new int[indexCount];
		/* The secret is that the strip vertices must be ordered 
		 * like this: NW, SW, NE, SE for each set of four corners 
		 * of a quad.  A convenient way to accomplish this is to 
		 * organize the landscape in horizontal strips and iterate 
		 * across the columns calculating two vertices at a time. 
		 */
		int pi = 0; // points index
		for (int row = 0; row < divisions; row++) {
			int width = row * (divisions + 1);
			for (int col = 0; col < (divisions + 1); col++) {
				int coordinateIndex = width + col;
				indices[pi] = coordinateIndex + (divisions + 1);
				indices[pi + 1] = coordinateIndex;
				pi = pi + 2;
			}
		}
		geometry.setCoordinateIndices(0, indices);
		geometry.setUserData(indices);
		// Normal indices never change so set them here. For
		// sanity's sake, the vertex normals are arranged just
		// like the coordinates so the indices are the same.
		geometry.setNormalIndices(0, indices);
		generateNormals(geometry, coordinates);
		return geometry;
	}

	
	protected int getVertexFormat() {
		int vertexFormat =
		GeometryArray.COORDINATES
				| GeometryArray.NORMALS
				| GeometryArray.BY_REFERENCE;
		return vertexFormat;
	}
	
	/* This object generates it's own normals to remove the need
	 * for the NormalGenerator utility class.  The utility class is
	 * very general whereas this implementation streamlines normal
	 * generation for a specific shading model (smooth) and 
	 * geometry (indexed triangle strip array).
	 */
	protected void generateNormals(IndexedGeometryStripArray geometry, float[] coordinates) {
		int[] indices = (int[]) geometry.getUserData();

		int stripCount = divisions;
		int stripIndexCount = (divisions + 1) * 2;
		//int triangles = divisions * divisions * 2;

		// The vertex normals are kept by reference within the 
		// geometry to reduce object creation overhead.
		float[] vertexNormals = geometry.getNormalRefFloat();
		if (vertexNormals == null) {
			vertexNormals = new float[coordinates.length];
			geometry.setNormalRefFloat(vertexNormals);
		}
		
		/* Loop through the coordinates via the indices
		 * indirection and calculate the facet normal for
		 * each triangle and vertex normal for each vertex.  
		 * Smooth shading depends on vertex normals to 
		 * interpolate shading intensities across a triangle.  
		 * A Gourand shading approach averages the 
		 * facet normals of the triangles sharing a vertex.  
		 * As the facet normal is determined, add the facet
		 * normal to each of the three vertex normals.  The
		 * vertex normals will be summed at the end of this
		 * loop and the average is completed later by
		 * normalizing the vector.
		 */
		Vector3f normal = new Vector3f();
		Vector3f v1 = new Vector3f();
		Vector3f v2 = new Vector3f();
		int triangleCount = 0;
		int stripOffset = 0;

		for (int strip = 0; strip < stripCount; strip++) {
			for (int i = 0; i < stripIndexCount - 2; i++) {
				int pi = i + stripOffset;

				// The first vertex is used as
				// the local origin for the vectors.
				int ci1 = 3 * indices[pi + 0];
				float x1 = coordinates[ci1 + 0];
				float y1 = coordinates[ci1 + 1];
				float z1 = coordinates[ci1 + 2];

				// The end point for the first vector
				int ci2 = 3 * indices[pi + 1];
				float x2 = coordinates[ci2 + 0];
				float y2 = coordinates[ci2 + 1];
				float z2 = coordinates[ci2 + 2];

				// The end point for the second vector
				int ci3 = 3 * indices[pi + 2];
				float x3 = coordinates[ci3 + 0];
				float y3 = coordinates[ci3 + 1];
				float z3 = coordinates[ci3 + 2];

				// Translate first vector to local origin
				v1.x = x2 - x1;
				v1.y = y2 - y1;
				v1.z = z2 - z1;

				// Translate second vector to local origin
				v2.x = x3 - x1;
				v2.y = y3 - y1;
				v2.z = z3 - z1;

				/* Calculate the normal for this triangle
				 * The iteration goes through the triangles
				 * and every other one has vertices in an 
				 * order that would result in the normal pointing
				 * the wrong way.  Alternate the cross product to
				 * compensate.
				 */
				if (triangleCount % 2 == 0) {
					normal.cross(v1, v2);
				} else {
					normal.cross(v2, v1);
				}
				/* Note that the normal is NOT normalized here as 
				 * you might expect.  Since the neighboring facet
				 * normals are going to be averaged, a normalize()
				 * would be a waste.  The NormalGenerator util class
				 * supports something called a crease angle.  This 
				 * angle essentially determines what is should be 
				 * smoothed, and what should have a definitive
				 * corner.  If you decide to add crease
				 * angle support in this code, you will need to
				 * normalize each normal.  The reason is that a dot
				 * product of two vectors u and v is:
				 * 		u*v = |u|||v|cos(theta)
				 * If the magnitude of both is one, then a dot product
				 * can be used to determine the angle between the 
				 * two vectors (theta) compared to a crease angle.
				 * I would take the cosine of the crease angle and
				 * compare it with the dot product (cosine of the 
				 * angle between the two normals) to determine if
				 * the crease angle has been met.
				 */

				/* Expand the facet normal to 3 vertex normals.  This is
				 * done by averaging the facet normals of the triangles
				 * that share a vertex.  The NormalGenerator uses a crease
				 * angle to decide if averaging can be done, but this
				 * code assumes it can always be done.  At this point,
				 * we only add the normals as they are encountered.  The
				 * divide (to complete the average) is done later when
				 * the vector is normalized.  The normals are not
				 * normalized yet.
				 */
				// First vertex of the triangle
				vertexNormals[3 * indices[pi + 0] + 0] = +normal.x;
				vertexNormals[3 * indices[pi + 0] + 1] = +normal.y;
				vertexNormals[3 * indices[pi + 0] + 2] = +normal.z;
				// Second vertex of the triangle
				vertexNormals[3 * indices[pi + 1] + 0] = +normal.x;
				vertexNormals[3 * indices[pi + 1] + 1] = +normal.y;
				vertexNormals[3 * indices[pi + 1] + 2] = +normal.z;
				// Third vertex of the triangle
				vertexNormals[3 * indices[pi + 2] + 0] = +normal.x;
				vertexNormals[3 * indices[pi + 2] + 1] = +normal.y;
				vertexNormals[3 * indices[pi + 2] + 2] = +normal.z;
				triangleCount++;
			}
			stripOffset = stripOffset + stripIndexCount;
		}
		
		// Since this is a closed shape, consider the sum of the
		// normals on the edges where the surface meets with itself.
		Vector3f left = new Vector3f(0, 0, 0);
		Vector3f right = new Vector3f(0, 0, 0);
		for (int row = 0; row < divisions; row++) {
			int leftVertexNormalIndex = 3 * row * (divisions + 1);
			int rightVertexNormalIndex = leftVertexNormalIndex +  3 * divisions;
			left.x = vertexNormals[leftVertexNormalIndex + 0];
			left.y = vertexNormals[leftVertexNormalIndex + 1];
			left.z = vertexNormals[leftVertexNormalIndex + 2];
			
			right.x = vertexNormals[rightVertexNormalIndex + 0];
			right.y = vertexNormals[rightVertexNormalIndex + 1];
			right.z = vertexNormals[rightVertexNormalIndex + 2];
			
			left.add(right);
			vertexNormals[leftVertexNormalIndex + 0] = left.x;
			vertexNormals[leftVertexNormalIndex + 1] = left.y;
			vertexNormals[leftVertexNormalIndex + 2] = left.z;
			vertexNormals[rightVertexNormalIndex + 0] = left.x;
			vertexNormals[rightVertexNormalIndex + 1] = left.y;
			vertexNormals[rightVertexNormalIndex + 2] = left.z;
		}

		/* The vertex normals currently hold the sum of the
		 * facet normals of the triangles sharing the vertex.
		 * Complete the averaging of the vertex normals.
		 * Since the vertexNormals are stored by reference
		 * with the geometry, updates to the array are 
		 * automatically reflected in the geometry.
		 */
		Vector3f aNormal = new Vector3f(0, 0, 0);
		for (int vni = 0; vni < coordinates.length; vni = vni + 3) {
			// vertex normal index
			aNormal.x = vertexNormals[vni + 0];
			aNormal.y = vertexNormals[vni + 1];
			aNormal.z = vertexNormals[vni + 2];
			// Normalizing the vector completes the averaging.
			aNormal.normalize();
			vertexNormals[vni + 0] = aNormal.x;
			vertexNormals[vni + 1] = aNormal.y;
			vertexNormals[vni + 2] = aNormal.z;
		}
		
	}
	
	protected Appearance createAppearance(){
		Appearance a = new Appearance();
		Material m = new Material();
		m.setAmbientColor(0.1f, 0.1f, 0);
		m.setDiffuseColor(0.2f, 0.2f, 0.2f);
		m.setShininess(0f);
		a.setMaterial(m);
		return a;
	}

}
