/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.examples;
import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.GraphicsConfiguration;

import javax.media.j3d.*;
import javax.vecmath.*;

import j3d.examples.particles.*;
import j3d.examples.particles.emitters.*;
import j3d.examples.particles.influences.*;
import j3d.examples.particles.shapes.*;
import j3d.examples.particles.generationshapes.*;
import com.sun.j3d.utils.geometry.*;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.universe.SimpleUniverse;
/*
 * This example uses particle systems to create a tornado.  Three
 * particle systems make up the tornado:
 * 
 * Body - Under the influence of an upward moving Vortex, this particle
 * system emits expanding cloud puffs to create the main body or trunk of
 * the tornado.
 * 
 * Top - A swirling top is accomplished with another particle system using
 * a very wide vortex with no upward movement.  The result is a donut shaped
 * group of clouds rotating about the center hole.
 * 
 * Debris - As the tornado rips through the ground, debris is emitted by this
 * particle system.  Because it is intended to simulated dust and debris, the 
 * particles are subject to scaling and fading to emulate dispersal and gravity
 * to emulate mass.
 */
public class TornadoExample extends Applet {

	public static void main(String[] args) {
		new MainFrame(new TornadoExample(false), 600, 600);
	}
	private Canvas3D canvas3D;
	private boolean recording;
	
	public TornadoExample(boolean recording) {
		initialize(recording);
	}

	public BranchGroup createSceneGraph(SimpleUniverse su) {
		BranchGroup objRoot = new BranchGroup();
		// Add a simple dark green plain for the tornado to destroy
		objRoot.addChild(new XZPlane(10, 100, true, new Color3f(0.12f,0.12f,0)));
		BoundingSphere bounds = new BoundingSphere();
		bounds.setRadius(800);
		
		// Add a behavior to manage the particle system animation
		ParticleSystemManager manager = ParticleSystemManager.getCurrent(0.05f);
		manager.setSchedulingBounds(bounds);
		objRoot.addChild(manager);
		
		// Add a behavior to measure frame rates
//		FrameRateBehavior fps = new FrameRateBehavior();
//		fps.setSchedulingBounds(bounds);
//		objRoot.addChild(fps);
		
		CloudPuff.shareGeometry = false;
		float demoLifeTime = 120;
		// Add the body of the tornado
		TransformGroup bodyTG = new TransformGroup();
		Transform3D bodyTransform = new Transform3D();
		bodyTransform.setTranslation(new Vector3f(0, 5, 0));
		bodyTG.setTransform(bodyTransform);
		ParticleEmitter bodyEmitter =
				new ParticleEmitter(
					new RadialGenerationShape(5, 2),
					10,// emission rate
					0, // emission rate variance
					0, // velocity
					0, // velocity variance
					7.5f, // lifetime
					0.5f, // lifetime variance
					demoLifeTime, // emitter lifetime in seconds
					false);

		/*
		Fujita Tornado Intensity Scale 
		Scale Wind Speed (m/s) 	Damage 
		F-0 	18-32 			Light -- tree branches broken, damage to chimneys and large signs. 
		F-1 	33-50 			Moderate -- trees snapped, surface of roofs peeled off, windows broken. 
		F-2 	51-70 			Considerable -- large trees uprooted, roofs torn off frame houses, mobile homes demolished. 
		F-3 	71-92 			Severe -- roof and some walls torn off well-constructed houses, cars overturned. 
		F-4 	93-116 			Devastating -- well-constructed houses leveled, cars and large objects thrown. 
		F-5 	117-142 		Incredible -- strong frame houses lifted off foundation and destroyed, cars-sized objects thrown more than 90 meters. 
		 */
		IExternalInfluence bodyVortex =
			new Vortex(Math.PI/16, 0, 50, 2*Math.PI, 25); // velocity = rw = 50*pi = 157 m/sec
		bodyEmitter.addInfluence(bodyVortex);
		bodyEmitter.addInfluence(new Scale(1,4f));
		bodyEmitter.addInfluence(new FadeShape(new ParticleAgeAlpha(0,0.05f,0.9f,0.05f)));
		IShape3DFactory factory =
			new CloudPuffFactory(
				10f,
				0,
				new Color3f(0.5f, 0.5f, 0.5f),
				new Color3f(0.1f, 0.1f, 0.1f),
				new Color3f(0.32f, 0.32f, 0.0f),
				new Color3f(0.07f, 0.07f, 0.0f),
				6,
				0, 
				CloudPuff.SMOKE);
		//IShape3DFactory factory = new ColorCubeFactory();
		Shape3DParticleSystem bodyParticleSystem =
			new Shape3DParticleSystem(bodyEmitter, factory);
		bodyTG.addChild(bodyParticleSystem);
		objRoot.addChild(bodyTG);
		
		// Add the debris at the bottom of the tornado
		TransformGroup debrisTG = new TransformGroup();
		Transform3D debrisTransform = new Transform3D();
		debrisTransform.setTranslation(new Vector3f(0, 10, 0));
		debrisTG.setTransform(debrisTransform);
		ParticleEmitter debrisEmitter =
				new ParticleEmitter(
					new DiskGenerationShape(Math.PI/3,7,3),
					6.5f,	// emission rate
					0.0f, 	// emission rate variance
					17, 	// velocity
					7.5f,	// velocity variance
					3f, 	// lifetime
					1.5f, 		// lifetime variance
					demoLifeTime-2,// emitter lifetime in seconds
					false);
		
		debrisEmitter.addInfluence(new FadeShape());
		debrisEmitter.addInfluence(new Scale(3f));
		debrisEmitter.addInfluence(new Gravity());
		debrisEmitter.addInfluence(new BounceShape());
		IShape3DFactory debrisFactory =
			new CloudPuffFactory(
				20f,
				10,
				new Color3f(0.25f, 0.25f, 0.25f),
				new Color3f(0.1f, 0.1f, 0.1f),
				new Color3f(0.18f, 0.18f, 0.0f),
				new Color3f(0.03f, 0.03f, 0.00f),
				4,
				0);
//		IShape3DFactory debrisFactory = new ColorCubeFactory();
		Shape3DParticleSystem debrisParticleSystem =
			new Shape3DParticleSystem(debrisEmitter, debrisFactory);
		debrisTG.addChild(debrisParticleSystem);
		objRoot.addChild(debrisTG);

		// Add the swirling top of the twister
		TransformGroup topTG = new TransformGroup();
		Transform3D topTransform = new Transform3D();
		topTransform.setTranslation(new Vector3f(0, 220, 0));
		topTG.setTransform(topTransform);
		ParticleEmitter topEmitter =
				new ParticleEmitter(
					new RadialGenerationShape(170, 110),
					10, // emission rate
					0, 	// emission rate variance
					0, 	// velocity
					0, 	// velocity variance
					10, // lifetime
					0, 	// lifetime variance
					demoLifeTime+2, // emitter lifetime in seconds
					false);

		IExternalInfluence topVortex =
			new Vortex(0, 0, 400, Math.PI/6, 0f);
		topEmitter.addInfluence(topVortex);
		topEmitter.addInfluence(new FadeShape(new ParticleAgeAlpha(0,0.2f,0.75f,0.05f)));
		IShape3DFactory topFactory =
			new CloudPuffFactory(
				90,
				5,
				new Color3f(0.35f, 0.35f, 0.35f),
				new Color3f(0.1f, 0.1f, 0.1f),
				new Color3f(0.32f, 0.32f, 0.0f),
				new Color3f(0.05f, 0.05f, 0.00f),
				7,
				0);
//		IShape3DFactory topFactory = new ColorCubeFactory();
		
		Shape3DParticleSystem topParticleSystem =
			new Shape3DParticleSystem(topEmitter, topFactory);
		topTG.addChild(topParticleSystem);
		objRoot.addChild(topTG);

		// Add Ambient light

		AmbientLight ambient = new AmbientLight();
		ambient.setColor(new Color3f(1f, 1f, 1f));
		ambient.setInfluencingBounds(bounds);
		objRoot.addChild(ambient);

		// Add a directional light
		DirectionalLight directional = new DirectionalLight();
		directional.setDirection(1.0f, -0.4f, 1f);
		directional.setColor(new Color3f(0.7f, 0.7f, 0.7f));
		directional.setInfluencingBounds(bounds);
		objRoot.addChild(directional);
		
		// Add a simply sky dome for sky effects
		Appearance appearance = new Appearance();
		ColoringAttributes ca = new ColoringAttributes();
		ca.setShadeModel(ColoringAttributes.NICEST);
		appearance.setColoringAttributes(ca);
		Material material = new Material();
		material.setAmbientColor(0.25f, 0.25f, 0);
		material.setSpecularColor(0.5f, 0.5f, 0.5f);
		appearance.setMaterial(material);
		PolygonAttributes polyAttrib = new PolygonAttributes();
		polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
		polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);
		appearance.setPolygonAttributes(polyAttrib);
		Sphere background = new Sphere(800, Sphere.GENERATE_NORMALS, 50, appearance);
		objRoot.addChild(background);

		// Add a keyboard navigator to move around
		TransformGroup vpTrans =
			su.getViewingPlatform().getViewPlatformTransform();
		KeyNavigatorBehavior keyNavBeh = new KeyNavigatorBehavior(vpTrans);
		keyNavBeh.setSchedulingBounds(bounds);
		objRoot.addChild(keyNavBeh);
		
		// Add an image capturing behavior
		if(recording){
			ImageCaptureBehavior capture = new ImageCaptureBehavior(canvas3D, 0, 2000);
			capture.setSchedulingBounds(bounds);
			objRoot.addChild(capture);
		}

		// Optimize the scene graph
		objRoot.compile();
		return objRoot;
	}
	private void initialize(boolean recording) {
		this.recording = recording;
		setLayout(new BorderLayout());
		GraphicsConfiguration config =
			SimpleUniverse.getPreferredConfiguration();
		canvas3D = new Canvas3D(config);
		add("Center", canvas3D);
		SimpleUniverse simpleU = new SimpleUniverse(canvas3D);

		//Position the view
		TransformGroup viewingPlatformGroup =
			simpleU.getViewingPlatform().getViewPlatformTransform();
		Transform3D t3d = new Transform3D();
		t3d.rotX(Math.PI/10);
		t3d.setTranslation(new Vector3f(0, 10, 300));
		viewingPlatformGroup.setTransform(t3d);

		BranchGroup scene = createSceneGraph(simpleU);
		simpleU.addBranchGraph(scene);

		canvas3D.getView().setBackClipDistance(500.0d);
		canvas3D.getView().setFrontClipDistance(0.1d);
		canvas3D.getView().setMinimumFrameCycleTime(0);
		canvas3D.getView().setTransparencySortingPolicy(View.TRANSPARENCY_SORT_GEOMETRY);
	}

}
