// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.hpp;

/**
 * This classes allows adjustment of the brightness (DC offset) and
 * contrast (intensity stretch) of an image.
 */
public class LinearMappingProcessor extends j2d.ImageProcessor {

    private int lookupTable[] = new int[256];
    private int min;
    private int max;

    /**
     * Create a new LinearMappingProcessor
     */
    public LinearMappingProcessor(){
        for (int j = 0; j< 256; j++){
            lookupTable[j] = j;
        }
    }

    /**
     * Implementation of ImageProcessor
     */
    public void processImage() throws Exception{
        int pixels[];
        int r, g, b;

        pixels = getPixels();

        for (int i = 0; i<pixels.length; i++){

            // Separate RGB components
            r = (pixels[i] & 0x00FF0000) >> 16;
            g = (pixels[i] & 0x0000FF00) >> 8;
            b = (pixels[i] & 0x000000FF);

            // Adjust the pixel
            r = lookupTable[r];
            g = lookupTable[g];
            b = lookupTable[b];

            // store the processed pixel
            pixels[i] = 0xFF000000 | (r << 16) | (g << 8) | b;
        }

        setPixels(pixels);
    }

    /**
     * Set the processing parameters.
     */

    public void setParameters(double offset, double slope){
        double newj;
        for (int j = 0; j<256; j++){
            // new intensity is a stretched and offset version of the original.
            newj = j*slope + offset;
            // saturate if the stretching takes the intensities out of range.
            lookupTable[j] = (int)Math.min(255, newj);
            lookupTable[j] = Math.max(0, lookupTable[j]);
        }
    }

    /**
     * Set offset and slope to maximize the use of the dynamic range.
     * To apply the parameters to the image, this must be followed by
     * a call to performAlgorithm().
     *
     * @return              Two-element array.  Element 0 contains the
     *                      offset (brightness adjustment).  Element 1
     *                      contains the slope (contrast adjustment).
     */
    public float[] setOptimalParameters(){
        float params[] = new float[2];
        int Dmin = 0;
        int Dmax = 255;

        computeExtent();

        float deltaV = max - min;
        float deltaD = Dmax - Dmin;

        params[1] = deltaD / deltaV;
        params[0] = (Dmin * max - Dmax * min) / deltaV;

        setParameters(params[0], params[1]);
        return params;
    }

    /**
     * Compute the minimum and maximum intensity values.
     */
    private void computeExtent() {
        int pixels[];
        int r, g, b;
        min = Integer.MAX_VALUE;
        max = Integer.MIN_VALUE;

        pixels = getPixels();

        for (int i = 0; i<pixels.length; i++){

            // Separate RGB components
            r = (pixels[i] & 0x00FF0000) >> 16;
            g = (pixels[i] & 0x0000FF00) >> 8;
            b = (pixels[i] & 0x000000FF);

            //min = Math.min((r+g+b)/3, min);
            //max = Math.max((r+g+b)/3, max);
            min = Math.min(r, min);
            min = Math.min(g, min);
            min = Math.min(b, min);
            max = Math.max(r, max);
            max = Math.max(g, max);
            max = Math.max(b, max);
        }
    }
}
