package j2d.filters.bufferedImageConvolution;

import j2d.ImageUtils;
import j2d.border.BorderUtils;
import math.Mat2;
import utils.CompactJava;

import java.awt.*;
import java.awt.image.Kernel;
import java.io.*;
import java.util.prefs.Preferences;

/**
 * Robert Distinti
 * The image matrix bean -- modified from DocJava PreferencesBean
 */
public class ImageMatrixBean implements Serializable {
    private static final String key = "ece430.ImageMatrixBean";

    private byte borderType;
    private byte[][] data = new byte[5][5];
    private int filterType = 0;
    private transient ImageMatrixBeanProcessorInterface imbpi;

    // This returns a count of the number of boxes checked
    public int getCount() {
        int matrixSize = data.length;
        int cnt=0;
        int x, y;
        for (x = 0; x < matrixSize; x++) {
            for (y = 0; y < matrixSize; y++) {
                if (data[x][y]>0){
                    cnt++;
                }
            }
        }
        return cnt;
    }
    /**
     * saves the properties to the Preferences of the userRoot
     */
    public void save() {
        try {
            Preferences p = Preferences.userRoot();
            ByteArrayOutputStream baos = new
                    ByteArrayOutputStream();
            ObjectOutputStream oos = new
                    ObjectOutputStream(baos);
            oos.writeObject(this);
            baos.close();
            p.putByteArray(key, baos.toByteArray());
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public String toString() {
        return CompactJava.toXml(this);
    }

    /**
     * restores the properties from the preference in the user root.
     */
    public static ImageMatrixBean restore() {
        try {
            Preferences p = Preferences.userRoot();
            byte b [] = p.getByteArray(key, null);
            if (b == null)
                return new ImageMatrixBean();
            ByteArrayInputStream bais = new
                    ByteArrayInputStream(b);
            ObjectInputStream ois = new
                    ObjectInputStream(bais);
            Object o = ois.readObject();
            bais.close();
            return (ImageMatrixBean) o;
        } catch (IOException e) {
            //e.printStackTrace();
        } catch (ClassNotFoundException e) {
            // e.printStackTrace();
        }
        return new ImageMatrixBean();
    }



    public byte[][] getData() {
        return data;
    }

    public Kernel getKernel() {
        int matrixSize = data.length;
        float[] rtn = new float[matrixSize * matrixSize];
        int x, y, z;
        float sum = 0;
        z = 0;
        for (x = 0; x < matrixSize; x++) {
            for (y = 0; y < matrixSize; y++) {
                rtn[z] = data[x][y];
                sum += rtn[z];
                z++;
            }
        }
        // now normalize
        if (sum != 0) {
            for (x = 0; x < rtn.length; x++) {
                rtn[x] /= sum;
                System.out.println(rtn[x]);
            }
        }
        return new Kernel(matrixSize, matrixSize, rtn);
    }

    public void setData(byte[][] data) {
        this.data = data;

    }

    public int getMatrixSize() {
        return data.length;
    }

    public void setMatrixSize(int matrixSize) {
        int i, j, ofst, c;
        // first make sure it is odd
        matrixSize = matrixSize | 1;
        byte newData[][] = new byte[matrixSize][matrixSize];
        // first see if the size has changed
        int thisMatrixSize = data.length;
        if (thisMatrixSize == matrixSize) return;
        // see if we are expanding or contracting
        if ((thisMatrixSize > matrixSize)) {
            //contracting -- just take center elements
            ofst = (thisMatrixSize - matrixSize) / 2;
            c = matrixSize;
            for (i = 0; i < c; i++) {
                for (j = 0; j < c; j++) {
                    newData[i][j] = data[i + ofst][j + ofst];
                }
            }
        } else {
            // expanding -- keep it centered
            ofst = (matrixSize - thisMatrixSize) / 2;
            c = thisMatrixSize;
            for (i = 0; i < c; i++) {
                for (j = 0; j < c; j++) {
                    newData[i + ofst][j + ofst] = data[i][j];
                }
            }
        }
        data = newData;
    }

    public byte getBorderType() {
        return borderType;
    }

    public void setBorderType(byte value) {
        borderType = value;
    }


    public String getString() {
        byte b[][] = getData();
        StringBuffer sb = new StringBuffer();
        for (int x = 0; x < b.length; x++) {
            for (int y = 0; y < b[0].length; y++) {
                sb.append(b[x][y]).append(" ");
            }
            sb.append('\n');
        }
        return sb.toString();
    }

    /**
     * "convolution",
     * "fftr2",
     * "pfa"
     */
    public void setFilterType(int selectedIndex) {
        filterType = selectedIndex;
    }

    /**
     * "convolution",
     * "fftr2",
     * "pfa"
     */
    public int getFilterType() {
        return filterType;
    }

    /**
     * create an image that has 0 and 255 for the pixels
     *
     * @return Image of the ImageMatrixBean
     */
    public Image getImage() {
        return ImageUtils.getImage(
                Mat2.scale(
                        Mat2.getShort(data),
                        (short) 255));
    }

    public ImageMatrixBeanProcessorInterface getImageProcessor() {
        imbpi.setImageMatrixBean(this);
        return imbpi;
    }

    public void setImageProcessor(ImageMatrixBeanProcessorInterface imbpi) {
        this.imbpi = imbpi;
    }
    /**
     * make an image that is widthxheight in size. Use
     * the kernal that is mxn in size and add to the top, bottom
     * left and right. For example, if you want a 10x10 and the kernal
     * is 3x3, you add
     * 10/2 - 3/2 = 4 to the left
     * 10/2 - 3/2 = 4 to the top
     * the bottom will be the top - 1 and the right will be the left - 1;
     * So add 3 to the right and 3 to the bottom.
     *
     * @param width
     * @param height
     * @return grown image
     */
    public Image getImage(int width, int height) {
        int left = (width - data.length)/2;
        int top = (height - data[0].length)/2;
        int bottom = height-data[0].length-top;
        int right = width - data.length-left;
        return BorderUtils.getBorderImage(
                        getImage(),
                        left, right, top, bottom, (int) getBorderType());
    }
}
