// ImagePanel.java
// ImagePanel contains an image for display.  The image is
// converted to a BufferedImage for filtering purposes.
package j2d.filters;

// Java core packages

import j2d.BufferedImageProcessor;
import j2d.ImageUtils;

import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;

public class ImagePanel extends JPanel {

    private BufferedImage bufferedDisplayImage; // filtered image
    private BufferedImage bufferedOriginalImageCopy; // original image copy
    private Image image; // image to load

    // ImagePanel constructor
    public ImagePanel(File f) {

        this(ImageUtils.getGifJpgPngImage(f));

    } // end ImagePanel constructor

    public  ImagePanel(Image im) {
        setImage(im);
        ImageUtils.waitForImage(this, getImage());

        // create BufferedImage from Image
        setBufferedOriginalImageCopy(makeBlankBufferedImage(getImage().getWidth(null), getImage().getHeight(null)));

        //displayImage = new BufferedImage( image.getWidth( null ),
        //   image.getHeight( null ), BufferedImage.TYPE_INT_RGB );

        setBufferedDisplayImage(getBufferedOriginalImageCopy());

        // draw Image on Graphics2D object
        Graphics2D graphics = getBufferedDisplayImage().createGraphics();
        graphics.drawImage(getImage(), null, null);
    }

    public static BufferedImage makeBlankBufferedImage(int w, int h) {
        return new BufferedImage(w,
                h, BufferedImage.TYPE_INT_RGB);
    }

    // apply Java2DImageFilter to Image
    public void applyFilter(BufferedImageProcessor filter) {
        // process Image using ImageFilter
        setBufferedDisplayImage(filter.process(getBufferedDisplayImage()));
        repaint();
    }

    // set Image to originalImage
    public void displayOriginalImage() {
        setBufferedDisplayImage(makeBlankBufferedImage(getImage().getWidth(this), getImage().getHeight(this)));

        Graphics2D graphics = getBufferedDisplayImage().createGraphics();
        graphics.drawImage(getBufferedOriginalImageCopy(), null, null);
        repaint();
    }

    // draw ImagePanel
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D graphics = (Graphics2D) g;
        graphics.drawImage(getBufferedDisplayImage(), 0, 0, null);
    }

    // get preferred ImagePanel size
    public Dimension getPreferredSize() {
        return new Dimension(getBufferedDisplayImage().getWidth(),
                getBufferedDisplayImage().getHeight());
    }

    // get minimum ImagePanel size
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

    public BufferedImage getBufferedDisplayImage() {
        return bufferedDisplayImage;
    }

    public void setBufferedDisplayImage(BufferedImage bufferedDisplayImage) {
        this.bufferedDisplayImage = bufferedDisplayImage;
    }

    public BufferedImage getBufferedOriginalImageCopy() {
        return bufferedOriginalImageCopy;
    }

    public void setBufferedOriginalImageCopy(BufferedImage bufferedOriginalImageCopy) {
        this.bufferedOriginalImageCopy = bufferedOriginalImageCopy;
    }

    public Image getImage() {
        return image;
    }

    public void setImage(Image image) {
        this.image = image;
    }
}


