package j2d.border;

import com.sun.media.jai.widget.DisplayJAI;
import gui.ClosableJFrame;
import gui.run.RunComboBox;
import gui.run.RunSpinnerSlider;

import javax.media.jai.PlanarImage;
import javax.swing.*;
import javax.swing.event.ChangeEvent;
import java.awt.*;
import java.awt.event.ActionEvent;

import j2d.ImageUtils;

/**
 * This class demonstrates the use of the border operator. The user can
 * interactively choose the image border dimensions and the border type.
 */
public class BorderFrame extends
        ClosableJFrame {
    // The component and original image.
    private DisplayJAI display;
    private PlanarImage image;

    // A JComboBox for the existing border types.
    private RunComboBox borderTypes;

    // JSpinners for the four border dimensions.
    private RunSpinnerSlider top, left, right, bottom;

    /**
     * The constructor for the class, which sets its GUI.
     *
     * @param image the image which will receive a border and be
     *              displayed.
     */
    public BorderFrame(PlanarImage image) {
        super("BorderFrame operator example");
        this.image = image;
        // Create a control panel for the combo box and spinners.
        JPanel controlPanel = new JPanel(new GridLayout(5, 2));
        // Add the border type combo box.

        borderTypes = new RunComboBox(new String[]{"Zero",
                                                   "Constant",
                                                   "Copy",
                                                   "Reflection",
                                                   "Wrap"}) {
            public void run() {
                resetImage();
            }
        };
        controlPanel.add(new JLabel("BorderFrame Type"));
        controlPanel.add(borderTypes);
        // Add the four sides spinners. We need four instances of SpinnerModel...
        top = new RunSpinnerSlider(new SpinnerNumberModel(10, 0, 1000, 5)) {
            public void run() {
                resetImage();
            }
        };
        controlPanel.add(new JLabel("Top border"));
        controlPanel.add(top);
        bottom = new RunSpinnerSlider(new SpinnerNumberModel(10, 0, 1000, 5)) {
            public void run() {
                resetImage();
            }
        };
        controlPanel.add(new JLabel("Bottom border"));
        controlPanel.add(bottom);
        left = new RunSpinnerSlider(new SpinnerNumberModel(10, 0, 1000, 5)) {
            public void run() {
                resetImage();
            }
        };
        controlPanel.add(new JLabel("Left border"));
        controlPanel.add(left);
        right = new RunSpinnerSlider(new SpinnerNumberModel(10, 0, 1000, 5)) {
            public void run() {
                resetImage();
            }
        };

        controlPanel.add(new JLabel("Right border"));
        controlPanel.add(right);
        // Small trick to make the control panel look better.
        JPanel positionControlPanel = new JPanel(new BorderLayout());
        positionControlPanel.add(controlPanel, BorderLayout.NORTH);
        // Create the display component.
        display = new DisplayJAI(image); // Temporarily uses the original image.
        resetImage(); // Now that the instance of DisplayJAI is created, set
        // an image with borders.
        // Add to the JFrame's ContentPane the display and the control panel.
        getContentPane().add(new JScrollPane(display),
                BorderLayout.CENTER);
        getContentPane().add(positionControlPanel, BorderLayout.EAST);
        // Set the closing operation so the application is finished.
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        pack(); // adjust the frame size using preferred dimensions.
        setVisible(true);
    }

    /**
     * This method will be called when the spinners' contents change.
     */
    public void stateChanged(ChangeEvent e) {
        resetImage();
    }

    /**
     * This method will be called when the combo box contents change.
     */
    public void actionPerformed(ActionEvent e) {
        resetImage();
    }

    /**
     * This method will create a border around the image with the values
     * from the spinners and combo box. The new image will be used for
     * displaying.
     */
    private void resetImage() {
        // Gets values from the spinners and combo box.
        int topValue = top.getValue();
        int bottomValue = bottom.getValue();
        int leftValue = left.getValue();
        int rightValue = right.getValue();
        int borderType = borderTypes.getSelectedIndex();
        // Create a ParameterBlock for the border operation.
        PlanarImage borderImage = BorderUtils.getBorderImage(image,
                leftValue,
                rightValue,
                topValue,
                bottomValue,
                borderType);
        display.set(borderImage);
    }

    public static void main(String[] args) {

        // Read the image.
        //PlanarImage input = JAI.create("fileload",
        //        futils.Futil.getReadFileName("select an image"));
        Image img = ImageUtils.getImage();
        PlanarImage input = ImageUtils.getPlanarImage(img);
        // Create an instance of the JFrame.
        new BorderFrame(input);
    }

}
