/***************************************************************************

 A log-polar transformation of an image.
 The rows and columns of the
 displayed image correspond to radial and angular coordinates,
 respectively.
 ***************************************************************************/
package j2d;

import java.awt.*;
import java.awt.image.BufferedImage;

public class LogXform implements ImageProcessorInterface{


    private BufferedImage sourceImage;   // original image
    private Rectangle imageRect;         // rectangle holding image dimensions
    private int numSectors, numRings;    // sampling density in log-polar space
    private float cx, cy;                // image centre coordinates
    private float dr, dtheta;            // increments in r and theta
    private float[] ctheta;              // lookup table for cos
    private float[] stheta;              // lookup table for sin


    public LogXform(int sectors, int rings) {
        numSectors = sectors;
        numRings = rings;


    }

    private void init(Image image) {
        sourceImage = ImageUtils.getBufferedImage(image);

        computeMappingParameters();
    }

    public void computeMappingParameters() {

        // Determine image dimensions and centre

        imageRect =
                new Rectangle(sourceImage.getWidth(), sourceImage.getHeight());
        cx = sourceImage.getWidth() / 2.0f;
        cy = sourceImage.getHeight() / 2.0f;

        // Compute r and theta increments

        dr = (float) (Math.sqrt(cx * cx + cy * cy) / numRings);
        dtheta = 360.0f / numSectors;

        // Compute lookup tables for cos and sin

        ctheta = new float[numSectors];
        stheta = new float[numSectors];
        double theta = 0.0, dt = dtheta * (Math.PI / 180.0);
        for (int j = 0; j < numSectors; ++j, theta += dt) {
            ctheta[j] = (float) Math.cos(theta);
            stheta[j] = (float) Math.sin(theta);
        }

    }


    // Create a log-polar warped version of source image

    public BufferedImage getLogPolarImage() {

        BufferedImage logPolarImage =
                new BufferedImage(numSectors, numRings, sourceImage.getType());

        // Fill first ring (r = 0) with data

        int centralValue = sourceImage.getRGB(Math.round(cx), Math.round(cy));
        for (int j = 0; j < numSectors; ++j) {
            logPolarImage.setRGB(j, 0, centralValue);
        }

        // Fill rings with r > 0

        int x, y;
        float r = dr;
        for (int k = 1; k < numRings; ++k, r += dr)
            for (int j = 0; j < numSectors; ++j) {
                x = Math.round(cx + r * ctheta[j]);
                y = Math.round(cy + r * stheta[j]);
                if (imageRect.contains(x, y))
                    logPolarImage.setRGB(j, k, sourceImage.getRGB(x, y));
                else
                    logPolarImage.setRGB(j, k, 0);
            }

        return logPolarImage;

    }

    public Image process(Image image) {
        init(image);
        return ImageUtils.getImage(getLogPolarImage());
    }
}
