package j2d;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageDecoder;
import com.sun.image.codec.jpeg.JPEGImageEncoder;
import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.ImageEncoder;
import com.sun.media.jai.codec.TIFFEncodeParam;
import com.sun.media.jai.widget.DisplayJAI;
import futils.Futil;
import futils.StreamSniffer;
import futils.WriterUtil;
import graphics.NumImage;
import gui.ClosableJFrame;
import gui.ImageBeanInterface;
import gui.In;
import ip.gui.frames.FFTFrame;
import ip.transforms.Kernels;
import j2d.border.BorderUtils;
import j2d.color.PseudoColorFilter;
import j2d.color.rgbImageFilters.GreyFilter;
import j2d.filters.GaussianSmoothingProcessor;
import j2d.filters.bufferedImageConvolution.ImageMatrixBean;
import j2d.hpp.*;
import j2d.io.gif.stills.WriteGIF;
import j2d.io.ppm.WritePPM;
import math.Mat2;
import math.MathUtils;
import math.fourierTransforms.r2.FFT2dComplex;
import utils.ResourceManager;
import utils.UByte;

import javax.imageio.ImageIO;
import javax.media.jai.*;
import javax.media.jai.operator.DFTDataNature;
import javax.media.jai.operator.DFTDescriptor;
import javax.media.jai.operator.DFTScalingType;
import javax.swing.*;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.*;
import java.awt.image.renderable.ParameterBlock;
import java.io.*;
import java.util.jar.JarInputStream;
import java.util.jar.JarOutputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;

// ...

/**
 * A toolkit of useful image processing methods. Declared as final, since
 * all methods are static.
 */
public final class ImageUtils {
    /**
     * make it so no one will instance this class.
     */
    private ImageUtils() {
    }

    public PlanarImage dftOP(PlanarImage image, Integer scalingType, Integer dataNature) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(scalingType).add(dataNature);
        return JAI.create("dft", pb);
    }

    public PlanarImage idftOP(PlanarImage image, Integer scalingType, Integer dataNature) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(scalingType).add(dataNature);
        return JAI.create("idft", pb);
    }


    public static Image getImage(TiledImage ti) {
        return getImage(ti.getAsBufferedImage());
    }

    public static Image concatenateImages(Image[] images,
                                          Dimension celldim,
                                          int cols,
                                          int rows) {
        final PlanarImage[] planarImages = getPlanarImages(images);
        final TiledImage tiledImage = concatenateImages(planarImages,
                celldim,
                cols,
                rows);
        return getImage(tiledImage);
    }

    public static PlanarImage[] getPlanarImages(Image images[]) {
        PlanarImage pi[] = new PlanarImage[images.length];
        for (int i = 0; i < images.length; i++) {
            pi[i] = getPlanarImage(images[i]);
        }
        return pi;

    }

    public static RenderedOp addImages(Image image1,
                                       Image image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(getPlanarImage(image1));
        pb.addSource(getPlanarImage(image2));
        return JAI.create("add", pb);
    }

    public static TiledImage concatenateImages(PlanarImage planarImageArray[],
                                               Dimension celldim,
                                               int cols,
                                               int rows) {
        int imWid = cols * celldim.width;
        int imHt = rows * celldim.height;
        PlanarImage img = planarImageArray[0];
        TiledImage ti = new TiledImage(0, 0, imWid, imHt, 0, 0,
                img.getSampleModel(), img.getColorModel());

        Graphics2D tg2d = ti.createGraphics();
        int x = 0;
        int y = 0;
        for (int i = 0; i < rows; i++) {
            for (int j = 0; j < cols; j++) {
                if (i * cols + j >= planarImageArray.length) break;
                img = planarImageArray[i * cols + j];
                if (img != null) {
                    double magx = celldim.width / (double) img.getWidth();
                    double magy = celldim.height / (double) img.getHeight();
                    AffineTransform atx = new AffineTransform();
                    atx.setToTranslation((double) x, (double) y);
                    atx.scale(magx, magy);
                    tg2d.drawRenderedImage(img, atx);
                    x += celldim.width;
                }
            }
            x = 0;
            y += celldim.height;
        }
        return ti;
    }

    /**
     * Do a bilinear inverse affine transform and throw and exception if
     * the xform is not invertable.
     *
     * @param scalex  scale about transx and transy
     * @param shearx  shear about transx and transy
     * @param transx  point about which xform is performed
     * @param radians rotation
     * @return transformed image.
     */
    public static BufferedImage affineXform(double scalex, double scaley,
                                            double shearx, double sheary,
                                            double transx, double transy,
                                            double radians,
                                            BufferedImage bi) {
        AffineTransform tx = new AffineTransform();
        tx.scale(scalex, scaley);
        tx.shear(shearx, sheary);
        tx.translate(transx, transy);
        tx.rotate(radians,
                bi.getWidth() / 2,
                bi.getHeight() / 2);

        AffineTransformOp op = new AffineTransformOp(tx,
                AffineTransformOp.TYPE_BILINEAR);
        return op.filter(bi, null);
    }

    public static Image getPpmJar() throws IOException {
        return getPpmJar(Futil.getReadFile("select a ppm.jar file"));
    }

    public static Image getPpmJar(File f) throws IOException {
        JarInputStream in = new JarInputStream(new FileInputStream(f));
        in.getNextEntry();
        //System.out.println("zip name = " + ze.getName());
        //System.out.println("zip size = " + ze.getSize());
        char c1, c2;

        c1 = (char) ShortImageBean.readByte(in);
        c2 = (char) ShortImageBean.readByte(in);

        if (c1 != 'P') {
            throw new IOException("not a PPM file");
        }
        if (c2 != '6') {
            throw new IOException("not a PPM file");
        }
        final int w = ShortImageBean.readInt(in);
        final int h = ShortImageBean.readInt(in);

// Read maximum value of each color, and ignore it.
// In PPM_RAW we know r,g,b use full range (0-255).
        ShortImageBean.readInt(in);
        ShortImageBean sib = new ShortImageBean(w, h);
        short r[][] = sib.getR();
        short g[][] = sib.getG();
        short b[][] = sib.getB();
        byte buf[] = new byte[4096];

        int read;
        byte[] allb = new byte[w * h * 3];
        int indx = 0;
        while ((read = in.read(buf)) != -1) {
            //System.out.println("read = " + read);
            for (int i = 0; i < read; i++) {
                allb[indx] = buf[i];
                indx = indx + 1;
            }
        }
        int j = 0;
        for (int col = 0; col < h; col++)
            for (int row = 0; row < w; row++) {
                r[row][col] = UByte.us(allb[j++]);
                g[row][col] = UByte.us(allb[j++]);
                b[row][col] = UByte.us(allb[j++]);
            }
        //System.out.println("j= " + j);
        return sib.getImage();
    }

    public static Image getPsd(Image img) {
        FFTFrame fftFrame = new FFTFrame("fftFrame");
        fftFrame.setVisible(false);
        fftFrame.setImage(img);
        fftFrame.fftR2();
        return fftFrame.getImage();
    }

    public static void main(String args[]) {
        quadSwapTest();
    }

    public static void testPsdImage() {
        Image img = getImage();
        displayImage(getPsd(img), "a nice Image!");
    }


    public static void saveAsPPMJar(Image img, File f) {
        ShortImageBean sib = new ShortImageBean(img);
        int indx = f.getName().indexOf(".jar");
        String zEntryName = f.getName().substring(0, indx);
        WritePPM wppm = new WritePPM(sib.getWidth(), sib.getHeight());
        try {
            JarOutputStream
                    os = new JarOutputStream(new FileOutputStream(f));
            os.putNextEntry(new ZipEntry(zEntryName));
            wppm.writeHeader(os);
            wppm.writeImage(os, sib.getR(), sib.getG(), sib.getB());
            os.finish();
            os.close();
        } catch (Exception e) {
            System.out.println("Save PPM Exception - 2!");
            e.printStackTrace();
        }
    }


    public static void saveAsPPMgz(Image img, String fn) {
        ShortImageBean sib = new ShortImageBean(img);
        WritePPM wppm = new WritePPM(sib.getWidth(), sib.getHeight());
        try {
            GZIPOutputStream
                    os = new GZIPOutputStream(new FileOutputStream(fn));
            wppm.writeHeader(os);
            wppm.writeImage(os, sib.getR(), sib.getG(), sib.getB());
            os.finish();
            os.close();
        } catch (Exception e) {
            System.out.println("Save PPM Exception - 2!");
        }
    }

    public static void saveAsPPMgz(Image img) {
        String fn = WriterUtil.getSaveFileName("Save as PPM.gz");
        if (fn == null) return;
        saveAsPPMgz(img, fn);
    }

    public static void saveAsPPM(Image img, File f) {
        ShortImageBean sib = new ShortImageBean(img);
        WritePPM.doIt(sib.getR(),
                sib.getG(),
                sib.getB(),
                f.toString());
    }

    public static void printImageWriterFormats() {
        //print(ImageIO.getWriterFormatNames());
    }

    public static void printImageReaderFormats() {
        String s[] = null; //ImageIO.getReaderFormatNames();
        print(s);
    }

    public static void print(Object o[]) {
        for (int i = 0; i < o.length; i++)
            System.out.println(o[i]);
    }

    public static java.awt.Image getImage(java.awt.Component c) {
        java.awt.Dimension d = c.getSize();
        java.awt.Frame f = new java.awt.Frame();
        f.addNotify();
        f.setSize(d);
        java.awt.Image i = f.createImage(d.width, d.height);

        c.paint(i.getGraphics());
        return i;
    }

    public static void testGetImage() {
        ClosableJFrame cjf = new ClosableJFrame("test image");
        Container c = cjf.getContentPane();
        c.setLayout(new GridLayout(1, 0));
        c.add(new ImagePanel(getImage()));
        c.add(new ImagePanel(getImage()));
        c.add(new ImagePanel(getImage()));
        c.add(new ImagePanel(getImage()));
        cjf.setSize(200, 200);
        cjf.setVisible(true);
    }

    public static java.awt.Image getImage() {
        java.io.File f =
                Futil.getReadFile("select an image");
        if (f == null) return null;
        InputStream is;
        try {
            is = new FileInputStream(f);
            new StreamSniffer(is);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return getGifJpgPngImage(f);
    }

    public static Image getImage(String fn) {

        File f = new File(fn);
        if (f.exists()) {
            final int imageType = StreamSniffer.getImageType(f);

            if (isGifPngOrJpg(imageType))
                return getGifJpgPngImage(f);
            else if (isPpm(imageType))
                return getPpmImage(f);
            else
                In.message("format not supported");
        }
        In.message(f +
                " not found ImageUtils.getImage,using cameraman image");
        return NumImage.getImage();
    }

    private static Image getPpmImage(File f) {
        ShortImageBean sib = null;
        try {
            sib = ShortImageBean.readPpm(f);
        } catch (IOException e) {
            In.message(e);

        }
        if (sib == null) return null;
        return sib.getImage();

    }

    private static boolean isPpm(int streamClass) {
        if (streamClass == StreamSniffer.PPM_RAWBITS) return true;
        return streamClass == StreamSniffer.PPM;
    }

    private static boolean isGifPngOrJpg(int i) {
        if (i == StreamSniffer.GIF87a) return true;
        if (i == StreamSniffer.GIF89a) return true;
        if (i == StreamSniffer.PNG_IMAGE) return true;
        if (i == StreamSniffer.JPEG) return true;
        return i == StreamSniffer.JPG;
    }

    public static java.awt.Image getGifJpgPngImage(java.io.File f) {
        java.awt.Image i =
                java.awt.Toolkit.getDefaultToolkit().getImage(f.toString());
        waitForImage(new gui.ClosableJFrame(), i);
        System.out.println("getGifJpgPngImage returns");
        return i;
    }

    public Image getGifOrJpg(File imageFile) {
        return Toolkit.getDefaultToolkit()
                .getImage(imageFile.getAbsolutePath());
    }

    public static void waitForImage(java.awt.Component c,
                                    java.awt.Image image) {
        if (c == null) waitForImage(image);
        if (image == null) System.out.println("Image is Null!");
        java.awt.MediaTracker tracker = new java.awt.MediaTracker(c);
        try {
            if (image == null) In.message("Image is null!");
            tracker.addImage(image, 0);
            tracker.waitForID(0);
        } catch (Exception e) {
            In.message(e);
        }
    }

    public static void waitForImage(java.awt.Image image) {
        JPanel jp = new JPanel();
        waitForImage(jp, image);
    }

    public static Image byte2Image(byte r[][]) {

        int w = r.length;
        int h = r[0].length;
        int v;
        Toolkit tk = Toolkit.getDefaultToolkit();
        int pels[] = new int[w * h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                if (r[x][y] == 1)
                    v = 255;
                else
                    v = 0;
                pels[y + x * h] =
                        0xff000000
                                |
                                (v << 16)
                                |
                                (v << 8)
                                | v;
            }
        return tk.createImage(new MemoryImageSource(w,
                h,
                ColorModel.getRGBdefault(),
                pels, 0,
                w));
    }

    public static Image short2Image(short r[][]) {

        int w = r.length;
        int h = r[0].length;
        int v;
        Toolkit tk = Toolkit.getDefaultToolkit();
        int pels[] = new int[w * h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                v = r[x][y];
                pels[y + x * h] =
                        0xff000000
                                |
                                (v << 16)
                                |
                                (v << 8)
                                | v;
            }
        return tk.createImage(new MemoryImageSource(w,
                h,
                ColorModel.getRGBdefault(),
                pels, 0,
                w));
    }


    public static Image getImage(short r[][], short g[][], short b[][],
                                 HppFilterInterface f) {

        int w = r.length;
        int h = r[0].length;

        return getImage(getPels(r, g, b, f),
                w, h);
    }

    public static Image getImage(short r[][], HppFilterInterface f) {
        int w = r.length;
        int h = r[0].length;
        return getImage(getPels(r, f),
                w, h);
    }

    private static int[] getPels(short[][] r, HppFilterInterface f) {
        int w = r.length;
        int h = r[0].length;

        int pels[] = new int[w * h];
        int v;
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                v = r[x][y];
                pels[y + x * h] =
                        packInt(f.getR(v),
                                f.getG(v),
                                f.getB(v));
            }
        return pels;
    }

    public static int[] short2Pels(short r[][], short g[][], short b[][]) {
        int width = r.length;
        int height = r[0].length;
        int pels[] = new int[width * height];
        for (int x = 0; x < width; x++)
            for (int y = 0; y < height; y++)
                pels[x + y * width]
                        = packInt(r[x][y],
                        g[x][y],
                        b[x][y]);
        return pels;
    }

    /**
     * Get the red part of a packed rgb int.
     *
     * @param rgb a pixel in argb format
     * @return just the red part.
     */
    public static short getRed(int rgb) {
        return (short) ((rgb & 0xFF0000) >> 16);
    }

    /**
     * Get the green part of a packed rgb int
     *
     * @param rgb a pixel in argb format
     * @return just the green part.
     */
    public static short getGreen(int rgb) {
        return (short) ((rgb & 0x00FF00) >> 8);
    }

    /**
     * Get the blue part of a packed rgb int.
     *
     * @param rgb a pixel in argb format
     * @return just the blue part
     */
    public static short getBlue(int rgb) {
        return (short) ((rgb & 0x0000FF));
    }

    /**
     * Adapter design pattern
     *
     * @param f a standard RGBImageFilter
     * @return HppFilter3 Homogenious point processor
     */
    public static HppFilter3Interface getHppFilter3(final RGBImageFilter f) {
        return new HppFilter3Interface() {
            public short getR(int r, int g, int b) {
                return getRed(f.filterRGB(0, 0, ImageUtils.packInt(r, g, b)));
            }

            public short getG(int r, int g, int b) {
                return getGreen(f.filterRGB(0, 0, ImageUtils.packInt(r, g, b)));
            }

            public short getB(int r, int g, int b) {
                return getBlue(f.filterRGB(0, 0, ImageUtils.packInt(r, g, b)));
            }
        };
    }

    public static RGBImageFilter getRGBImageFilter(final HppFilter3Interface f) {
        return new RGBImageFilter() {
            public int filterRGB(int x, int y, int rgb) {
                Color c = new Color(rgb);
                int r = c.getRed();
                int g = c.getGreen();
                int b = c.getBlue();
                return packInt(f.getR(r, g, b),
                        f.getB(r, g, b),
                        f.getG(r, g, b));
            }
        };
    }

    private static int[] getPels(short r[][],
                                 short g[][],
                                 short b[][],
                                 HppFilterInterface f) {
        int w = r.length;
        int h = r[0].length;
        int pels[] = new int[w * h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++)
                pels[y + x * h] =
                        packInt(f.getR(r[x][y]),
                                f.getG(g[x][y]),
                                f.getB(b[x][y]));
        return pels;
    }

    /**
     * Return an int packed as ARGB
     *
     * @param r red component (0..255)
     * @param g green component (0..255)
     * @param b blue component (0..255)
     * @return the packed int.
     */
    public static int packInt(int r, int g, int b) {
        return 0xFF000000
                |
                ((0xFF & r) << 16)
                |
                ((0xFF & g) << 8)
                | (0xFF & b);
    }

    /**
     * Get a convole operator instance from a <code>Kernel</code>
     * instance.
     *
     * @param k The <code> kernel</code>instance
     * @return an instance of a <code>ConvolveOp</code>.
     */
    public static ConvolveOp getConvolveOp(Kernel k) {
        return new ConvolveOp(k,
                ConvolveOp.EDGE_ZERO_FILL,
                null);
    }

    public static Kernel makeKernel(int w, int h, float blurMatrix[]) {
        return new Kernel(w, h, blurMatrix);
    }

    public static Kernel makeKernel(float k2d[][]) {
        int w = k2d.length;
        int h = k2d[0].length;
        float k[] = new float[w * h];
        int x = 0;
        for (int i = 0; i < w; i++)
            for (int j = 0; j < h; j++, x++)
                k[x] = k2d[i][j];
        return new Kernel(w, h, k);
    }

    public static boolean hasAlpha(Image image) {
        // If buffered image, the color model is readily available
        if (image instanceof BufferedImage) {
            BufferedImage bimage = (BufferedImage) image;
            return bimage.getColorModel().hasAlpha();
        }

        // Use a pixel grabber to retrieve the image's color model;
        // grabbing a single pixel is usually sufficient
        PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
        try {
            pg.grabPixels();
        } catch (InterruptedException e) {
            In.message(e);
        }

        // Get the image's color model
        ColorModel cm = pg.getColorModel();
        return cm.hasAlpha();
    }

    public static BufferedImage getBufferedImage2(Image image) {
        if (image instanceof BufferedImage) {
            return (BufferedImage) image;
        }

        // This code ensures that all the pixels in the image are loaded
        image = new ImageIcon(image).getImage();

        // Determine if the image has transparent pixels; for this method's
        // implementation, see e665 Determining If an Image Has Transparent Pixels
        boolean hasAlpha = hasAlpha(image);
        int w = image.getWidth(null);
        int h = image.getHeight(null);

        // Create a buffered image with a format that's compatible with the screen
        BufferedImage bimage = new BufferedImage(w,
                h,
                BufferedImage.TYPE_INT_RGB);
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        try {
            // Determine the type of transparency of the new buffered image
            int transparency = Transparency.OPAQUE;
            if (hasAlpha) {
                transparency = Transparency.BITMASK;
            }

            // Create the buffered image
            GraphicsDevice gs = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gs.getDefaultConfiguration();
            bimage =
                    gc.createCompatibleImage(image.getWidth(null),
                            image.getHeight(null),
                            transparency);
        } catch (Exception e) {
            // The system does not have a screen
        }

        if (bimage == null) {
            // Create a buffered image using the default color model
            int type = BufferedImage.TYPE_INT_RGB;
            if (hasAlpha) {
                type = BufferedImage.TYPE_INT_ARGB;
            }
            bimage =
                    new BufferedImage(image.getWidth(null),
                            image.getHeight(null),
                            type);
        }

        // Copy image to buffered image
        Graphics g = bimage.createGraphics();

        // Paint the image onto the buffered image
        g.drawImage(image, 0, 0, null);
        g.dispose();

        return bimage;
    }


    public static BufferedImage getBufferedImage3(Image img) {
        ImageBean ib = new ImageBean();
        ib.setImage(img);
        return getBufferedImage(ib);
    }

    public static PlanarImage getPlanarImage(Image img) {
        return getPlanarImage(getBufferedImage(img));
    }

    public static PlanarImage getPlanarImage(BufferedImage bi) {
        if (bi == null) In.message("ImageUtils.PlanarImage, arg is null! ER!");
        PlanarImage planarImage = PlanarImage.wrapRenderedImage(bi);
        if (planarImage == null) In.message("ImageUtils.PlanarImage, could not make planarImage! ER!");
        return planarImage;
    }

    public static BufferedImage getBufferedImage(Image img) {
        int w = img.getWidth(null);
        int h = img.getHeight(null);
        BufferedImage bi = new BufferedImage(w,
                h,
                BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = bi.createGraphics();
        g2d.drawImage(img, 0, 0, w, h, null);
        return bi;
    }

    public static BufferedImage getBufferedImage(ImageBeanInterface ib) {
        int w = ib.getImageWidth();
        int h = ib.getImageHeight();
        BufferedImage bi = new BufferedImage(w,
                h,
                BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = bi.createGraphics();
        g2d.drawImage(ib.getImage(), 0, 0, w, h, null);
        return bi;
    }

    public static ImageBean getImageBean(BufferedImage bi) {
        ImageBean ib = new ImageBean();
        ib.setImage(Toolkit.getDefaultToolkit().createImage(new FilteredImageSource(bi.getSource(),
                new BufferedImageFilter(new AffineTransformOp(new AffineTransform(),
                        AffineTransformOp.TYPE_BILINEAR)))));
        return ib;
    }

    public static Image getImage(BufferedImage bi) {
        return
                Toolkit.getDefaultToolkit().createImage(new FilteredImageSource(bi.getSource(),
                        new BufferedImageFilter(new AffineTransformOp(new AffineTransform(),
                                AffineTransformOp.TYPE_BILINEAR))));
    }

    public static ImageBean changeColors(float colorMatrix[][],
                                         ImageBean ib) {
        return getImageBean(changeColors(colorMatrix, getBufferedImage(ib)));
    }

    public static Image changeColors(float colorMatrix[][],
                                     Image img) {
        ImageBean ib = new ImageBean();
        ib.setImage(img);
        ib = changeColors(colorMatrix, ib);

        return ib.getImage();
    }


    public static BufferedImage changeColors(float colorMatrix[][],
                                             BufferedImage bi) {
        // create filter to change colors
        BandCombineOp bco =
                new BandCombineOp(colorMatrix, null);

        // create source and display Rasters
        Raster inputRaster = bi.getRaster();

        WritableRaster outputRaster =
                inputRaster.createCompatibleWritableRaster();

        // filter Rasters with changeColors filter
        bco.filter(inputRaster, outputRaster);

        // create new BufferedImage from display Raster
        return new BufferedImage(bi.getColorModel(),
                outputRaster, true, null);
    }

    public static ColorModel getRgbColorModel() {
        return ColorModel.getRGBdefault();
    }

    public static void fitScreen(Component c) {
        Toolkit tk = Toolkit.getDefaultToolkit();
        Dimension d
                = tk.getScreenSize();
        c.setSize(d.width, d.height);
    }

    //todo this colol model is not always correct...BUG?
    public static Image getImage(PlanarImage pi) {
        ColorModel colorModel = pi.getColorModel();
        if (colorModel == null) {
            colorModel = FloatDoubleColorModel.getRGBdefault();
        }
        BufferedImage bi = pi.getAsBufferedImage(pi.getBounds(), colorModel);
        return getImage(bi);
    }

    public static Image getImage(Image img, final HppFilterInterface f) {
        RGBImageFilter rgbFilter = getRGBImageFilter(f);
        return getImage(img, rgbFilter);
    }

    public static Image getImage(Image img, RGBImageFilter rgbFilter) {
        ImageProducer ip = img.getSource();
        ip = new FilteredImageSource(ip, rgbFilter);
        Toolkit tk = Toolkit.getDefaultToolkit();
        return tk.createImage(ip);
    }

    private static RGBImageFilter getRGBImageFilter(final HppFilterInterface f) {
        return new RGBImageFilter() {
            public int filterRGB(int x, int y, int rgb) {
                Color c = new Color(rgb);
                int r = f.getR(c.getRed());
                int g = f.getG(c.getGreen());
                int b = f.getB(c.getBlue());
                return packInt(r, g, b);
            }
        };
    }

    public static Image getImage(int pels[], int w, int h) {
        Toolkit tk = Toolkit.getDefaultToolkit();
        return tk.createImage(new MemoryImageSource(w,
                h,
                getRgbColorModel(),
                pels, 0,
                w));
    }

    public static void testDctImage(Image image, String title) {
        ClosableJFrame cf = new ClosableJFrame(title);
        Container c = cf.getContentPane();
        ImagePanel ip = new ImagePanel(image);
        c.add(ip);
        c.setLayout(new FlowLayout());
        cf.pack();
        cf.setVisible(true);
        RenderedOp dctOp = ImageUtils.DCTImage(getPlanarImage(image));
        displayImage(dctOp.createInstance(), "DCTImage");
        RenderedOp iDctOp = ImageUtils.inverseDCTImage(dctOp);
        displayImage(iDctOp, "inverse dct");
    }

    public static RenderableOp testFftImage(Image image, String title) {
        ClosableJFrame cf = new ClosableJFrame(title);
        Container c = cf.getContentPane();
        ImagePanel ip = new ImagePanel(image);
        c.add(ip);
        c.setLayout(new FlowLayout());
        cf.pack();
        cf.setVisible(true);

        PlanarImage planarImage = getPlanarImage(image);
        RenderedOp dftop = DFTImage(planarImage,
                DFTDescriptor.SCALING_NONE,
                DFTDescriptor.REAL_TO_COMPLEX);

        RenderedOp magop = magnitudeImage(dftop);
        RenderedOp phaseop = magnitudeImage(dftop);

        //todo apply operator here!

        RenderedOp complexop = polarToComplexImage(magop, phaseop);
        DFTScalingType scalingNone = DFTDescriptor.SCALING_NONE;
        DFTDataNature complexToReal = DFTDescriptor.COMPLEX_TO_REAL;
        return idftImage(complexop, scalingNone, complexToReal);

    }

    public static RenderableOp idftImage(RenderedOp complexop, DFTScalingType scalingNone, DFTDataNature complexToReal) {
        PlanarImage pi = complexop.createInstance();
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(pi);
        pb.add(scalingNone).add(complexToReal);
        return JAI.createRenderable("idft", pb);
    }


    public static RenderableOp inverseDFTImage(RenderableOp image,
                                               DFTScalingType scalingType,
                                               DFTDataNature dataNature) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(scalingType).add(dataNature);
        return JAI.createRenderable("idft", pb);
    }


    public static void displayImage(Image image, String title) {
        ClosableJFrame cf = new ClosableJFrame(title);
        Container c = cf.getContentPane();
        ImagePanel ip = new ImagePanel(image);
        c.add(ip);
        c.setLayout(new FlowLayout());
        cf.pack();
        cf.setVisible(true);
    }

    public static Image getImage(short r[][], short g[][], short b[][]) {
        int w = r.length;
        int h = r[0].length;
        int pels[] = new int[w * h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++)
                pels[x + y * w]
                        = 0xFF000000
                        |
                        ((0xFF & r[x][y]) << 16)
                        |
                        ((0xFF & g[x][y]) << 8)
                        | (0xFF & b[x][y]);
        return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(w,
                h,
                ColorModel.getRGBdefault(),
                pels, 0,
                w));
    }

    /**
     * Use a radix 2 fft to filter img1 with img2.
     * Both images must be integral powers of two and both
     * must be of the same size.
     * @param img1 original image
     * @param img2 kernal image
     * @return filtered image
     */
    public static Image filterFFTR2(Image img1, Image img2){
        return FFT2dComplex.filter(img1,img2);
    }
    public static int[] getPels(Image image) {
        int w = image.getWidth(null);
        int h = image.getHeight(null);
        int pels[] = new int[w * h];

        try {
            new PixelGrabber(image, 0, 0,
                    w, h, pels, 0, w).grabPixels();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return pels;
    }

    public static int[] getPels(Image img, int width, int height) {
        int pels[] = new int[width * height];

        PixelGrabber grabber =
                new PixelGrabber(img, 0, 0,
                        width, height, pels, 0, width);

        try {
            grabber.grabPixels();
        } catch (InterruptedException e) {
            In.message(e);
        }
        return pels;
    }

    public static void pelsToShort(short r[][], short g[][], short b[][],
                                   int[] pels, int width, int height) {
        int i;
        ColorModel cm = getRgbColorModel();
        for (int x = 0; x < width; x++)
            for (int y = 0; y < height; y++) {
                i = x + y * width;
                b[x][y] = (short) cm.getBlue(pels[i]);
                g[x][y] = (short) cm.getGreen(pels[i]);
                r[x][y] = (short) cm.getRed(pels[i]);
            }
    }

    /**
     * This image converts to gray and rotates and mirrors the image
     * around. Do not use unless you know you need these extra features.
     *
     * @param g
     * @return a grayscale image
     */
    public static Image getGrayImage(short g[][]) {
        Toolkit tk =
                Toolkit.getDefaultToolkit();
        ColorModel cm = tk.getColorModel();
        int width = g.length;
        int height = g[0].length;
        int pels[] = new int[width * height];
        for (int x = 0; x < width; x++)
            for (int y = 0; y < height; y++) {
                pels[x + y * width] =
                        0xff000000
                                |
                                (g[y][x] << 16)
                                |
                                (g[y][x] << 8)
                                | g[y][x];
            }
        return tk.createImage(new
                MemoryImageSource(width,
                height,
                cm,
                pels, 0,
                width));
    }

    public static void writeHexImage(ImageBeanInterface ib) {
        WriterUtil.writeString(getHexImage(ib));
    }

    public static String getHexImage(ImageBeanInterface ib) {
        StringBuffer sb = new StringBuffer("");
        int width = ib.getImageWidth();
        int height = ib.getImageHeight();
        Image img = ib.getImage();

        int pixels[] = new int[width * height];
        PixelGrabber pg = new PixelGrabber(img,
                0,
                0,
                width,
                height,
                pixels,
                0,
                width);
        try {
            pg.grabPixels();
        } catch (InterruptedException e) {
            In.message(e);
        }
        String newline = "\n";
        int i;
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                i = x + y * width;

                sb.append("0x").append(Integer.toHexString(pixels[i])).append(", ");
            }
            sb.append(newline);
        }
        return sb.toString();
    }

    public static Image getImage(short g[][]) {
        return getTransposedImage(Mat2.transpose(g));
    }

    /**
     * This method appears to transpose images
     *
     * @param g a wxh array of short
     * @return an Image, possibly transposed.
     */
    public static Image getTransposedImage(short g[][]) {
        if (g == null) return null;
        int width = g.length;
        int height = g[0].length;
        Toolkit tk =
                Toolkit.getDefaultToolkit();
        ColorModel cm = tk.getColorModel();
        int pels[] = new int[width * height];
        for (int x = 0; x < width; x++)
            for (int y = 0; y < height; y++) {
                pels[x + y * width] =
                        0xff000000
                                |
                                (g[x][y] << 16)
                                |
                                (g[x][y] << 8)
                                | g[x][y];
            }
        return tk.createImage(new
                MemoryImageSource(width,
                height,
                cm,
                pels, 0,
                width));
    }

    public static short[][] getGreenFromImage(Image img,
                                              ImageObserver io) {
        if (img == null) return null;
        int width = img.getWidth(io);
        int height = img.getHeight(io);
        int pels[] = new int[width * height];
        ColorModel cm = ColorModel.getRGBdefault();

        PixelGrabber grabber =
                new PixelGrabber(img, 0, 0,
                        width, height, pels, 0, width);

        try {
            grabber.grabPixels();
        } catch (InterruptedException e) {
            In.message(e);
        }
        short g[][] = new short[width][height];
        int i;
        for (int x = 0; x < width; x++)
            for (int y = 0; y < height; y++) {
                i = x + y * width;
                g[x][y] = (short) cm.getGreen(pels[i]);
            }
        return g;
    }

    public static BufferedImage convolve(BufferedImage bi,
                                         float[][] blurMatrix) {
        // create ConvolveOp for blurring BufferedImage
        BufferedImageOp convolveOp =
                getConvolveOp(makeKernel(blurMatrix));
        // apply blurFilter to BufferedImage
        return convolveOp.filter(bi, null);
    }

    /**
     * Convolve an image with a kernel using JAI.
     *
     * @param img
     * @param k
     * @return Image that has been filtered with k.
     */
    public static Image convolution(Image img, Kernel k) {
        final BufferedImage bufferedImage = getBufferedImage(img);
        BufferedImageOp convolveOp = getConvolveOp(k);
        return getImage(convolveOp.filter(bufferedImage, null));
    }

    public static Image convolution(Image img, float k[][]) {
        return getImageBean(convolve(getBufferedImage(img),
                k)).getImage();
    }

    public static void print(Component c, String title) {
        java.awt.Toolkit tk = java.awt.Toolkit.getDefaultToolkit();
        java.awt.PrintJob pj =
                tk.getPrintJob(new Frame(),
                        title,
                        null);
        if (pj == null) return;
        // warning, this is not a Graphics2d
        final Graphics graphics = pj.getGraphics();
        c.printAll(graphics);
        pj.end();
    }

    public static void print(Component c) {
        print(c, null);
    }

    public static Image combineBands(float ar,
                                     float ag,
                                     float ab,
                                     Image img) {
        HppFilter3ImageProcessor h3ip =
                getLinearCombineBandsProcessor(ar, ag, ab);
        return h3ip.process(img);
    }

    public static HppFilter3ImageProcessor
            getLinearCombineBandsProcessor(float ar,
                                           float ag,
                                           float ab) {
        return new HppFilter3ImageProcessor(new GreyHppFilter3(ar, ag, ab));
    }


    public static Image sobel(Image img) {
        return convolution(img,
                Kernels.getSobel());
    }


    public static void clip(short a[][], short min, short max) {
        int w = a.length;
        int h = a[0].length;
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                if (a[x][y] < min) a[x][y] = min;
                if (a[x][y] > max) a[x][y] = max;
            }
    }

    /**
     * pasteBean alters is bigOne argument
     *
     * @param littleOne a smaller image pasted into
     * @param bigOne    a bigger image
     * @param x1        At this location
     */
    public static void pasteBean(ShortImageBean littleOne,
                                 ShortImageBean bigOne,
                                 int x1, int y1) {
        int w = littleOne.getWidth();
        int h = littleOne.getHeight();
        int x2 = x1 + w;
        int y2 = y1 + h;
        int xd = 0;
        int yd = 0;
        short _r[][] = bigOne.getR();
        short _g[][] = bigOne.getG();
        short _b[][] = bigOne.getB();
        for (int x = x1; x < x2; x++)
            for (int y = y1; y < y2; y++) {
                _r[x][y] = littleOne.getR()[xd][yd];
                _g[x][y] = littleOne.getG()[xd][yd];
                _b[x][y] = littleOne.getB()[xd][yd];
                yd++;
            }
    }


    public static ShortImageBean
            cutBean(ShortImageBean sib,
                    int x1, int y1, int w, int h) {
        int xd = 0;
        int yd = 0;
        int x2 = x1 + w;
        int y2 = y1 + h;
        ShortImageBean cutBean = new ShortImageBean(w, h);
        short _r[][] = cutBean.getR();
        short _g[][] = cutBean.getG();
        short _b[][] = cutBean.getB();

        for (int x = x1; x < x2; x++)
            for (int y = y1; y < y2; y++) {
                _r[xd][yd] = sib.getR()[x][y];
                _g[xd][yd] = sib.getG()[x][y];
                _b[xd][yd] = sib.getB()[x][y];
                yd++;
            }
        return cutBean;
    }

    /**
     * Compute the average image between img1 and img2.
     *
     * @return average image
     */
    public static Image average(Image img1, Image img2) {
        ShortImageBean sib1 = new ShortImageBean(img1);
        ShortImageBean sib2 = new ShortImageBean(img2);
        sib1.average(sib2);
        return sib1.getImage();
    }

    /**
     * Computer the average image from the image array, ia.
     *
     * @return average image.
     */
    public static Image average(Image ia[]) {
        ShortImageBean sib1 = new ShortImageBean(ia[0]);
        for (int i = 1; i < ia.length; i++) {
            ShortImageBean sib2 = new ShortImageBean(ia[i]);
            sib1.average(sib2);
        }
        return sib1.getImage();
    }

    public static Image getImageResource(Window w, String iconFileName) {
        return w.getToolkit().getImage(w.getClass().getResource("images/" + iconFileName));
    }

    public static Image getImageResource(Object o, String iconFileName) {
        Toolkit tk = Toolkit.getDefaultToolkit();
        return tk.getImage(o.getClass().getResource("images/" + iconFileName));
    }

    /**
     * Helper function to load an icon image from the JAR file of which
     * this class is a part.
     */
    public static ImageIcon fetchIcon(Window w,
                                      String iconFileName) {
        return new ImageIcon(getImageResource(w, iconFileName));
    }


    public static void saveAsJpeg(BufferedImage buf, File f) {
        try {
            JPEGCodec.createJPEGEncoder(new FileOutputStream(f)).encode(buf);
        } catch (java.io.IOException e) {
            e.printStackTrace();
        }
    }

    public static void saveAsGif(Image img, File f) {
        try {
            WriteGIF.doIt(img, f + "");
        } catch (IOException e) {
            e.printStackTrace();
        } catch (AWTException e) {
            e.printStackTrace();
        }
    }

    public static Image gaussian(int kernelWidth,
                                 double sigma,
                                 Image img) {

        ImageProcessorInterface ipi =
                new GaussianSmoothingProcessor(kernelWidth, sigma);
        return ipi.process(img);
    }

    public static Image negate(Image img) {
        ImageProcessorInterface ipi = InvertFilter.getProcessor();
        return ipi.process(img);
    }

    public static Image threshold(Image img, double b) {
        Threshold3Processor t3p = new Threshold3Processor(b);
        return t3p.process(img);
    }

    public static Image enahe(double alpha, Image img) {
        EnaheFilter e = new EnaheFilter(img, alpha / 10.0);
        return new HppFilterImageProcessor(e).process(img);
    }

    public static Image unahe(Image img) {
        EqualizationFilter cf = new EqualizationFilter(img);
        HppFilterImageProcessor hip = new HppFilterImageProcessor(cf);
        return hip.process(img);
    }

    public static Image smooth(int kernelWidth,
                               double sigma,
                               Image image) {
        GaussianSmoothingProcessor gsp =
                new GaussianSmoothingProcessor(kernelWidth, sigma);
        return gsp.process(image);
    }

    public static Image colorize(double ar,
                                 double ag,
                                 double ab,
                                 Image img) {
        PseudoColorFilter pcf = new PseudoColorFilter(ar, ag, ab);
        HppFilterImageProcessor hip = new HppFilterImageProcessor(pcf);
        return hip.process(img);
    }

    public static Image getContrastBrightnessAdjustedImage(double c,
                                                           double b,
                                                           Image img) {
        ContrastFilter cf = new ContrastFilter(c, b);
        HppFilterImageProcessor hpp = new HppFilterImageProcessor(cf);
        return hpp.process((img));
    }

    // - - Here is the BLogic.....
    public static Image getImage(File f) {
        if (!f.canRead()) {
            f = Futil.getReadFile("f cannot read, where is it?");
        }
        try {
            FileInputStream fis =
                    new FileInputStream(f);
            int type = getType(fis);
            if (type == StreamSniffer.GIF87a)
                return getGifJpgPngImage(f);
            if (type == StreamSniffer.GIF89a)
                return getGifJpgPngImage(f);
            if (type == StreamSniffer.JPEG)
                return getGifJpgPngImage(f);
            if (type == StreamSniffer.JPG)
                return getGifJpgPngImage(f);
            if (type == StreamSniffer.PNG_IMAGE)
                return getGifJpgPngImage(f);
            if (type == StreamSniffer.PPM)
                return getPpmImage(f);
            if (type == StreamSniffer.PPM_RAWBITS)
                return getPpmImage(f);
            if (type == StreamSniffer.ZIP_ARCHIVE)
                return getShortImageZip(f);
            else
                System.out.println("type not recognized:" + type);
            Futil.close(fis);
        } catch (FileNotFoundException e) {
            In.message(e);

        } catch (IOException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static Image getShortImageZip(File f) throws IOException, ClassNotFoundException {
        FileInputStream fis = new FileInputStream(f);
        JarInputStream zis = new JarInputStream(fis);

        zis.getNextEntry();
        ObjectInputStream ois =
                new ObjectInputStream(zis);
        final short[][] r = (short[][]) ois.readObject();
        ShortImageBean sib = new ShortImageBean(r.length, r[0].length);
        sib.setR(r);
        zis.getNextEntry();
        sib.setG((short[][]) ois.readObject());
        zis.getNextEntry();
        sib.setB((short[][]) ois.readObject());
        zis.close();
        fis.close();
        return sib.getImage();
    }

    public static int getType(FileInputStream fis) {
        StreamSniffer ss =
                new StreamSniffer(fis);
        return ss.classifyStream();
    }

    public static void testSniff() {
        FileInputStream fis =
                Futil.getFileInputStream("select a new data file");
        StreamSniffer ss =
                new StreamSniffer(fis);

        System.out.println("fee fi fo fum I smell:" + ss);
        Futil.close(fis);
    }

    public static void saveAsGif(Image img, String fn) {
        try {
            WriteGIF.doIt(img,
                    fn);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Captures the entire screen
     *
     * @return SreenImage
     */
    public static BufferedImage captureWholeScreen() throws AWTException {
        return new Robot().createScreenCapture(getFullScreen());
    }

    /**
     * Capture the screen using the component size and location for the
     * image size and location.
     */
    public static BufferedImage captureScreen(Component c)
            throws AWTException {
        return new Robot().createScreenCapture(new Rectangle(c.getX(), c.getY(),
                c.getWidth(), c.getHeight()));
    }

    public static BufferedImage captureScreen(Rectangle inputRectangle)
            throws AWTException {
        return new Robot().createScreenCapture(inputRectangle);
    }

    public static Rectangle getFullScreen() {
        return new Rectangle(new Point(0, 0), getScreenSize());
    }

    public static double getSNRinDb(Image in, Image out) {
        ShortImageBean sib = new ShortImageBean(in);
        return sib.getSNRinDb(new ShortImageBean(out));
    }

    public static Dimension getScreenSize() {
        Toolkit tk = Toolkit.getDefaultToolkit();
        return tk.getScreenSize();
    }

    /**
     * Fast scaling of a buffered image by using replicated pixels. Used
     * the Image.scale method with a Image.SCALE_FAST for fastest possible
     * result. If either the <code>width</code> or <code>height</code> is a
     * negative number then a value is substituted to maintain the aspect
     * ratio of the original image dimensions.
     */
    public static BufferedImage scaleFast(BufferedImage bi,
                                          int width,
                                          int height) {

        return ImageUtils.getBufferedImage(bi.getScaledInstance(width, height, Image.SCALE_FAST));
    }

    /**
     * Uses the Image.scale method with a Image.SCALE_SMOOTH for a better
     * result than SCALE_FAST. If either the <code>width</code> or
     * <code>height</code> is a negative number then a value is substituted
     * to maintain the aspect ratio of the original image dimensions.
     */
    public static BufferedImage scaleSmooth(BufferedImage bi,
                                            int width,
                                            int height) {
        Image img = getImage(bi);
        img = scaleSmooth(img, width, height);
        return ImageUtils.getBufferedImage(img);
    }

    /**
     * Uses the Image.SCALE_FAST rendering hint to make use of replicated
     * pixels for creating the new image. No averaging here... If either
     * the <code>width</code> or <code>height</code> is a negative number
     * then a value is substituted to maintain the aspect ratio of the
     * original image dimensions.
     */
    public static Image scaleFast(Image img, int width, int height) {
        if (img == null) return null;
        return img.getScaledInstance(width, height, Image.SCALE_FAST);
    }

    /**
     * Uses the Image.SCALE_SMOOTH rendering hint to make use of averaging.
     * If either the <code>width</code> or <code>height</code> is a
     * negative number then a value is substituted to maintain the aspect
     * ratio of the original image dimensions.
     */
    public static Image scaleSmooth(Image img, int width, int height) {
        return img.getScaledInstance(width, height, Image.SCALE_SMOOTH);
    }

    public static String getImageIoFormat() {
        return (String) In.multiPrompt(getImageIOFormatNames(),
                "select an output format", "format selector dialog");
    }

    public static String[] getImageIOFormatNames() {
        final String[] writerFormatNames;
        writerFormatNames = ImageIO.getWriterFormatNames();
        return writerFormatNames;
    }

    /**
     * The following code does not seem to work on a mac. I dont know why.
     */
    public static void saveAsTiff(BufferedImage bi, File f)
            throws IOException {
        OutputStream os = new FileOutputStream(f);
        TIFFEncodeParam tep = new TIFFEncodeParam();
        ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF",
                os,
                tep);
        encoder.encode(bi);
        os.close();
    }

    /**
     * grab a screen capture in png format.
     */
    public static byte[] grabAPngImage(Rectangle rect)
            throws AWTException, IOException {
        // create screen shot
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Robot robot = new Robot();
        BufferedImage image = robot.createScreenCapture(rect);
        // save captured image to PNG file
        ImageIO.write(image, "png", baos);
        baos.close();
        return baos.toByteArray();
    }

    public static BufferedImage getBufferedImage(String s) {
        ResourceManager rm = ResourceManager.getResourceManager();
        return getBufferedImage(getImage(rm.getImageFile(s)));
    }

    public static BufferedImage jpegEncode(BufferedImage bi,
                                           int xDensity,
                                           int yDensity, float quality)
            throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        JPEGImageEncoder encoder =
                JPEGCodec.createJPEGEncoder(out);
        JPEGEncodeParam param =
                encoder.getDefaultJPEGEncodeParam(bi);


        param.setQuality(quality, false);

        param.setDensityUnit(JPEGEncodeParam.DENSITY_UNIT_DOTS_INCH);

        param.setXDensity(xDensity);

        param.setYDensity(yDensity);

        encoder.setJPEGEncodeParam(param);
        encoder.encode(bi);

        ByteArrayInputStream in = new ByteArrayInputStream(out.toByteArray());
        JPEGImageDecoder decoder =
                JPEGCodec.createJPEGDecoder(in);
        return decoder.decodeAsBufferedImage();
    }

    public static BufferedImage captureScreen(Rectangle inputRectangle,
                                              Dimension outputDim) {
        BufferedImage bi =
                null;
        try {
            bi = captureScreen(inputRectangle);
        } catch (AWTException e) {
            e.printStackTrace();

        }
        if (outputDim.equals(inputRectangle.getSize())) {
            return bi; // no need to scale
        }
        bi = scaleSmooth(bi,
                outputDim.width,
                outputDim.height);
        return bi;
    }

    public static Image getImage(byte r[][]) {

        int w = r.length;
        int h = r[0].length;
        int v;
        Toolkit tk = Toolkit.getDefaultToolkit();
        int pels[] = new int[w * h];
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                if (r[x][y] == 1)
                    v = 255;
                else
                    v = 0;
                pels[y + x * h] =
                        0xff000000
                                |
                                (v << 16)
                                |
                                (v << 8)
                                | v;
            }
        Image i;
        i = tk.createImage(new MemoryImageSource(w,
                h,
                ColorModel.getRGBdefault(),
                pels, 0,
                w));
        return i;
    }

    public static RenderedOp addImages(PlanarImage image1,
                                       PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("add", pb);
    }

    public static RenderedOp subtractImages(PlanarImage image1,
                                            PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("subtract", pb);
    }

    public static RenderedOp multiplyImages(PlanarImage image1,
                                            PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("multiply", pb);
    }

    public static RenderedOp divideImages(PlanarImage image1,
                                          PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("divide", pb);
    }

    public static RenderedOp andImages(PlanarImage image1,
                                       PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("and", pb);
    }

    public static RenderedOp orImages(PlanarImage image1,
                                      PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("or", pb);
    }

    public static RenderedOp xorImages(PlanarImage image1,
                                       PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("xor", pb);
    }

    public static RenderedOp notImage(PlanarImage image1) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        return JAI.create("not", pb);
    }

    public static RenderedOp minImages(PlanarImage image1,
                                       PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("min", pb);
    }

    public static RenderedOp maxImages(PlanarImage image1,
                                       PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("max", pb);
    }

    public static RenderedOp overlayImages(PlanarImage image1,
                                           PlanarImage image2) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image1);
        pb.addSource(image2);
        return JAI.create("overlay", pb);
    }

    public static RenderedOp compositeImages(PlanarImage sourceImage1,
                                             PlanarImage sourceImage2,
                                             boolean alphaPremultiplied,
                                             int destAlpha) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(sourceImage1);
        pb.addSource(sourceImage2);
        pb.add(sourceImage1);
        pb.add(sourceImage2);
        pb.add(alphaPremultiplied ? Boolean.TRUE : Boolean.FALSE);
        pb.add(destAlpha);
        return JAI.create("composite", pb);
    }

    public static RenderedOp logImage(PlanarImage image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("log", pb);
    }

    public static RenderedOp expImage(PlanarImage image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("exp", pb);
    }

    public static RenderedOp cropImage(PlanarImage image, Rectangle rect) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add((float) rect.x);
        pb.add((float) rect.y);
        pb.add((float) rect.width);
        pb.add((float) rect.height);
        return JAI.create("crop", pb);
    }

    public static RenderedOp cropImage(BufferedImage image, Rectangle rect) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add((float) rect.x);
        pb.add((float) rect.y);
        pb.add((float) rect.width);
        pb.add((float) rect.height);
        RenderedOp renderedOp = null;
        try {
            renderedOp = JAI.create("crop", pb);
        } catch (Exception e) {
            e.printStackTrace();
            System.out.println("rect=" + rect);
        }
        return renderedOp;
    }

    public static RenderedOp cropImage(PlanarImage image, Rectangle2D rect) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add((float) rect.getX());
        pb.add((float) rect.getY());
        pb.add((float) rect.getWidth());
        pb.add((float) rect.getHeight());
        return JAI.create("crop", pb);
    }

    public static RenderedOp boxFilterImage(PlanarImage image,
                                            int wid, int ht,
                                            Point pixelPos) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(wid);
        pb.add(ht);
        pb.add(pixelPos.x);
        pb.add(pixelPos.y);
        return JAI.create("BoxFilter", pb);
    }

    public static RenderedOp medianFilterImage(PlanarImage image,
                                               int maskShape,
                                               int maskSize) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(maskShape);
        pb.add(maskSize);
        return JAI.create("MedianFilter", pb);
    }

    public static void displayPSD(Image img) {
        displayImage(getMagnitudeImage(DFTImage(img)), "PSD");
    }


    public static RenderedOp DFTImage(Image img) {
        return ImageUtils.DFTImage(getPlanarImage(img),
                DFTDescriptor.SCALING_DIMENSIONS,
                DFTDescriptor.REAL_TO_COMPLEX);
    }

    public static RenderedOp DFTImage(PlanarImage image,
                                      DFTScalingType scalingType,
                                      DFTDataNature dataNature) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(scalingType);
        pb.add(dataNature);
        return JAI.create("dft", pb);
    }


    public static RenderedOp DCTImage(PlanarImage image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("dct", pb);
    }

    public static RenderedOp inverseDCTImage(RenderedOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("idct", pb);
    }

    public static RenderedOp magnitudeImage(RenderedOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("magnitude", pb);
    }

    public static RenderedOp magnitudeSquaredImage(RenderedOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("magnitudeSquared", pb);
    }

    public static RenderedOp phaseImage(RenderedOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.create("phase", pb);
    }

    public static RenderedOp periodicShiftImage(RenderedOp image,
                                                int shiftX,
                                                int shiftY) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(shiftX).add(shiftY);
        return JAI.create("idft", pb);
    }

    public static RenderedOp polarToComplexImage(RenderedOp mag,
                                                 RenderedOp phase) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(mag);
        pb.addSource(phase);
        return JAI.create("polartocomplex", pb);
    }

    public static RenderableOp DFTImage(RenderableOp image,
                                        DFTScalingType scalingType,
                                        DFTScalingType dataNature) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(scalingType).add(dataNature);
        return JAI.createRenderable("dft", pb);
    }


    public static RenderableOp DCTImage(RenderableOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.createRenderable("dct", pb);
    }

    public static RenderableOp InverseDCTImage(RenderableOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.createRenderable("idct", pb);
    }

    public static RenderableOp magnitudeImage(RenderableOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.createRenderable("magnitude", pb);
    }

    public static RenderableOp magnitudeSquaredImage(RenderableOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.createRenderable("magnitudeSquared", pb);
    }

    public static RenderableOp getPhaseImage(RenderableOp image) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        return JAI.createRenderable("phase", pb);
    }

    public static RenderableOp getPeriodicShiftImage(RenderableOp image,
                                                     int shiftX,
                                                     int shiftY) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(shiftX).add(shiftY);
        return JAI.createRenderable("idft", pb);
    }

    public static RenderedOp getSobelHorizontalImage(PlanarImage image) {
        KernelJAI sobelHorizKernel = KernelJAI.GRADIENT_MASK_SOBEL_HORIZONTAL;
        return JAI.create("convolve", image, sobelHorizKernel);
    }

    public static RenderedOp getSobelVertImage(PlanarImage image) {
        KernelJAI sobelVertKernel = KernelJAI.GRADIENT_MASK_SOBEL_VERTICAL;
        return JAI.create("convolve", image, sobelVertKernel);
    }

    public static RenderedOp getSovelGradientMagnitudeImage(PlanarImage image) {
        KernelJAI sobelVertKernel = KernelJAI.GRADIENT_MASK_SOBEL_VERTICAL;
        KernelJAI sobelHorizKernel = KernelJAI.GRADIENT_MASK_SOBEL_HORIZONTAL;
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);
        pb.add(sobelHorizKernel);
        pb.add(sobelVertKernel);
        return JAI.create("gradientmagnitude", pb);
    }

    public static RenderedOp getSharpenImage(PlanarImage image) {
        float sharpMatrix[] = {
                -1.0f, -1.0f, -1.0f,
                -1.0f, 9.0f, -1.0f,
                -1.0f, -1.0f, -1.0f
        };
        KernelJAI sharpKernel = new KernelJAI(3, 3, sharpMatrix);
        return JAI.create("convolve", image, sharpKernel);
    }

    public static RenderedOp getBlurImage(PlanarImage image) {
        float blurmatrix[] = {
                1 / 16f, 1 / 8f, 1 / 16f,
                1 / 8f, 1 / 4f, 1 / 8f,
                1 / 16f, 1 / 8f, 1 / 16f
        };
        KernelJAI blurKernel = new KernelJAI(3, 3, blurmatrix);
        return JAI.create("convolve", image, blurKernel);
    }

    public static RenderedOp getGrayImage(PlanarImage image) {
        return JAI.create("ColorConvert", image, ColorSpace.getInstance(ColorSpace.
                CS_GRAY));
    }

    public static RenderedOp getGrayBandCombine(PlanarImage image) {
        double[][] grayBandCombineMatrix = {
                {0.212671f, 0.715160f, 0.071169f, 0.0f}
        };
        return JAI.create("BandCombine", image, grayBandCombineMatrix);
    }

    public static RenderedOp getBandSelect(PlanarImage image, int bandNum) {
        int numbands = image.getData().getNumBands();
        if (bandNum >= numbands) return null;
        return JAI.create("BandSelect", image, new int[]{bandNum});
    }

    public static int getArgbToInt(short alpha, short red, short green, short blue) {
        return alpha << 24 | red << 16 | green << 8 | blue;
    }

    public static int[] getArgbToInt(short[] alpha, short[] red,
                                     short[] green, short[] blue) {

        int length = alpha.length;
        int[] imageInt = new int[length];

        for (int i = 0; i < length; i++) {
            imageInt[i] = alpha[i] << 24 | red[i] << 16 | green[i] << 8 | blue[i];
        }
        return imageInt;
    }

    public static int[] getArgbToInt(short[] alpha, float[] red,
                                     float[] green, float[] blue) {

        int length = alpha.length;
        int[] imageInt = new int[length];

        for (int i = 0; i < length; i++) {
            imageInt[i] = alpha[i] << 24 | ((int) red[i]) << 16 |
                    ((int) green[i]) << 8 | ((int) blue[i]);
        }
        return imageInt;
    }

    public static int[] getArgbToInt(short[] alpha, double[] red,
                                     double[] green, double[] blue) {

        int length = alpha.length;
        int[] imageInt = new int[length];

        for (int i = 0; i < length; i++) {
            imageInt[i] = alpha[i] << 24 | ((int) red[i]) << 16 |
                    ((int) green[i]) << 8 | ((int) blue[i]);
        }
        return imageInt;
    }

    public static short getAlpha(int argb) {
        return (short) ((argb >> 24) & 0xFF);
    }

    public static short[] getRed(int[] argb) {
        int length = argb.length;
        short red[] = new short[length];

        for (int i = 0; i < length; i++)
            red[i] = (short) ((argb[i] & 0x00FF0000) >> 16);
        return red;
    }

    public static short[] getGreen(int[] argb) {
        int length = argb.length;
        short green[] = new short[length];

        for (int i = 0; i < length; i++)
            green[i] = (short) ((argb[i] & 0x0000FF00) >> 8);
        return green;
    }

    public static short[] getBlue(int[] argb) {
        int length = argb.length;
        short blue[] = new short[length];

        for (int i = 0; i < length; i++)
            blue[i] = (short) (argb[i] & 0x000000FF);
        return blue;
    }

    public static short[] getAlpha(int[] argb) {
        int length = argb.length;
        short alpha[] = new short[length];

        for (int i = 0; i < length; i++)
            alpha[i] = (short) ((argb[i] & 0xFF000000) >> 24);
        return alpha;
    }

    public static PlanarImage getMagnitudeImage(PlanarImage planarImage) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(planarImage);
        return JAI.create("magnitude", pb);
    }

    public PlanarImage getCroppedImage(PlanarImage image, Rectangle rect) {
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(image);

        pb.add((float) rect.x);
        pb.add((float) rect.y);
        pb.add((float) rect.width);
        pb.add((float) rect.height);
        // Create the output image by cropping the input image.

        PlanarImage output = JAI.create("crop", pb, null);
//      return output; //siva
        pb = new ParameterBlock();
        pb.addSource(output);
        pb.add((float) -rect.x);
        pb.add((float) -rect.y);

        //return the output image by translating itself.
        return JAI.create("translate", pb, null);
    }

    public PlanarImage getConvolvedImage(PlanarImage image, float[] kernelData, int w, int h) {
        KernelJAI kernel = new KernelJAI(w, h, kernelData);

        return JAI.create("convolve", image, kernel);

    }


    public static void saveAsPPM(Image img, String fn) {
        ShortImageBean sib = new ShortImageBean(img);
        WritePPM.doIt(sib.getR(),
                sib.getG(),
                sib.getB(),
                fn);
    }

    public static void displayImageAwt(PlanarImage image, String title) {
        String imageInfo = "Dimensions: " + image.getWidth() + "x" + image.getHeight() +
                " Bands:" + image.getNumBands();
        // Create a frame for display.
        Frame f = new Frame();
        f.setTitle(title);
        // Get the JFrame's ContentPane.

        f.setLayout(new BorderLayout());
        // Create an instance of DisplayJAI.
        DisplayJAI dj = new DisplayJAI(image);
        // Add to the JFrame's ContentPane an instance of JScrollPane containing the
        // DisplayJAI instance.
        f.add(new JScrollPane(dj), BorderLayout.CENTER);
        // Add a text label with the image information.
        f.add(new JLabel(imageInfo), BorderLayout.SOUTH);
        // Set the closing operation so the application is finished.
        f.setSize(400, 400); // adjust the frame size.
        f.setVisible(true); // show the frame.
    }

    public static void displayImage(PlanarImage image, String title) {
        String imageInfo = "Dimensions: " + image.getWidth() + "x" + image.getHeight() +
                " Bands:" + image.getNumBands();
        // Create a frame for display.
        JFrame frame = new JFrame();
        frame.setTitle(title);
        // Get the JFrame's ContentPane.
        Container contentPane = frame.getContentPane();
        contentPane.setLayout(new BorderLayout());
        // Create an instance of DisplayJAI.
        DisplayJAI dj = new DisplayJAI(image);
        // Add to the JFrame's ContentPane an instance of JScrollPane containing the
        // DisplayJAI instance.
        contentPane.add(new JScrollPane(dj), BorderLayout.CENTER);
        // Add a text label with the image information.
        contentPane.add(new JLabel(imageInfo), BorderLayout.SOUTH);
        // Set the closing operation so the application is finished.
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setSize(400, 400); // adjust the frame size.
        frame.setVisible(true); // show the frame.

    }

    public static Image getGrayImage(Image image) {
        return getImage(
                image, new GreyFilter());
    }

    public static int[] imagetoInt(short[] imageR, short[] imageG,
                                   short[] imageB, int numPixels) {
        int[] imageData = new int[numPixels];

        for (int i = 0; i < numPixels; i++) {
            imageData[i] = ((0xFF) << 24) |
                    ((imageR[i] & 0xFF) << 16) |
                    ((imageG[i] & 0xFF) << 8) |
                    ((imageB[i] & 0xFF));
        }
        return imageData;
    }

    public static int[] imagetoInt(float[] imageR, float[] imageG,
                                   float[] imageB) {
        // Assume the Green and Blue arrays are the same size as Red.
        int numPixels = imageR.length;
        int[] imageData = new int[numPixels];

        for (int i = 0; i < numPixels; i++) {
            imageData[i] = ((0xFF) << 24) |
                    ((((int) imageR[i]) & 0xFF) << 16) |
                    ((((int) imageG[i]) & 0xFF) << 8) |
                    ((((int) imageB[i]) & 0xFF));
        }
        return imageData;
    }

    public static int[] imagetoInt(double[] imageR, double[] imageG,
                                   double[] imageB, int numPixels) {
        int[] imageData = new int[numPixels];

        for (int i = 0; i < numPixels; i++) {
            imageData[i] = ((0xFF) << 24) |
                    ((((int) imageR[i]) & 0xFF) << 16) |
                    ((((int) imageG[i]) & 0xFF) << 8) |
                    ((((int) imageB[i]) & 0xFF));
        }
        return imageData;
    }

    public static short getA(int valARGB) {
        return ((short) ((valARGB >> 24) & 0xFF));
    }

    public static int putRGB(short r, short g, short b) {
        int pixRGB;

        pixRGB = (0xFF << 24) | (((int) r & 0xFF) << 16) | (((int) g & 0xFF) << 8) |
                ((int) b & 0xFF);

        return (pixRGB);
    }

    // Maximum of R,G and B
    public static short max(short r, short g, short b) {
        if (r > g) {
            if (r > b) {
                return (r);
            } else {
                return (b);
            }
        } else {
            if (g > b) {
                return (g);
            } else {
                return (b);
            }
        }
    }

    // Minimum of R,G and B
    public static short min(short r, short g, short b) {
        if (r < g) {
            if (r < b) {
                return (r);
            } else {
                return (b);
            }
        } else {
            if (g < b) {
                return (g);
            } else {
                return (b);
            }
        }
    }

    public static short[] getRedArray(int[] inputArray) {

        int totalPix = inputArray.length;
        short r[] = new short[totalPix];

        for (int i = 0; i < totalPix; i++) {
            r[i] = (short) ((inputArray[i] & 0x00FF0000) >> 16);
        }
        return r;
    }

    public static short[] getGreenArray(int[] inputArray) {

        int totalPix = inputArray.length;
        short g[] = new short[totalPix];

        for (int i = 0; i < totalPix; i++) {
            g[i] = (short) ((inputArray[i] & 0x0000FF00) >> 8);
        }
        return g;
    }

    public static short[] getBlueArray(int[] inputArray) {

        int totalPix = inputArray.length;
        short b[] = new short[totalPix];

        for (int i = 0; i < totalPix; i++) {
            b[i] = (short) (inputArray[i] & 0x000000FF);
        }
        return b;
    }

    /**
     * Convolve an image using the ImageMatrixBean and the
     * jai implementation of convolution. Grow the image according
     * to the kernel growing policy and use the convolution kernel in
     * the IMB to do the convolution. Then crop the output image to remove
     * the JAI artifacts, as the edges.
     *
     * @param image
     * @param imb
     * @return image that has been convolved with the kernel in the imb.
     */
    public static Image convolveJai(Image image, ImageMatrixBean imb) {
        final int n = imb.getMatrixSize() / 2 + 1;
        final PlanarImage pi =
                BorderUtils.getBorderImage(
                        getPlanarImage(image),
                        n, n, n, n, (int) imb.getBorderType());
        BufferedImageOp convolveOp = getConvolveOp(imb.getKernel());
        Rectangle r = new Rectangle(n, n,
                image.getWidth(null), image.getHeight(null));
        return getImage(
                cropImage(
                        convolveOp.filter(pi.getAsBufferedImage(), null), r));
    }

    /**
     * Convolve an image using the kernel in the ImageMatrixBean.
     * Use a Radix2 algorithm. Truncate the image to an integral power of two.
     * Grow the kernal, according to the kernal growing policy, to an image
     * of the newly truncated image size. Perform the FFTR2 on both images.
     * Multiply them together in the complex number plane. Perform the IFFT.
     * Get the real-number plane off the IFFT image. Return it.
     *
     * @param image a real-valued input image to transform
     * @return a filtered output image.
     */
    public static Image convolveR2(Image image) {
        image = cropImageToIntegralPowerOfTwo(image);
        //final int n = imb.getMatrixSize() / 2 + 1;


        return cropImageToIntegralPowerOfTwo(image);
    }

    public static Image cropImageToIntegralPowerOfTwo(Image image) {
        int w = image.getWidth(null);
        int h = image.getHeight(null);
        w = MathUtils.roundToIntegralPowerOf2(w);
        h = MathUtils.roundToIntegralPowerOf2(h);
        Rectangle r = new Rectangle(0, 0, w, h);
        return cropImage(image, r);
    }

    /**
     * Crop an image to the size of the rectangle, r.
     * The input image must be larger than the rectangle.
     *
     * @param image input larger than r.
     * @param r
     * @return cropped image.
     */
    public static Image cropImage(Image image, Rectangle r) {
        return getImage(cropImage(getBufferedImage(image), r));
    }

    public static TiledImage viewDFT(PlanarImage inputImg) {
        /** ToneScale the image */
        DataBuffer db = inputImg.getData().getDataBuffer();
        db.setElemFloat(0, (float) 0);
        RenderedImage ext = JAI.create("extrema", inputImg);
        double[][] extrema = (double[][]) ext.getProperty("extrema");
        double max = extrema[1][0];
        double min = extrema[0][0];
        db.setElemFloat(0, (float) max);
        RenderedImage rescaleddisplay = JAI.create("rescale", inputImg,
                new double[]{(0xFF / (max - min))},
                new double[]{((0xFF * min) / (min - max))});
        RenderedImage display = JAI.create("format", rescaleddisplay);
        /** Format the image so that the DC-value is located in the center */
        int width = display.getWidth();
        int height = display.getHeight();
        int rangeX = width / 2;      //X-indexrange
        int rangeY = height / 2;     //Y-indexrange
        int DCoriginX = width / 2;     //X-coordinate of DC-component
        int DCoriginY = height / 2;    //Y-coordinate of DC-component
        float displacementXneg = (float) -rangeX;
        // X-displacement to move negative freq.block
        float displacementYneg = (float) -rangeY;
        // Y-displacement to move negative freq.block
        float displacementXpos = (float) rangeX;
        // X-displacement to move positive freq.block
        float displacementYpos = (float) rangeY;
        // Y-displacement to move positive freq.block
// create empty image where we will build the manipulated image
        TiledImage outImg = new
                TiledImage(0, 0, width + 1, height + 1, 0, 0, display.getSampleModel(), display.getColorModel());
// map quadrant 2 to 4 : x,y (0..256) to (256..512)
        ROIShape roi4 = new ROIShape(new
                Rectangle(DCoriginX, DCoriginY, rangeX + 1, rangeY + 1));//quadrant 4
        ParameterBlock pb = new ParameterBlock();
        pb.addSource(display);
        pb.add(displacementXpos);
        pb.add(displacementYpos);
        outImg.setData(JAI.create("translate", pb).getTile(0, 0), roi4);
// map quadrant 4 to 2 : x,y (256..511) to (0..255)
        ROIShape roi2 = new ROIShape(new Rectangle(0, 0, rangeX, rangeY));//quadrant 2
        pb.set(displacementXneg, 0);
        pb.set(displacementYneg, 1);
        outImg.setData(JAI.create("translate", pb).getTile(0, 0), roi2);
// map quadrant 1 to 3 : x(256..511) to (0..255) and y(0..256) to (256..512)
        ROIShape roi3 = new ROIShape(new
                Rectangle(0, DCoriginY, rangeX, rangeY + 1));//quadrant 3
        pb.set(displacementXneg, 0);
        pb.set(displacementYpos, 1);
        outImg.setData(JAI.create("translate", pb).getTile(0, 0), roi3);
// map quadrant 3 to 1 : x(0..256) to (256..512) and y(256..511) to (0..255)
        ROIShape roi1 = new ROIShape(new
                Rectangle(DCoriginX, 0, rangeX + 1, rangeY));//quadrant 1
        pb.set(displacementXpos, 0);
        pb.set(displacementYneg, 1);
        outImg.setData(JAI.create("translate", pb).getTile(0, 0), roi1);
        return (outImg);
    } //viewDFT

    public static void quadSwapTest() {
        Image img = getImage();
        displayImage(img, "original");
        Image qs = quadSwap(img);
        displayImage(qs, "qs");
        qs = quadSwap(qs);
        displayImage(qs, "qs,qs");
    }

    /**
     * Centers the image by translating the image to the center.
     * This can be done via a phase shift in the frequency domain,
     * but I think this is pretty fast (using JAI, etc.).
     * <br>  <code>
     * q1 q2
     * q3 q4
     * becomes:
     * q4 q3
     * q2 q1
     *   Does the quad swap
     * </code>
     * @param img
     * @return centered image.
     */
    public static Image quadSwap(Image img) {
        ShortImageBean sib = new ShortImageBean(img);
        sib.setR(Mat2.quadSwap(sib.getR()));
        sib.setG(Mat2.quadSwap(sib.getG()));
        sib.setB(Mat2.quadSwap(sib.getB()));
        return sib.getImage();
    }


}
