package graphics.grapher;
import java.awt.*;
import java.util.Enumeration;
import java.util.Vector;



/*************************************************************************
**
**    Class  DataSet
**                                              Version 1.0   October 1995
**                                              Version 1.1   December 1995
**
**************************************************************************
**    Copyright (C) 1995, 1996 Leigh Brookshaw
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**************************************************************************
**
**    This class is designed to be used in conjunction with 
**    the Graph2D class and Axis class for plotting 2D graphs.
**
*************************************************************************/


/*
**  Class DataSet
**               This object is designed to hold the data to be plotted
**               The data is stored in an array with the x,y data
**               stored adjacently. The x data in even indices, y data in odd.
**               Along with the data are a number of flags associated
**               with the data defining how this data is to be plotted.
*/


public class DataSet  {


/**************************
** Public Static Values     
**************************/
/*
**    The static value specifying that no straight line segment
**    is to join the points.
*/
      public final static int NOLINE    =  0;
/*
**    The static value specifying that a straight line segment
**    is to join the points.
*/
      public final static int LINE      =  1;



/**********************
** Public Variables      
**********************/

/* 
** g2d
**    The Graphics canvas that is driving the whole show
**    See class Graph2D
*/
      public GraphComponent g2d;

/*
** linestyle
**    The linestyle to employ when joining the data points with
**    straight line segments. Currently only solid and no line
**    are supported. Dashed lines will be added
*/
      private int   linestyle     = LINE;
/*
** linecolor
**    The color of the straight line segments
*/
      public Color linecolor     = null;
/*
** marker
**    The index of the marker to use at the data points
**    markers are stroked.
*/
      public int    marker       = 0;
/*
** markercolor
**    The marker color
*/
      public Color  markercolor  = null;
/*
** markerscale
**    The scaling factor for the marker
*/
      public double markerscale  = 1.0;
/*
** xaxis
**    The Axis object the X data is attached to. From the Axis object
**    the scaling for the data can be calculated.
*/
      public Axis xaxis;
/*
** yaxis
**    The Axis object the Y data is attached to.
*/
      public Axis yaxis;
/*
** xmax, xmin, ymax, ymin
**    The plottable range of the data. This can be very different from
**    true data range. The data is clipped when plotted.
*/
      public double xmax, xmin, ymax, ymin;

/*
** legend_length
**    The length of the line in the legend
*/
      int legend_length = 20;

/*
** legend_text
**    The legend text
*/
      TextLine legend_text = null;
/*
** legend_ix, legend_iy
**    
*/
      int legend_ix, legend_iy;
      double legend_dx, legend_dy;
/*
** increment
**    The amount to increment the data array when the append method is being
**    used.
*/
      int increment = 100;

/**********************
** Protected Variables      
**********************/
/*
** dxmax, dxmin, dymax, dymin
**    The true data range. Once the data is loaded this will never change.
*/
      protected double dxmax, dxmin, dymax, dymin;

/*
** data[]
**    The actual data
*/
      protected double data[];
/*
** length
**    The actual number of data points stored in 'data'
*/
      protected int length;
/*
** xrange, yrange
**    The range of the clipped data
*/
      protected double xrange, yrange;


/********************
** Constructors
********************/
/*
** DataSet()
**   Create an empty data set.
*/
      public DataSet ( ) {
               length = 0;
               range();
      }

/*
** DataSet(double[], int)
**   The double array contains the data. The X data is expected in
**   the even indices, the y data in the odd. The integer n is the
**   number of data Points. This manes that the length of the data
**   array is 2*n.
*/
      public DataSet ( double d[], int n ) throws Exception {
           int i;
           int k = 0;

           length = 0;
           if ( d == null) {
           	System.out.println("DataSet: d == null");
           }
           if ( n <= 0) {
           	System.out.println("DataSet: n <= 0");
           }
           if (  d.length == 0) {
           	System.out.println("DataSet: d.length == 0");
           }
           if (  d.length < n) {
           	System.out.println("DataSet: (  d.length < n)");
           }
	
           if ( d  == null || d.length == 0 || n <= 0 ) {
              throw new Exception("DataSet: Error in parsed data!");
           }

//     Copy the data locally.

           data = new double[n*2];
           length = n*2;

           System.arraycopy(d, 0, data, 0, length);


//     Calculate the data range.

           range();


      }



/******************
** Public Methods
******************/

/*
** append(double[], int)
**    Append data to the data set.
*/
      public void append( double d[], int n ) throws Exception {
           int i;
           int k = 0;
           double tmp[];

           if ( d  == null || d.length == 0 || n <= 0 ) {
              throw new Exception("DataSet: Error in append data!");
           }

           if(data == null) data = new double[increment];

//     Copy the data locally.

           if( n*2+length < data.length ) {
               System.arraycopy(d, 0, data, length, n*2);
               length += n*2;
	   } else {
               tmp = new double[n*2+length+increment];

               if( length != 0 ) {
                 System.arraycopy(data, 0, tmp, 0, length);
               }
               System.arraycopy(d, 0, tmp, length, n*2);

               length += n*2;
               data = tmp;
	     }

//     Calculate the data range.

           range();
//     Update the range on Axis that this data is attached to
           if(xaxis != null) xaxis.resetRange();
           if(yaxis != null) yaxis.resetRange();



      }
/*
** delete(int start, int end)
**    delete data from the data set (start and end are inclusive).
*/
      public void  delete( int start, int end ) {
           int End   = 2*end;
           int Start = 2*start;

           if(length <= 0) return;

           if( End   < Start )         return;
           if( Start < 0 )             Start = 0;
           if( End > length-2 )        End = length-2;

           if( End < length-2) {
               System.arraycopy(data, End+2, 
                                data, Start, length - End - 2);
	     }

           length -= End+2-Start;


//     Calculate the data range.

           range();


      }

/*
** draw_data(Graphics)
**    Actually draw the straight line segments and/or the markers at the
**    data points.
**
**    If this data has been attached to an Axis then scale the data
**    based on the axis maximum/minimum otherwise scale using
**    the data's maximum/minimum
**
**    Use the clipRect of the graphics object to define the window we
**    have to draw into.
*/

      public void draw_data(Graphics g, Rectangle bounds) {
           Color c;

           if ( xaxis != null ) {
                xmax = xaxis.maximum;
                xmin = xaxis.minimum;
           }

           if ( yaxis != null ) {
                ymax = yaxis.maximum;
                ymin = yaxis.minimum;
           }

           
           xrange = xmax - xmin;
           yrange = ymax - ymin;

           c = g.getColor();

           if( linestyle != DataSet.NOLINE ) {
               if ( linecolor != null) g.setColor(linecolor);
               else                    g.setColor(c);
               draw_lines(g,bounds);
           }    


           if( marker > 0 ) {
               if(markercolor != null) g.setColor(markercolor);
               else                    g.setColor(c);
               draw_markers(g,bounds);
           }    





           g.setColor(c);
      }


      public double getXmax() {  return dxmax; } 
      public double getXmin() {  return dxmin; } 
      public double getYmax() {  return dymax; } 
      public double getYmin() {  return dymin; }



      

      public void legend(int x, int y, String text) {
           if(text == null)  {legend_text = null;  return; }
           if(legend_text == null) legend_text = new TextLine(text);
           else                    legend_text.setText(text);
           legend_text.setJustification(TextLine.LEFT);
           legend_ix    = x;
           legend_iy    = y;
           legend_dx    = 0.0;
           legend_dy    = 0.0;

      }

      public void legend(double x, double y, String text) {
           if(text == null) { legend_text = null;  return; }
           if(legend_text == null) legend_text = new TextLine(text);
           else                    legend_text.setText(text);
           legend_text.setJustification(TextLine.LEFT);
           legend_dx    = x;
           legend_dy    = y;
           legend_ix    = 0;
           legend_iy    = 0;
      }
      public void legendFont(Font f) {
           if(f == null) return;
           if(legend_text == null) legend_text = new TextLine();

           legend_text.setFont(f);
      }

      public void legendColor(Color c) {
           if(c == null) return;
           if(legend_text == null) legend_text = new TextLine();

           legend_text.setColor(c);
      }

      public int dataPoints() {  return length/2; }

      public double[] getPoint(int index) {
            double point[] = new double[2];
            int i = index*2;
            if( index < 0 || i > length-2 ) return null;

            point[0] = data[i];
            point[1] = data[i+1];
            
            return point;
	  }

      public double[] getClosestPoint(double x, double y) {
            double point[] = {0.0, 0.0, 0.0};
            int i;
            double xdiff, ydiff, dist2;

            xdiff = data[0] - x;
            ydiff = data[1] - y;
              
            point[0] = data[0];
            point[1] = data[1];
            point[2] = xdiff*xdiff + ydiff*ydiff;
            
 

            for(i=1; i<length-1; ) {

             i++;
                xdiff = data[i] - x;
             i++;
                ydiff = data[i] - y;


                dist2 = xdiff*xdiff + ydiff*ydiff;

                if(dist2 < point[2]) {
                    point[0] = data[i-1];
                    point[1] = data[i];
                    point[2] = dist2;
		  }

           }

           //System.out.println("DataSet: closestpoint "+point[0]+", "+point[1]+", "+point[2]);

           return point;
            
	  }


/******************
** Protected Methods
******************/

/*
** draw_lines(Graphics)
**
** Actually draw into the clipRect the straight line segments
** At some future date this method will be extended to draw dashed lines.
**
** To save time don't bother to try and draw segments that will be 
** completely clipped. Only Draw segments that are not clipped or partially
** clipped. 
*/

      protected void draw_lines(Graphics g, Rectangle w) {
          int i;
          int j;
          boolean inside0 = false;
          boolean inside1 = false;
          double x,y;
          int x0 = 0 , y0 = 0;
          int x1 = 0 , y1 = 0;
//     Calculate the clipping rectangle
          Rectangle clip = g.getClipBounds();
          int xcmin = clip.x;
          int xcmax = clip.x + clip.width;
          int ycmin = clip.y;
          int ycmax = clip.y + clip.height;


//    Is there any data to draw? Sometimes the draw command will
//    will be called before any data has been placed in the class.
          if( data == null || data.length < 2 ) return;
          

//          System.out.println("Drawing Data Lines!");


//    Is the first point inside the drawing region ?
          if( (inside0 = inside(data[0], data[1])) ) {

              x0 = (int)(w.x + ((data[0]-xmin)/xrange)*w.width);
              y0 = (int)(w.y + (1.0 - (data[1]-ymin)/yrange)*w.height);

              if( x0 < xcmin || x0 > xcmax || 
                  y0 < ycmin || y0 > ycmax)  inside0 = false;

          }


          for(i=2; i<length; i+=2) {

//        Is this point inside the drawing region?

              inside1 = inside( data[i], data[i+1]);
             
//        If one point is inside the drawing region calculate the second point
              if ( inside1 || inside0 ) {

               x1 = (int)(w.x + ((data[i]-xmin)/xrange)*w.width);
               y1 = (int)(w.y + (1.0 - (data[i+1]-ymin)/yrange)*w.height);

               if( x1 < xcmin || x1 > xcmax || 
                   y1 < ycmin || y1 > ycmax)  inside1 = false;

              }
//        If the second point is inside calculate the first point if it
//        was outside
              if ( !inside0 && inside1 ) {

                x0 = (int)(w.x + ((data[i-2]-xmin)/xrange)*w.width);
                y0 = (int)(w.y + (1.0 - (data[i-1]-ymin)/yrange)*w.height);

              }
//        If either point is inside draw the segment
              if ( inside0 || inside1 )  {
                      g.drawLine(x0,y0,x1,y1);
              }

/*
**        The reason for the convolution above is to avoid calculating
**        the points over and over. Now just copy the second point to the
**        first and grab the next point
*/
              inside0 = inside1;
              x0 = x1;
              y0 = y1;

          }

      }


/*
** boolean inside(double, double)
**         
**    return true if the point(x,y) is inside the alowed range.
*/

      protected boolean inside(double x, double y) {
          if( x >= xmin && x <= xmax && 
              y >= ymin && y <= ymax )  return true;
          
          return false;
      }

/*
**  draw_markers(Graphics)
**
**    Draw the markers. The markers have been preloaded from a file.
**    (see the method Graph2D.loadMarker(URL)).
**    Only markers inside the specified range will be drawn. Also markers
**    close the edge of the clipping region will be clipped.
**
**    The markers are stroked based on the preloaded marker definitions.
*/
      protected void draw_markers(Graphics g, Rectangle w) {
          int x1,y1;
          int i;
//     Calculate the clipping rectangle
          Rectangle clip = g.getClipBounds();
          int xcmin = clip.x;
          int xcmax = clip.x + clip.width;
          int ycmin = clip.y;
          int ycmax = clip.y + clip.height;
/*
**        Load the marker specified for this data
*/
          Vector m = g2d.getMarkerVector(marker);


          if( m == null) return;

//          System.out.println("Drawing Data Markers!");

          for(i=0; i<length; i+=2) {
              if( inside( data[i], data[i+1]) ) {

                x1 = (int)(w.x + ((data[i]-xmin)/xrange)*w.width);
                y1 = (int)(w.y + (1.0 - (data[i+1]-ymin)/yrange)*w.height);

                if( x1 >= xcmin && x1 <= xcmax && 
                   y1 >= ycmin && y1 <= ycmax ) stroke_marker(m, g, x1, y1);

                }
          }


      }
/*
** stroke_marker(Vector, Graphics, int, int)
**
**         Stroke the marker centered at the point x,y.
**   The vector contains the definition of the marker. Marker definitions
**   are centered on (0,0) and specify move and draw commands for each
**   point.
*/


      protected void stroke_marker(Vector m, Graphics g, int x, int y) {
          int x0 = x, x1 = x, y0 = y, y1 = y;
          MVertex v;

          for (Enumeration e = m.elements() ; e.hasMoreElements() ;) {
             v = (MVertex)e.nextElement();

             if( v.draw ) {
                 x1 = x + (int)(v.x*markerscale);
                 y1 = y + (int)(v.y*markerscale);

                 g.drawLine(x0,y0,x1,y1);

                 x0 = x1;
                 y0 = y1;
             } else {
                 x0 = x + (int)(v.x*markerscale);
                 y0 = y + (int)(v.y*markerscale);

             }



          }

     }

/*
** draw_legend( Graphics g )
**             Draw a legend for this data set using the data colors and the
**             supplied text
*/

      protected void draw_legend(Graphics g, Rectangle w) {
          Color c = g.getColor();
          Vector m = null;


          if( legend_text == null) return;
          if( legend_text.isNull() ) return;

          if( legend_ix == 0 && legend_iy == 0 ) {
                legend_ix = (int)(w.x + ((legend_dx-xmin)/xrange)*w.width);
                legend_iy = (int)(w.y + (1.0 - (legend_dy-ymin)/yrange)*w.height);
	  }




          if( linestyle != DataSet.NOLINE ) {
              if ( linecolor != null) g.setColor(linecolor);
              g.drawLine(legend_ix,legend_iy,legend_ix+legend_length,legend_iy);
          }

          if( marker > 0 ) {
               m = g2d.getMarkerVector(marker);
               if( m != null) {
                  if(markercolor != null) g.setColor(markercolor);
                  else                    g.setColor(c);

                  stroke_marker(m, g, legend_ix+legend_length/2, legend_iy);
               }
          }


          legend_text.draw( g,
                        legend_ix+legend_length+legend_text.charWidth(g,' '),
                        legend_iy+legend_text.getAscent(g)/3);

          g.setColor(c);

      }

/*
**   Calculate the range of the data
*/

      protected void range() {
           int i;


           if( length > 0 ) {
              dxmax = data[0];
              dymax = data[1];
              dxmin = dxmax;
              dymin = dymax;
           } else {
               dxmin = 0.0;
               dxmax = 0.0;
               dymin = 0.0;
               dymax = 0.0;
           }

           for(i=1; i<length-1; ) {

             i++;
             if( dxmax < data[i] )  dxmax = data[i]; 
             else
             	if( dxmin > data[i] )  dxmin = data[i]; 

             i++;
             if( dymax < data[i] )  dymax = data[i]; 
             else
             	if( dymin > data[i] ) dymin = data[i]; 
           }

           xmin = dxmin;
           xmax = dxmax;
           ymin = dymin;
           ymax = dymax;
     }

    public int getLinestyle() {
        return linestyle;
    }

    public void setLinestyle(int linestyle) {
        this.linestyle = linestyle;
    }
}


