package classUtils.resolver.xml;

import classUtils.resolver.interfaces.Auditor;
import classUtils.resolver.interfaces.WalkObserver;
import classUtils.resolver.model.AuditMaster;
import classUtils.resolver.model.AuditableJarFile;
import classUtils.resolver.util.WalkObserverImpl;
import futils.Futil;

import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;


/**
 * @author Douglas Tillman, VTE Consulting, 12/2002
 * @version 2.0 Outputs results of Jar file inventory and audit as an XML
 *          document. The results are retained as a master list and a
 *          duplicate list - both stored in TreeMap structures.  I opted to
 *          keep them separate rather than having a single data structure
 *          with an isDuplicate boolean flag on the JarFile object because
 *          as the audits get larger, it seemed more efficient if one
 *          wanted to process just the duplicates, to not have to iterate
 *          over the entire inventory. This may change in a later version
 *          depending on the feedback to the utility.
 */
public class XMLBuilder {

    private static String docStartTag = "<JarInventory>";
    private static String docEndTag = "</JarInventory>";
    private static String jarFileStartTag = "<JarFile>";
    private static String jarNameStartTag = "<Name>";
    private static String jarNameEndTag = "</Name>";
    private static String jarPathStartTag = "<Path>";
    private static String jarPathEndTag = "</Path>";
    private static String jarFileEndTag = "</JarFile>";
    private static String classFileStartTag = "<Class>";
    private static String classFileEndTag = "</Class>";


    private File root;
    private StringBuffer xmlDocSB;
    private String xmlString;

    public XMLBuilder(String rootDirPath) {

        root = new File(rootDirPath);
        xmlDocSB = new StringBuffer();
        buildXML();

    }

    /**
     * Creates the Tree and Duplicates Tree nodes
     */
    public void buildXML() {

        //objects for walking directory and callback functionality as .jar
        //files are encountered.
        Auditor auditMaster = new AuditMaster();
        WalkObserver woImpl = new WalkObserverImpl(auditMaster);

        walk(root, woImpl, 0);

        xmlDocSB.append("<?xml version=\"1.0\"?>");
        xmlDocSB.append(docStartTag);
        xmlDocSB.append("<MasterList>");  //starts Master list section

        //loop through collection of values from TreeMap of found .jar's
        Collection jarnames = auditMaster.getMasterJarList().values();

        Iterator iter = jarnames.iterator();
        while (iter.hasNext()) {
            AuditableJarFile jf = (AuditableJarFile) iter.next();

            xmlDocSB.append(jarFileStartTag);
            xmlDocSB.append(jarNameStartTag);
            xmlDocSB.append(jf.getFile().getName());
            xmlDocSB.append(jarNameEndTag);
            xmlDocSB.append(jarPathStartTag);
            xmlDocSB.append(jf.getFile().getAbsolutePath());
            xmlDocSB.append(jarPathEndTag);

            ArrayList a = jf.getExplodedContents();
            for (int x = 0; x < a.size(); x++) {
                xmlDocSB.append(classFileStartTag);
                xmlDocSB.append((String) a.get(x));
                xmlDocSB.append(classFileEndTag);
            }

            xmlDocSB.append(jarFileEndTag);
        }

        xmlDocSB.append("</MasterList>");

        xmlDocSB.append("<Duplicates>");

        //loop through duplicates
        //loop through collection of values from TreeMap of duplicates
        Collection dupjarnames = auditMaster.getDuplicateJarList().values();
        Iterator dupIter = dupjarnames.iterator();
        while (dupIter.hasNext()) {
            AuditableJarFile dupjf = (AuditableJarFile) dupIter.next();

            xmlDocSB.append(jarFileStartTag);
            xmlDocSB.append(jarNameStartTag);
            xmlDocSB.append(dupjf.getFile().getName());
            xmlDocSB.append(jarNameEndTag);
            xmlDocSB.append(jarPathStartTag);
            xmlDocSB.append(dupjf.getFile().getAbsolutePath());
            xmlDocSB.append(jarPathEndTag);

            ArrayList a = dupjf.getExplodedContents();
            for (int x = 0; x < a.size(); x++) {
                xmlDocSB.append(classFileStartTag);
                xmlDocSB.append((String) a.get(x));
                xmlDocSB.append(classFileEndTag);
            }

            xmlDocSB.append(jarFileEndTag);
        }

        xmlDocSB.append("</Duplicates>");
        xmlDocSB.append(docEndTag);

        //System.out.println(xmlDocSB.toString());
        xmlString = xmlDocSB.toString();
    }

    /**
     * Walks a directory tree, calling the WalkObserver at each file.
     *
     * @param rootDirectory Root of directory tree to walk
     * @param o             The directory WalkObserver object for
     *                      callbacks
     * @param indentLevel   The current level of recursion Recursive
     *                      function borrowed from: http://www.microsoft.com/mind/0498/java9110498.asp
     *                      by Jonathan Locke in a Microsoft online article
     *                      in Java911, 1998
     */
    private void walk(File rootDirectory,
                      WalkObserver o,
                      int indentLevel) {

        // Get list of files in directory
        String[] file = rootDirectory.list();

        // Traverse files
        for (int i = 0; i < file.length; i++) {
            // Call observer callback with file
            File f = new File(rootDirectory, file[i]);
            o.at(f, indentLevel);

            // If the file is a directory, recurse into it
            if (f.isDirectory()) {
                walk(f, o, indentLevel + 1);
            }
        }

    }

    /**
     * Writes the XML output file to the location and file name dictated by
     * the third command line argument as set in class Main.
     */
    public void writeXMLFile() {
        try {
            File f = Futil.getWriteFile("enter output xml file");
            PrintWriter out = new PrintWriter(
                    new BufferedWriter(
                            new FileWriter(f)));

            out.write(xmlString);
            out.close();


        } catch (IOException ioe) {
            System.out.println(
                    "Error writing XML output to path specified.");
        }

    }


}
