/************************************************************************************
 * Copyright (C) 2002  Cristiano Sadun crsadun@tin.it
 *
 * You can redistribute this program and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation-
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ************************************************************************************/

package classUtils.pack.util;

import classUtils.pack.Constants;
import classUtils.putils.ClassPathBean;
import classUtils.putils.ClassPathUtils;
import collections.sortable.SortableVector;
import gui.In;

import java.io.*;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

/**
 * Pack dependencies on a given set of classes.
 * <p/>
 * The available attributes are: <ul> <li><b>classes</b>: a comma-separated
 * list of classes to pack <li><b>packages</b>: a comma-separated list of
 * packages to pack. Each class in the package will be included, together
 * with its dependents. <li><b>classpath</b>: additional classpath to use
 * when packing classes (optional) <li><b>targetJar</b>: the name of the
 * jar file to produce <li><b>mainfestClasspath</b>: the (optional)
 * manifest Class-Path entry <li><b>mainfestMainclass</b>: the (optional)
 * manifest Main-Class entry <li><b>excludePkg</b>: a comma-separated list
 * of package prefixes to exclude. Defaults to <b>java,javax,sun</b>
 * <li><b>includePkg</b>: a comma-separated list of package prefixes to
 * include. Only classes in matching packages will be included. Has lower
 * precedence than excludePkg <li><b>resolveFiltered</b>: if <b>true</b>,
 * allows resolution of classes which are filtered out. Defaults to
 * <b>false</b>. <li><b>cacheClassFiles</b>: if <b>false</b>, disables
 * classfile caching - slower packing but saving memory </ul>
 * <p/>
 * Additional classpath can be also specified by a nested
 * <code>&lt;classpath&gt;</code> element.
 * <p/>
 * &lt;pack&gt; also supports inclusion of explicitly named additional
 * classes and/or files in the jar. Dependencies for such additional
 * classes will be computed and added too. This is done by declaring
 * internal <b>&lt;additionalclass&gt;</b> and <b>&lt;additionalfileset&gt;</b>
 * elements.
 * <p/>
 * <b>&lt;additionalclass&gt;</b> has a single <b>name</b> attribute which
 * contains the fully qualified name of the class to include. The class
 * must: <ul> <li> be in classpath; <li> not conflict with the filter
 * established by <b>excludePkg/includePkg</b>. </ul>
 * <p/>
 * For example,
 * <pre>
 * &lt;additionalclass name="javax.transaction.TransactionManager"/&gt;
 * </pre>
 * will add the <code>javax.transaction.TransactionManager</code> class and
 * all its dependent classes to the produced jar.
 * <p/>
 * <b>&lt;additionalfileset&gt;</b> is a standard Ant <code>FileSet</code>
 * structure which specifies a set of files to unconditionally add to the
 * produced jar.
 * <p/>
 * For example,
 * <pre>
 *   &lt;additionalfileset dir="${basedir}"&gt;
 * 	  &lt;include name="META-INF/services/*"/&gt;
 *   &lt;/additionalfileset&gt;
 * </pre>
 * <p/>
 * will add any file under the <code>META-INF/service</code> subdirectory
 * of the current <code>${basedir}</code> directory.
 * <p/>
 * <p/>
 * $Revision$
 *
 * @author Cristiano Sadun, Doug Lyon (Removed ant junk)
 * @version 2.0dj
 */
public class Pack {
    private ClassMap clsMap;

    private final PackTask task = new PackTask(this);
    private ClassFinderUtils classFinderUtils;

    private ClassPathBean cpb = ClassPathBean.restore();

    //todo make a gui option out of this
    private static boolean detectrmi = false;

    public Pack() {
        init();
    }

    private void init() {
        classFinderUtils =
                new ClassFinderUtils(getClassPath());
        clsMap = new ClassMap();
    }


    private String classes;
    private String packages;
    private File targetJarFile;
    private boolean resolveFiltered = false;
    private String excludePkg = Constants.DEFAULT_EXCLUDE_PACKAGES;
    private String includePkg = Constants.DEFAULT_INCLUDE_PACKAGES;

    private String manifestClassPath;
    private String manifestMainClass;
    private boolean hasCacheClassFiles = true;

    private ClassFilter acceptFilter;
    private HashSet refusedNames;
    private Set additionalClasses = new HashSet();
    private Set resources = new HashSet();

    private Set ignorableClasses = new HashSet();

    void writeOutJar() {
        try {
            JarOutputStream jos = new JarOutputStream(new BufferedOutputStream(
                    new FileOutputStream(targetJarFile)));
            if (manifestClassPath != null |
                    manifestMainClass != null)
                addManifest(jos);

            log("Packing " + targetJarFile);

            processDependencies(jos);
            // this is busted - > addAdditionalFiles(jos);
            addResources(jos);

            jos.close();

        } catch (IOException e) {
            e.printStackTrace();

        }
    }

    private void addResources(JarOutputStream jos) throws IOException {
        if (resources.isEmpty()) {
            log("Pack:resources are empty...not adding resources");
            return;
        }
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource res = (Resource) i.next();
            log("Adding resource " + res);
            InputStream is = classFinderUtils.openResource(res.name);
            if (is == null)
                throw new IOException("resource " +
                        res.name +
                        " not found. ClassPath is " +
                        classFinderUtils.getClassPath());
            JarEntry entry = new JarEntry(res.name);
            jos.putNextEntry(entry);
            int c;
            while ((c = is.read()) != -1)
                jos.write(c);
        }
    }

    private void addManifest(JarOutputStream jos) throws IOException {
        log("adding manifest");
        Manifest manifest = new Manifest();
        manifest.getMainAttributes().put(Attributes.Name.MANIFEST_VERSION,
                "1.0");
        if (manifestClassPath != null) {
            manifest.getMainAttributes()
                    .put(Attributes.Name.CLASS_PATH,
                            manifestClassPath);
        }
        if (manifestMainClass != null) {
            manifest.getMainAttributes()
                    .put(Attributes.Name.MAIN_CLASS,
                            manifestMainClass);
        }
        JarEntry entry = new JarEntry("META-INF/MANIFEST.MF");
        jos.putNextEntry(entry);
        manifest.write(jos);
    }

    void addAdditionalClasses() throws IOException,
                                       ClassNotFoundException {

            for (Iterator i = additionalClasses.iterator(); i.hasNext();) {
                ClassSpec cls = (ClassSpec) i.next();
                log("Finding dependencies for additional class " +
                        cls.name);
                // Find the dependencies for each additional class
                findDependencies(this, classFinderUtils,
                        acceptFilter,
                        refusedNames,
                        resolveFiltered,
                        cls.name,
                        clsMap,
                        true);
            }

    }

    void computeDependencies(String[] clsNames) throws IOException,
                                                       ClassNotFoundException {
        for (int i = 0; i < clsNames.length; i++)
            listDependencies(clsNames[i]);
    }

    private void listDependencies(final String clsName)
            throws IOException, ClassNotFoundException {
        log("Calculating dependencies for " +
                clsName);
        log("Classpath is " +
                classFinderUtils.getClassPath());
        // Find the dependecies for each class
        findDependencies(this, clsName,
                clsMap);
    }

    public void validateInput() {
        if (targetJarFile == null)
            log(
                    "Missing mandatory targetJar attribute");
        if (classes == null && packages == null)
            log(
                    "Missing mandatory classes or packages attribute");
        if (classes != null && packages != null)
            log(
                    "Only one of classes or packages can be specified");


    }

    public void copyIgnorableClassesToLocalHashSet() {
        Set hs = new HashSet();
        for (Iterator i = ignorableClasses.iterator(); i.hasNext();) {
            hs.add(((ClassSpec) i.next()).name);
        }
        ignorableClasses = hs;
    }

    private void processDependencies(JarOutputStream jos)
            throws IOException {
        for (Iterator i = clsMap.getKeySet()
                .iterator(); i.hasNext();) {
            String clsName = (String) i.next();
            String entryName = clsName.replace('.', '/') +
                    ".class";
            JarEntry entry = new JarEntry(entryName);
            jos.putNextEntry(entry);
            byte[] bytecode = (byte[]) clsMap.get(clsName);
            ByteArrayInputStream is = new ByteArrayInputStream(bytecode);
            int c;
            while ((c = is.read()) != -1)
                jos.write(c);
        }
    }

    public void printDependencies() throws Exception {
        String cn = In.getString("enter class name");
        if (cn == null) return;
        printDependencies(cn);
    }

    private void printDependencies(String cn) throws Exception {
        setManifestMainClass(cn);
        setClasses(cn);

        final String targetJar = cn + ".jar";
        In.message("am creating output file:" + targetJar);
        setTargetJarFile(new File(targetJar));
        try {
            packIt();
        } catch (Exception e) {
            In.message(e + " for class:" + cn);
            printDependencies(cn);
        }
        printDependcies();
    }

    private void printDependcies() {
        SortableVector sv = new SortableVector();
        if (clsMap == null) return;
        for (Iterator i = clsMap.getKeySet()
                .iterator(); i.hasNext();) {
            String clsName = (String) i.next();
            sv.addElement(clsName);

        }
        sv.sort();
        log("Dependencies:" + sv.size());
        for (int i = 0; i < sv.size(); i++)
            log(sv.elementAt(i).toString());
    }


    public static void findDependencies(Pack pack, String clsName,
                                        ClassMap clsMap)
            throws IOException, ClassNotFoundException {
        System.out.println(clsName);
        findDependencies(pack,
                pack.classFinderUtils,
                pack.acceptFilter,
                pack.refusedNames,
                pack.resolveFiltered,
                clsName,
                clsMap,
                false);
    }

    /**
     * @param pack
     * @param classFinderUtils1
     * @param acceptFilter
     * @param refusedNames1
     * @param resolveFiltered1
     * @param clsName
     * @param clsMap
     * @param failOnUnaccepted
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public static void findDependencies(Pack pack,
                                        ClassFinderUtils classFinderUtils1,
                                        ClassFilter acceptFilter,
                                        HashSet refusedNames1,
                                        boolean resolveFiltered1,
                                        String clsName,
                                        ClassMap clsMap,
                                        boolean failOnUnaccepted)
            throws IOException,
                   ClassNotFoundException {

        if (pack.ignorableClasses.contains(clsName)) {
            pack.log(clsName +
                    " ignored as configured");
            return;
        }

        String[] tmp_f = ClassPathUtils.splitClassName(clsName);
        boolean accepted_f = acceptFilter.accept(tmp_f[0],
                tmp_f[1],
                null);
        if (accepted_f) {
            if (clsMap.contains(clsName)) {
                //log(clsName+" already accepted.", project.MSG_VERBOSE);
                return;
            }
            if (refusedNames1.contains(clsName)) {
                //log(clsName+" already refused.", project.MSG_VERBOSE);
                return;
            }

            // Is the name an array? Try to find the component class and return
            if (clsName.startsWith("[L")) {
                String clsName2 = clsName.substring(2, clsName.length() - 1);
                Pack.findDependencies(pack, clsName2, clsMap);
                return;
            }
            if (clsName.startsWith("[")) {
                String clsName2 = clsName.substring(1);
                if ("B".equals(clsName2))
                    return;
                else if ("C".equals(clsName2))
                    return;
                else if ("D".equals(clsName2))
                    return;
                else if ("F".equals(clsName2))
                    return;
                else if ("I".equals(clsName2))
                    return;
                else if ("J".equals(clsName2))
                    return;
                else if ("S".equals(clsName2))
                    return;
                else if ("Z".equals(clsName2))
                    return;
                else if ("V".equals(clsName2)) return;
                Pack.findDependencies(pack, clsName2, clsMap);
                return;
            }

            // Load the class
            byte[] bytecode = classFinderUtils1.getClassBytes(clsName);
            CPoolReader.ClassFile cf = classFinderUtils1.readClassData(
                    bytecode);
            // Add it to the set, if not filtered out
            String[] tmp = ClassPathUtils.splitClassName(clsName);
            boolean accepted = acceptFilter.accept(tmp[0],
                    tmp[1],
                    cf);
            if (failOnUnaccepted && !accepted)
                In.message("The class " +
                        tmp[0] +
                        "." +
                        tmp[1] +
                        " is not acceptable with the current " +
                        "includePkg/excludePkg settings (" +
                        acceptFilter +
                        ")");
            if (accepted) {
                clsMap.put(clsName, bytecode);
                pack.log(clsName + " accepted.");
            } else {
                refusedNames1.add(clsName);
                pack.log(clsName + " refused.");
            }
            if (accepted || resolveFiltered1) {
                //todo why are we check for stubs? This causes SDA failure.

                checkForRmiStubs(detectrmi,
                        cf,
                        clsName,
                        classFinderUtils1,
                        clsMap);

                scanForDependencies(cf, pack, clsMap);
            }
        }
    }

    private static void scanForDependencies(CPoolReader.ClassFile cf,
                                            Pack pack, ClassMap clsMap)
            throws IOException, ClassNotFoundException {
        // Browse trhu all the class names mentioned in
        // the constant pool, and find all their dependencies
        String[] usedClasses = cf.getUsedClasses();
        for (int i = 0; i <
                usedClasses.length; i++) {
            String usedClassName = usedClasses[i].replace('/', '.');
            Pack.findDependencies(pack, usedClassName,
                    clsMap);
        }
    }

    //todo this needs more testing
    public static void checkForRmiStubs(boolean isRmiDetectionOn,
                                         CPoolReader.ClassFile cf,
                                         String clsName,
                                         ClassFinderUtils classFinderUtils1,
                                         ClassMap clsMap)
            throws IOException, ClassNotFoundException {
        // If RMI detection is active and the
        // class implements UnicastRemoteObject,
        // try to find the stubs
        if (!isRmiDetectionOn) return;
        if (cf.isInterface()) return;

        String superClass = cf.getSuperClass();

        if (!isUnicastRemoteObject(superClass)) return;

        print("RMI scanner found UnicastRemoteObject:" + clsName);
        String stubClsName = clsName +
                "_Stub";
        String skelClsName = clsName +
                "_Skel";
        print("looking for:" + stubClsName);
        clsMap.put(stubClsName,
                classFinderUtils1.getClassBytes(stubClsName));
        print("looking for:" + skelClsName);
        clsMap.put(skelClsName,
                classFinderUtils1.getClassBytes(skelClsName));
    }
    public static boolean isUnicastRemoteObject(String aClass){
        if (aClass.equals("java.rmi.server.UnicastRemoteObject")) return true;
        return aClass.equals("java/rmi/server/UnicastRemoteObject");
    }
    public static boolean isRemote(String[] interfaces) {
        for (int i = 0; i < interfaces.length; i++)
            if (isRemote(interfaces[i])) return true;
        return false;
    }

    public static boolean isRemote(String anInterface) {
        if (anInterface.equals("java.rmi.Remote")) return true;
        return anInterface.equals("java/rmi/Remote");
    }

    public static void print(String[] interfaces) {
        for (int i = 0; i < interfaces.length; i++)
            print(interfaces[i]);
    }

    public static void print(String s) {
        System.out.println(s);
    }

    /**
     * Returns the classes.
     *
     * @return String
     */
    public String getClasses() {
        return classes;
    }

    /**
     * Sets the classes.
     *
     * @param classes The classes to set
     */
    public void setClasses(String classes) {
        this.classes = classes;
    }

    /**
     * Returns the targetJar.
     *
     * @return String
     */
    public File getTargetJarFile() {
        return targetJarFile;
    }

    /**
     * Sets the targetJar.
     *
     * @param targetJarFile The targetJar to set
     */
    public void setTargetJarFile(File targetJarFile) {
        this.targetJarFile = targetJarFile;
    }

    /**
     * Returns the resolveFiltered.
     *
     * @return boolean
     */
    public boolean getResolveFiltered() {
        return resolveFiltered;
    }

    /**
     * Sets the resolveFiltered.
     *
     * @param resolveFiltered The resolveFiltered to set
     */
    public void setResolveFiltered(boolean resolveFiltered) {
        this.resolveFiltered = resolveFiltered;
    }

    /**
     * Returns the excludePkg.
     *
     * @return String
     */
    public String getExcludePkg() {
        return excludePkg;
    }

    /**
     * Sets the excludePkg.
     *
     * @param excludePkg The excludePkg to set
     */
    public void setExcludePkg(String excludePkg) {
        this.excludePkg = excludePkg;
    }

    /**
     * Returns the classpath.
     *
     * @return String
     */
    public String getClassPath() {
        System.out.println("ClassPath @ Pack.getClassPath()" + cpb.getClassPath());
    	return cpb.getClassPath();
    }

    /**
     * Returns the manifestClassPath.
     *
     * @return String
     */
    public String getManifestClassPath() {
        return manifestClassPath;
    }

    /**
     * Returns the manifestMainClass.
     *
     * @return String
     */
    public String getManifestMainClass() {
        return manifestMainClass;
    }

    /**
     * Sets the manifestClassPath.
     *
     * @param manifestClassPath The manifestClassPath to set
     */
    public void setManifestClassPath(String manifestClassPath) {
        this.manifestClassPath =
                manifestClassPath;
    }

    /**
     * Sets the manifestMainClass.
     *
     * @param manifestMainClass The manifestMainClass to set
     */
    public void setManifestMainClass(String manifestMainClass) {
        this.manifestMainClass =
                manifestMainClass;
    }

    /**
     * Returns the includePkg.
     *
     * @return String
     */
    public String getIncludePkg() {
        return includePkg;
    }

    /**
     * Sets the includePkg.
     *
     * @param includePkg The includePkg to set
     */
    public void setIncludePkg(String includePkg) {
        this.includePkg = includePkg;
    }


    /**
     * Ant entry point for <code>additionalClass</code> subelements.
     * <p/>
     *
     * @return AdditionalClass an object containing info about the class to
     *         add
     */
    public ClassSpec createAdditionalClass() {
        ClassSpec cls = new ClassSpec();
        additionalClasses.add(cls);
        return cls;
    }

    /**
     * Ant entry point for <code>ignoreClass</code> subelements.
     * <p/>
     *
     * @return AdditionalClass an object containing info about the class to
     *         add
     */
    public ClassSpec initIgnorableClasses() {
        ClassSpec cls = new ClassSpec();
        ignorableClasses.add(cls);
        return cls;
    }

    /**
     * Ant entry point for <code>additionalClass</code> subelements.
     * <p/>
     *
     *  AdditionalClass an object containing info about the class to
     *         add
     */
    public void addResource(Resource r){
        resources.add(r);
    }

    /**
     * Returns the cacheClassFiles.
     *
     * @return boolean
     */
    public boolean hasCacheClassFiles() {
        return hasCacheClassFiles;
    }

    /**
     * Sets the cacheClassFiles.
     *
     * @param hasCacheClassFiles The cacheClassFiles to set
     */
    public void setHasCacheClassFiles(boolean hasCacheClassFiles) {
        this.hasCacheClassFiles = hasCacheClassFiles;
    }

    /**
     * Returns the packages.
     *
     * @return String
     */
    public String getPackages() {
        return packages;
    }

    /**
     * Sets the packages.
     *
     * @param packages The packages to set
     */
    public void setPackages(String packages) {
        this.packages = packages;
    }

    public void packIt() throws IOException, ClassNotFoundException {
        task.packIt();
    }

    public void log(String s) {
        task.log(s);
    }



    void setRefusedNames(HashSet refusedNames) {
        this.refusedNames = refusedNames;
    }

    void setAcceptFilter(ClassFilter acceptFilter) {
        this.acceptFilter = acceptFilter;
    }

    void setCache(boolean b) {
        classFinderUtils.setCache(true);
    }
}
