package classUtils.dumper;

import futils.DirList;
import futils.Futil;
import futils.WildFilter;
import gui.In;

import java.io.*;
import java.util.Vector;

/**
 * This class is used to manipulate Java class
 * files in strange and mysterious ways. Usage it
 * typically to feed it an array of bytes that are
 * a class file, manipulate the class, then
 * convert the class back into bytes, and feed the
 * final result to <TT>defineClass()</TT>.
 *
 * @version 1.6, 19 Aug 1995
 */
public class ClassFile {
    private int magic;
    private short majorVersion;
    private short minorVersion;
    private ConstantPoolInfo constantPool[];
    private short accessFlags;
    private ConstantPoolInfo thisClass;
    private ConstantPoolInfo superClass;
    private ConstantPoolInfo interfaces[];
    private FieldInfo fields[];
    private MethodInfo methods[];
    private AttributeInfo attributes[];
    private boolean isValidClass = false;
    public static final int ACC_PUBLIC = 0x1;
    public static final int ACC_PRIVATE = 0x2;
    public static final int ACC_PROTECTED = 0x4;
    public static final int ACC_STATIC = 0x8;
    public static final int ACC_FINAL = 0x10;
    public static final int ACC_SYNCHRONIZED = 0x20;
    public static final int ACC_THREADSAFE = 0x40;
    public static final int ACC_TRANSIENT = 0x80;
    public static final int ACC_NATIVE = 0x100;
    public static final int ACC_INTERFACE = 0x200;
    public static final int ACC_ABSTRACT = 0x400;
    public boolean dumpConstants = false;

    /**
     * Read a class from InputStream <i>in</i>.
     */
    public boolean read(InputStream in)
            throws IOException {
        DataInputStream di = new DataInputStream(in);
        int count;
        setMagic(di.readInt());
        if (getMagic() != 0xCAFEBABE) {
            return (false);
        }
        setMajorVersion(di.readShort());
        setMinorVersion(di.readShort());
        count = di.readShort();
        setConstantPool(new ConstantPoolInfo[count]);
        getConstantPool()[0] =
                new ConstantPoolInfo();
        for (int i = 1;
             i < getConstantPool().length;
             i++) {
            getConstantPool()[i] =
                    new ConstantPoolInfo();
            if (!getConstantPool()[i].read(di))
                return (false);

            // These two types take up "two" spots in the table
            if ((getConstantPool()[i].type ==
                    ConstantPoolInfo.LONG) ||
                    (getConstantPool()[i].type ==
                    ConstantPoolInfo.DOUBLE))
                i++;
        }

        /*
         * Update pointers in the constant table. This turns the
         * table into a real datastructure.
         *
         * TODO: Have it verify that the right arguments are present
         */

        for (int i = 1;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i] == null)
                continue;
            if (getConstantPool()[i].index1 > 0)
                getConstantPool()[i].arg1 =
                        getConstantPool()[getConstantPool()[i].index1];
            if (getConstantPool()[i].index2 > 0)
                getConstantPool()[i].arg2 =
                        getConstantPool()[getConstantPool()[i].index2];
        }
        if (dumpConstants) {
            for (int i = 1;
                 i < getConstantPool().length;
                 i++) {
                System.out.println("C" +
                        i +
                        " - " +
                        getConstantPool()[i]);
            }
        }
        setAccessFlags(di.readShort());
        setThisClass(getConstantPool()[di.readShort()]);
        setSuperClass(getConstantPool()[di.readShort()]);
        /*
         * Identify all of the interfaces implemented by this class
         */

        count = di.readShort();
        if (count != 0) {
            setInterfaces(new ConstantPoolInfo[count]);
            for (int i = 0; i < count; i++) {
                int iindex = di.readShort();
                if ((iindex < 1) ||
                        (iindex >
                        getConstantPool().length -
                        1))
                    return (false);
                getInterfaces()[i] =
                        getConstantPool()[iindex];
            }
        }

        /*
         * Identify all fields in this class.
         */

        count = di.readShort();
        if (count != 0) {
            setFields(new FieldInfo[count]);
            for (int i = 0; i < count; i++) {
                getFields()[i] = new FieldInfo();
                if (!getFields()[i].read(di,
                        getConstantPool())) {
                    return (false);
                }
            }
        }


        /*

         * Identify all the methods in this class.

         */

        count = di.readShort();
        if (count != 0) {
            setMethods(new MethodInfo[count]);
            for (int i = 0; i < count; i++) {
                getMethods()[i] =
                        new MethodInfo();
                if (!getMethods()[i].read(di,
                        getConstantPool())) {
                    return (false);
                }
            }
        }





        /*

         * Identify all of the attributes in this class

         */

        count = di.readShort();
        if (count != 0) {
            setAttributes(new AttributeInfo[count]);
            for (int i = 0; i < count; i++) {
                getAttributes()[i] =
                        new AttributeInfo();
                if (!getAttributes()[i].read(di,
                        getConstantPool())) {
                    return (false);
                }
            }
        }
        setValidClass(true);
        return (true);
    }

    /**
     * Write the class out as a stream of bytes to
     * the output stream. Generally you will read
     * a class file, manipulate it in some way,
     * and then write it out again before passing
     * it to <TT>defineClass</TT> in some class
     * loader.
     */
    public void write(OutputStream out)
            throws IOException, Exception {
        DataOutputStream dos = new DataOutputStream(out);
        if (!isValidClass()) {
            throw new Exception("ClassFile::write() - Invalid Class");
        }
        dos.writeInt(getMagic());
        dos.writeShort(getMajorVersion());
        dos.writeShort(getMinorVersion());
        dos.writeShort(getConstantPool().length);
        for (int i = 1;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i] != null)
                getConstantPool()[i].write(dos,
                        getConstantPool());
        }
        dos.writeShort(getAccessFlags());
        dos.writeShort(ConstantPoolInfo.indexOf(getThisClass(),
                getConstantPool()));
        dos.writeShort(ConstantPoolInfo.indexOf(getSuperClass(),
                getConstantPool()));
        if (getInterfaces() == null)
            dos.writeShort(0);
        else {
            dos.writeShort(getInterfaces().length);
            for (int i = 0;
                 i < getInterfaces().length;
                 i++)
                dos.writeShort(ConstantPoolInfo.indexOf(getInterfaces()[i],
                        getConstantPool()));
        } // end else
        if (getFields() == null) {
            dos.writeShort(0);
        } else {
            dos.writeShort(getFields().length);
            for (int i = 0;
                 i < getFields().length;
                 i++) {
                getFields()[i].write(dos,
                        getConstantPool());
            }
        }
        if (getMethods() == null) {
            dos.writeShort(0);
        } else {
            dos.writeShort(getMethods().length);
            for (int i = 0;
                 i < getMethods().length;
                 i++) {
                getMethods()[i].write(dos,
                        getConstantPool());
            }
        }
        if (getAttributes() == null) {
            dos.writeShort(0);
        } else {
            dos.writeShort(getAttributes().length);
            for (int i = 0;
                 i < getAttributes().length;
                 i++) {
                getAttributes()[i].write(dos,
                        getConstantPool());
            }
        }
    }

    /**
     * Returns a string that represents what the
     * access flags are set for. So 0x14 returns
     * "public final "
     */
    public static String accessString(short flags) {
        StringBuffer x = new StringBuffer();
        if ((flags & ACC_PUBLIC) != 0) {
            x.append("public ");
        }
        if ((flags & ACC_PRIVATE) != 0) {
            x.append("private ");
        }
        if ((flags & ACC_PROTECTED) != 0) {
            x.append("protected ");
        }
        if ((flags & ACC_STATIC) != 0) {
            x.append("static ");
        }
        if ((flags & ACC_FINAL) != 0) {
            x.append("final ");
        }
        if ((flags & ACC_SYNCHRONIZED) != 0) {
            x.append("synchronized ");
        }
        if ((flags & ACC_THREADSAFE) != 0) {
            x.append("threadsafe ");
        }
        if ((flags & ACC_TRANSIENT) != 0) {
            x.append("transient ");
        }
        if ((flags & ACC_NATIVE) != 0) {
            x.append("native ");
        }
        if ((flags & ACC_INTERFACE) != 0) {
            x.append("interface ");
        }
        if ((flags & ACC_ABSTRACT) != 0) {
            x.append("abstract ");
        }
        return (x.toString());
    }

    /**
     * Takes a type signature and a string
     * representing a variable name and returns a
     * declaration for that variable name. For
     * example, passing this the strings "[B" and
     * "myArray" will return the string "byte
     * myArray[]"
     */
    public static String typeString(String typeString, String varName) {
        int isArray = 0;
        int ndx = 0;
        StringBuffer x = new StringBuffer();
        while (typeString.charAt(ndx) == '[') {
            isArray++;
            ndx++;
        }
        switch (typeString.charAt(ndx)) {
            case 'B':
                x.append("byte ");
                break;
            case 'C':
                x.append("char ");
                break;
            case 'D':
                x.append("double ");
                break;
            case 'F':
                x.append("float ");
                break;
            case 'I':
                x.append("int ");
                break;
            case 'J':
                x.append("long ");
                break;
            case 'L':
                for (int i = ndx + 1;
                     i < typeString.indexOf(';');
                     i++) {
                    if (typeString.charAt(i) !=
                            '/')
                        x.append(typeString.charAt(i));
                    else
                        x.append('.');
                }
                x.append(" ");
                break;
            case 'V':
                x.append("void ");
                break;
            case 'S':
                x.append("short ");
                break;
            case 'Z':
                x.append("boolean ");
                break;
        }
        x.append(varName);
        while (isArray > 0) {
            x.append("[]");
            isArray--;
        }
        return (x.toString());
    }

    /**
     * Returns the next signature from a string of
     * concatenated signatures. For example if the
     * signature was "[BII", this method would
     * return "II"
     */
    public static String nextSig(String sig) {
        int ndx = 0;
        String x;
        while (sig.charAt(ndx) == '[')
            ndx++;
        if (sig.charAt(ndx) == 'L') {
            while (sig.charAt(ndx) != ';')
                ndx++;
        }
        ndx++;
        x = (sig.substring(ndx));
        return (x);
    }

    /**
     * Print the name of a class in "canonical
     * form"
     */
    private String getClassName(String s) {
        StringBuffer x;
        if (s.charAt(0) == '[') {
            return (typeString(s, ""));
        }
        x = new StringBuffer();
        for (int j = 0; j < s.length(); j++) {
            if (s.charAt(j) == '/')
                x.append('.');
            else
                x.append(s.charAt(j));
        }
        return (x.toString());
    }

    public String getClassName() {
        final String s = getThisClass().arg1.strValue;
        if (s == null) return null;
        return getClassName(s);
    }

    /**
     * The boring version of display().
     */
    public String toString() {
        return ("Class File (Version " +
                getMajorVersion() +
                "." +
                getMinorVersion() +
                ") for class " +
                getThisClass().arg1);
    }

    /**
     * Write out a text version of this class.
     */
    public void display()
            throws Exception {
        int i;
        String myClassName;
        String packageName = null;
        if (!isValidClass()) {
            In.message("not a valid class");
            throw new Exception();
        }
        myClassName =
                getClassName(getThisClass().arg1.strValue);
        if (myClassName.indexOf('.') > 0) {
            packageName =
                    myClassName.substring(0,
                            myClassName.lastIndexOf('.'));
            myClassName =
                    myClassName.substring(myClassName.lastIndexOf('.') +
                    1);
        }
        for (i = 1;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i] == null)
                continue;
            if ((getConstantPool()[i] ==
                    getThisClass()) ||
                    (getConstantPool()[i] ==
                    getSuperClass()))
                continue;
            if (getConstantPool()[i].type ==
                    ConstantPoolInfo.CLASS) {
                String s = getConstantPool()[i].arg1.strValue;
                if (s.charAt(0) == '[')
                    continue;
                s =
                        getClassName(getConstantPool()[i].arg1.strValue);
                if ((packageName != null) &&
                        (s.startsWith(packageName)))
                    continue;
            }
        }
        DataInputStream dis;
        ConstantPoolInfo cpi;
        if (getAttributes() != null) {
            for (i = 0;
                 i < getAttributes().length;
                 i++) {
                String attrName = getAttributes()[i].getName()
                        .strValue;
                dis =
                        new DataInputStream(new ByteArrayInputStream(getAttributes()[i].getData()));
                if (attrName.compareTo("SourceFile") ==
                        0) {
                    cpi = null;
                    try {
                        cpi =
                                getConstantPool()[dis.readShort()];
                    } catch (IOException e) {
                    }
                }
            }
        }
    }



    public ConstantPoolInfo getConstantRef(short index) {
        return (getConstantPool()[index]);
    }

    /**
     * Add a single constant pool item and return
     * its index. If the item is already in the
     * pool then the index of the <i>preexisting</i>
     * item is returned. Thus you cannot assume
     * that a pointer to your item will be
     * useful.
     */
    public short addConstantPoolItem(ConstantPoolInfo item)
            throws Exception {
        ConstantPoolInfo newConstantPool[];
        ConstantPoolInfo cp;
        cp = item.inPool(getConstantPool());
        if (cp != null)
            return ConstantPoolInfo.indexOf(cp,
                    getConstantPool());
        newConstantPool =
                new ConstantPoolInfo[getConstantPool()
                .length +
                1];
        for (int i = 1;
             i < getConstantPool().length;
             i++) {
            newConstantPool[i] =
                    getConstantPool()[i];
        }
        newConstantPool[getConstantPool().length] =
                item;
        setConstantPool(newConstantPool);
        return ConstantPoolInfo.indexOf(item,
                getConstantPool());
    }

    /**
     * Add some items to the constant pool. This
     * is used to add new items to the constant
     * pool. The items references in arg1 and arg2
     * are expected to be valid pointers (if
     * necessary). Pruning is done to prevent
     * adding redundant items to the list and to
     * preserve string space. The algorithm is
     * simple, first identify pool items
     * containing constants in the list of items
     * to be added that are already in the
     * constant pool. If any are found to already
     * exist, change the pointers in the
     * non-constant items to point to the ones in
     * the pool rather than the ones in the list.
     * Next check to see if any of the
     * non-constant items are already in the pool
     * and if so fix up the others in the list to
     * point to the ones in the pool. Finally, add
     * any items (there must be at least one) from
     * the item list that aren't already in the
     * pool, all of the pointers will already be
     * fixed. NOTE: Since constants in the
     * constant pool may be referenced
     * <i>inside</i> the opaque portion of
     * attributes the constant table cannot be
     * re-ordered, only extended.
     */
    public void addConstantPoolItems(ConstantPoolInfo items[]) {
        ConstantPoolInfo newArg;
        ConstantPoolInfo newConstantPool[];
        boolean delete[] = new boolean[items.length];



        /* Step one, look for matching constants */

        for (int j = 0; j < items.length; j++) {
            if ((items[j].type ==
                    ConstantPoolInfo.ASCIZ) ||
                    (items[j].type ==
                    ConstantPoolInfo.UNICODE) ||
                    (items[j].type ==
                    ConstantPoolInfo.INTEGER) ||
                    (items[j].type ==
                    ConstantPoolInfo.LONG) ||
                    (items[j].type ==
                    ConstantPoolInfo.FLOAT) ||
                    (items[j].type ==
                    ConstantPoolInfo.DOUBLE)) {
                // Look for this item in the constant pool

                delete[j] = false;
                newArg =
                        items[j].inPool(getConstantPool());
                if (newArg != null) {
                    // replace the references in our list.

                    delete[j] = true;	// mark it for deletion
                    for (int i = 0;
                         i < items.length;
                         i++) {
                        if (items[i].arg1 ==
                                items[j])
                            items[i].arg1 =
                                    newArg;
                        if (items[i].arg2 ==
                                items[j])
                            items[i].arg2 =
                                    newArg;
                    }
                }
            }
        }



        /* Step two : now match everything else */

        for (int j = 0; j < items.length; j++) {
            if ((items[j].type ==
                    ConstantPoolInfo.CLASS) ||
                    (items[j].type ==
                    ConstantPoolInfo.FIELDREF) ||
                    (items[j].type ==
                    ConstantPoolInfo.METHODREF) ||
                    (items[j].type ==
                    ConstantPoolInfo.STRING) ||
                    (items[j].type ==
                    ConstantPoolInfo.INTERFACE) ||
                    (items[j].type ==
                    ConstantPoolInfo.NAMEANDTYPE)) {
                // Look for this item in the constant pool

                delete[j] = false;
                newArg =
                        items[j].inPool(getConstantPool());
                if (newArg != null) {
                    // replace the references in our list.

                    delete[j] = true;	// mark it for deletion
                    for (int i = 0;
                         i < items.length;
                         i++) {
                        if (items[i].arg1 ==
                                items[j])
                            items[i].arg1 =
                                    newArg;
                        if (items[i].arg2 ==
                                items[j])
                            items[i].arg2 =
                                    newArg;
                    }
                }
            }
        }



        /* Step three: Add the surviving items to the pool */

        int count = 0;
        for (int i = 0; i < items.length; i++) {
            if (!delete[i])
                count++;
        }

        // count == # of survivors

        newConstantPool =
                new ConstantPoolInfo[getConstantPool()
                .length +
                count];
        for (int i = 1;
             i < getConstantPool().length;
             i++) {
            newConstantPool[i] =
                    getConstantPool()[i];
        }

        // newConstantPool == existing constantPool



        int ndx = 0;
        for (int i = getConstantPool().length;
             i < newConstantPool.length;
             i++) {
            while (delete[ndx])
                ndx++;
            newConstantPool[i] = items[ndx];
            ndx++;
        }

        // newConstantPool == existing + new

        setConstantPool(newConstantPool);

        // all done.
    }

    /**
     * Add a new optional class Attribute. Items
     * is an array of constant pool items that are
     * first added to the constant pool. At a
     * minimum items[0] must be an ASCIZ item with
     * the name of the attribute. If the body of
     * the attribute references constant pool
     * items these should be in the item list as
     * well.
     */
    public void addAttribute(AttributeInfo newAttribute) {
        if (getAttributes() == null) {
            setAttributes(new AttributeInfo[1]);
            getAttributes()[0] = newAttribute;
        } else {
            AttributeInfo newAttrList[] = new AttributeInfo[1 +
                    getAttributes()
                    .length];
            for (int i = 0;
                 i < getAttributes().length;
                 i++) {
                newAttrList[i] =
                        getAttributes()[i];
            }
            newAttrList[getAttributes().length] =
                    newAttribute;
            setAttributes(newAttrList);
        }
    }

    /**
     * Return the attribute named 'name' from the
     * class file.
     */
    public AttributeInfo getAttribute(String name) {
        if (getAttributes() == null)
            return null;
        for (int i = 0;
             i < getAttributes().length;
             i++) {
            if (name.compareTo(getAttributes()[i].getName()
                    .toString()) ==
                    0)
                return getAttributes()[i];
        }
        return (null);
    }

    /**
     * Return a constant pool item from this
     * class. (note does fixup of indexes to
     * facilitate extracting nested or linked
     * items.
     */
    public ConstantPoolInfo getConstantPoolItem(short index)
            throws Exception {
        ConstantPoolInfo cp;
        if ((index <= 0) ||
                (index >
                (getConstantPool().length - 1)))
            return (null);
        cp = getConstantPool()[index];
        if (cp.arg1 != null)
            cp.index1 =
                    ConstantPoolInfo.indexOf(cp.arg1,
                            getConstantPool());
        if (cp.arg2 != null)
            cp.index2 =
                    ConstantPoolInfo.indexOf(cp.arg2,
                            getConstantPool());
        return cp;
    }



    /* Examples of mysterious things you can do to a class file before

     * writing it back out. These methods are not currently functional.

     * (that would be too easy :-)

     */



    /**
     * Map occurences of class <i>oldClass</i> to
     * occurrences of class <i>newClass</i>. This
     * method is used to retarget accesses to one
     * class, seamlessly to another. The format
     * for the class name is slash (/) separated
     * so the class <tt>util.ClassFile</tt> would
     * be represented as <tt>util/ClassFile</tt>
     */
    public void mapClass(String oldClass,
                         String newClass) {
        for (int i = 0;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i].type ==
                    ConstantPoolInfo.CLASS) {
                String cname = getConstantPool()[i].arg1.strValue;
                if (cname.compareTo(oldClass) ==
                        0) {
                    getConstantPool()[i].arg1.strValue =
                            newClass;
                }
            }
        }
    }

    /**
     * Map occurences of package <i>oldPackage</i>
     * to package <i>newPackage</i>. The format
     * for the package name is slash (/) separated
     * so the package <tt>java.util</tt> would be
     * represented as <tt>java/util</tt>
     */
    public void mapPackage(String oldPackage,
                           String newPackage) {
        for (int i = 0;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i].type ==
                    ConstantPoolInfo.CLASS) {
                String cname = getConstantPool()[i].arg1.strValue;
                if (cname.startsWith(oldPackage)) {
                    getConstantPool()[i].arg1.strValue =
                            newPackage +
                            cname.substring(cname.lastIndexOf('/'));
                }
            }
        }
    }

    /**
     * Delete a named method from this class. This
     * method is used to excise specific methods
     * from the loaded class. The actual method
     * code remains, however the method signature
     * is deleted from the constant pool. If this
     * method is called by a class the exception
     * IncompatibleClassChangeException is
     * generated by the runtime.
     */
    public void deleteMethod(String name,
                             String signature) {
        for (int i = 0;
             i < getConstantPool().length;
             i++) {
            if (getConstantPool()[i].type ==
                    ConstantPoolInfo.CLASS) {
            }
        }
    }

    int getMagic() {
        return magic;
    }

    void setMagic(int magic) {
        this.magic = magic;
    }

    short getMajorVersion() {
        return majorVersion;
    }

    void setMajorVersion(short majorVersion) {
        this.majorVersion = majorVersion;
    }

    short getMinorVersion() {
        return minorVersion;
    }

    void setMinorVersion(short minorVersion) {
        this.minorVersion = minorVersion;
    }

    ConstantPoolInfo[] getConstantPool() {
        return constantPool;
    }

    void setConstantPool(ConstantPoolInfo[] constantPool) {
        this.constantPool = constantPool;
    }

    short getAccessFlags() {
        return accessFlags;
    }

    void setAccessFlags(short accessFlags) {
        this.accessFlags = accessFlags;
    }

    ConstantPoolInfo getThisClass() {
        return thisClass;
    }

    void setThisClass(ConstantPoolInfo thisClass) {
        this.thisClass = thisClass;
    }

    ConstantPoolInfo getSuperClass() {
        return superClass;
    }

    void setSuperClass(ConstantPoolInfo superClass) {
        this.superClass = superClass;
    }

    ConstantPoolInfo[] getInterfaces() {
        return interfaces;
    }

    void setInterfaces(ConstantPoolInfo[] interfaces) {
        this.interfaces = interfaces;
    }

    FieldInfo[] getFields() {
        return fields;
    }

    void setFields(FieldInfo[] fields) {
        this.fields = fields;
    }

    MethodInfo[] getMethods() {
        return methods;
    }

    void setMethods(MethodInfo[] methods) {
        this.methods = methods;
    }

    AttributeInfo[] getAttributes() {
        return attributes;
    }

    void setAttributes(AttributeInfo[] attributes) {
        this.attributes = attributes;
    }

    boolean isValidClass() {
        return isValidClass;
    }

    void setValidClass(boolean validClass) {
        isValidClass = validClass;
    }

    public static void testGetClassFiles() {
        ClassFile cf[] = getClassFiles();
        for (int i = 0; i < cf.length; i++)
            System.out.println(cf[i]);
    }

    public static ClassFile[] getClassFiles() {
        DirList dl = new DirList(Futil.getReadFileDir("select a start point"),
                new WildFilter(".class"));
        File f[] = dl.getFilesNotDirectories();
        Vector v = new Vector();
        for (int i = 0; i < f.length; i++)
            v.addElement(getClassFile(f[i]));
        ClassFile cf[] = new ClassFile[v.size()];
        v.copyInto(cf);
        return cf;
    }

    public static ClassFile getClassFile() {
        final FileInputStream fis =
                Futil.getFileInputStream("Select a classfile");
         if (fis == null) return null;
        return getClassFile(fis);
    }

    public static ClassFile getClassFile(File f) {
        ClassFile cf = null;
        try {
            cf = getClassFile(new FileInputStream(f));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return cf;
    }

    public static ClassFile getClassFile(FileInputStream fis) {
        ClassFile cf = null;
        try {
            cf = readClassFile(fis);
            fis.close();
        } catch (Exception e) {
            In.message(e);
        }
        return cf;
    }

    public static ClassFile readClassFile(InputStream fis) throws Exception {
        ClassFile cf = new ClassFile();
        if (!cf.read(fis)) {
            System.out.println("Couldn't load");
        }
        cf.display();
        return cf;
    }

    public boolean hasMainMethod() {
        MethodInfo mi[] = getMethods();
        Vector v = new Vector();
        if (mi == null) return false;
        for (int i = 0; i < mi.length; i++) {
            String ms = mi[i].toString();
            boolean b = ms.indexOf("public") >= 0;
            boolean b1 = ms.indexOf("static") >= 0;
            boolean b3 = ms.indexOf("main") >= 0;
            boolean b4 = ms.indexOf("java.lang.String ") >= 0;
            boolean b5 = ms.indexOf("void") >= 0;
            if (b && b1 && b3 && b4 && b5)
                v.addElement(mi[i]);
        }
        if (v.size() == 0) return false;
        return true;
    }

    public static void main(String[] args) {
        testGetClassFiles();
    }
}

