/*
 * Copyright (C) 1996 Emanuel Borsboom <manny@zerius.victoria.bc.ca>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package sound.zrs.wave;

import futils.Futil;
import gui.ClosableJFrame;
import gui.In;
import gui.run.RunButton;
import gui.run.RunMenu;
import gui.run.RunMenuItem;
import gui.run.RunScroll;
import sound.zrs.misc.Clipboard;
import sound.zrs.misc.ClipboardObserver;
import sound.zrs.misc.WindowRegistry;
import sound.zrs.misc.WindowRegistryAble;
import sound.zrs.ui.PropertiesObserver;

import javax.swing.*;
import javax.swing.border.BevelBorder;
import java.awt.*;
import java.io.*;
import java.net.URL;

public class WaveFrame extends ClosableJFrame
        implements  ClipboardObserver, WaveCanvasObserver,
                   PropertiesObserver, WindowRegistryAble {
    public static int CLOSE = 1;
    static int untitledCounter = 0;
    static SingleAudioPlayer singleAudioPlayer = new SingleAudioPlayer();
    WaveWindowObserver observer = null;
    WaveComponent waveComponent;
    String name;
    File file = null;
    RunScroll scrollbar;
    WavePropertiesDialog propertiesDialog;
    boolean opening;

    public WaveFrame() {
        initGUI();
        untitledCounter++;
        name = "Untitled " + untitledCounter;
        setTitle();
        pack();
    }

    public void setObserver(WaveWindowObserver observer) {
        this.observer = observer;
    }

    public WaveFrame(Wave wave, String name) {
        initGUI();
        untitledCounter++;
        if (name == null)
            this.name = "Untitled " + untitledCounter;
        else
            this.name = name;
        setTitle();
        waveComponent.setWave(wave);
        pack();
        updateScrollbar();
    }

    public WaveFrame(Wave wave) {
        this(wave, null);
    }

    public WaveFrame(File file) throws IOException, WaveFormatException {
        initGUI();
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(file);
            load(fis, file, file.getName());
        } finally {
            if (fis != null) fis.close();
        }
        this.file = file;
        pack();
        updateScrollbar();
    }

    void initGUI() {
        initMenuBar();
        JPanel jp = getButtonPanel();
        waveComponent = new WaveComponent();
        waveComponent.allowSelection(true);
        waveComponent.setObserver(this);
        scrollbar = new RunScroll(Scrollbar.HORIZONTAL) {
            public void run() {
                int[] view = waveComponent.getView();
                int l = view[1] - view[0];
                boolean flag = true;
                int v = this.getValue();
                view[0] = v;
                if (flag) {
                    view[1] = view[0] + l;
                    waveComponent.setView(view[0], view[1]);
                    updateScrollbar();
                }
            }
        };
        Container c = getContentPane();
        c.add("North", jp);
        c.add("Center", waveComponent);
        c.add("South", scrollbar);
        WaveComponent.clipboard.addObserver(this);
    }

    private JPanel getButtonPanel() {
        JPanel jp = new JPanel();
        jp.setLayout(new FlowLayout());
        jp.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        jp.add(new RunButton("close") {
            public void run() {
                quit();
            }
        });

        jp.add(new RunButton("Play") {
            public void run() {
                singleAudioPlayer.play(waveComponent.getWave());
            }
        });
        jp.add(new RunButton("Loop") {
            public void run() {
                singleAudioPlayer.loop(waveComponent.getWave());
            }
        });
        jp.add(new RunButton("Stop") {
            public void run() {
                singleAudioPlayer.stop();
            }
        });
        jp.add(new RunButton("Cut") {
            public void run() {
                waveComponent.cutSelection();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("Copy") {
            public void run() {
                waveComponent.copySelection();
            }
        });
        jp.add(new RunButton("Paste") {
            public void run() {
                waveComponent.pasteBeforeSelection();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("Clear") {
            public void run() {
                waveComponent.clearSelection();
            }
        });
        jp.add(new RunButton("Z.In") {
            public void run() {
                waveComponent.zoomIn();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("Z.Out") {
            public void run() {
                waveComponent.zoomOut();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("All") {
            public void run() {
                waveComponent.viewAll();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("Sel") {
            public void run() {
                waveComponent.viewSelection();
                updateScrollbar();
            }
        });
        jp.add(new RunButton("1-1") {
            public void run() {
                waveComponent.viewOneToOne();
                updateScrollbar();
            }
        });
        return jp;
    }

    private void initMenuBar() {
        JMenuBar menuBar = new JMenuBar();
        RunMenu m = new RunMenu("File");
        m.add(new RunMenuItem("new") {
            public void run() {
                new WaveFrame().showAndRegister();
            }
        });
        m.add(new RunMenuItem("Open from file") {
            public void run() {
                openFromFile();
            }
        });
        m.add(new RunMenuItem("Open from url") {
            public void run() {
                openFromURL();
            }
        });
        m.add(new RunMenuItem("Save as...") {
            public void run() {
               saveAs();
            }
        });
        m.add(new RunMenuItem("Save") {
            public void run() {
               save(file);
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("close") {
            public void run() {
                WaveFrame.this.setVisible(false);
            }
        });
        menuBar.add(m);
        m = new RunMenu("Edit");
        m.add(new RunMenuItem("select all") {
            public void run() {
                waveComponent.selectAll();
            }
        });
        m.add("Select View");
        m.add(new RunMenuItem("Select None") {
            public void run() {
                waveComponent.selectNone();
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("Cut") {
            public void run() {
            }
        });
        m.add(new JMenuItem("Copy"));
        m.add(new JMenuItem("Paste"));
        m.add(new JMenuItem("Clear"));
        m.addSeparator();
        m.add(new RunMenuItem("Amplify") {
            public void run() {
                waveComponent.amplify(
                        In.getFloat("enter amplification 0..1"));
            }
        });
        m.add(new RunMenuItem("Reverse") {
            public void run() {
                waveComponent.reverse();
            }
        });
        menuBar.add(m);
        m.add(new RunMenuItem("View Selection") {
            public void run() {
                waveComponent.selectView();
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("Spectrum") {
            public void run() {
                new SpectrumWindow(waveComponent.getSelectedWave(), name).setVisible(true);
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("Properties ...") {
            public void run() {
                properitesProcess();
            }
        });
        menuBar.add(m);
        this.setJMenuBar(menuBar);
    }

    private void properitesProcess() {
        if (waveComponent.wave != null) {
            if (propertiesDialog == null)
                propertiesDialog = new WavePropertiesDialog(this);
            propertiesDialog.showAndPack();
        }
    }

    public void showAndRegister() {
        setVisible(true);
        WindowRegistry.add(this);
    }

    void setTitle() {
        setTitle("Wave - " + name);
    }

    public String getWindowIdentifier() {
        return "Wave - " + name;
    }

    public void load(InputStream is, File file, String name)
            throws IOException, WaveFormatException {
        WindowRegistry.saveCursors();
        WindowRegistry.setAllCursors(Cursor.WAIT_CURSOR);
        try {
            waveComponent.load(is);
        } finally {
            WindowRegistry.restoreCursors();
        }
        this.name = name;
        this.file = file;
        setTitle();
        hideProperties();
    }

    public String getName() {
        return name;
    }

    public File getFile() {
        return file;
    }

    public Wave getWave() {
        return waveComponent.getWave();
    }

    void quit() {
        setVisible(false);
        dispose();
        if (observer != null)
            observer.waveWindowEvent(this, CLOSE);
        WindowRegistry.remove(this);
    }

    public void clipboardEvent(Clipboard clipboard, int action) {
        if (action == Clipboard.PUT) {
            if (waveComponent.getWave() == null ||
                    waveComponent.getSelection() != null) {
            }
        }
    }

    public void waveCanvasEvent(WaveComponent wc, int action) {
    }

    public void hideProperties() {
        if (propertiesDialog != null) {
            propertiesDialog.setVisible(false);
            propertiesDialog.dispose();
            propertiesDialog = null;
        }
    }

    void updateScrollbar() {
        if (waveComponent.getWave() == null)
            scrollbar.setEnabled(false);
        else {
            int[] view = waveComponent.getView();
            int visible = view[1] - view[0] + 1;
            int minimum = 0;
            int maximum = waveComponent.getWave().length() -
                    (view[1] - view[0] + 1);
            int pageIncrement = (view[1] - view[0] + 1) * 3 / 4;
            int lineIncrement = (view[1] - view[0] + 1) / 16;
            scrollbar.setValues(view[0], visible, minimum, maximum);
            scrollbar.setBlockIncrement(pageIncrement < 1 ?
                    1 :
                    pageIncrement);
            scrollbar.setUnitIncrement(lineIncrement < 1 ?
                    1 :
                    lineIncrement);
            scrollbar.setEnabled(true);
        }
    }

    private void saveAs(){
        file = Futil.getWriteFile("select and output au file");
        save(file);
    }
    private void save(File wf) {
        WindowRegistry.saveCursors();
        WindowRegistry.setAllCursors(Cursor.WAIT_CURSOR);
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(wf);
            waveComponent.getWave().write(fos);
            file = wf;
            name = file.getName();
            setTitle();
        } catch (Exception e) {
            In.message(e);
        } finally {
            try {
                if (fos != null) fos.close();
            } catch (IOException e) {
            }
        }
        WindowRegistry.restoreCursors();
    }
    private void openFromFile(){
        if (file == null){
            file = Futil.getReadFile("enter au file");

        }
        openFromFile(file);
    }
    private void openFromFile(File rf) {
        try {
            new WaveFrame(rf).showAndRegister();

            updateScrollbar();
        } catch (WaveFormatException e) {
            In.message(e);

        } catch (IOException e) {
            In.message(e);

        }
    }

    private void openFromURL() {
        try {
            URL url = In.getUrl("enter and audio url");
            if (url ==  null) return;
            Wave w = new Wave(url.openStream());
            new WaveFrame(w, url.toString()).showAndRegister();
        } catch (IOException e) {
            In.message(e);

        } catch (WaveFormatException e) {
            In.message(e);

        }
    }



    public static void main(String[] args) {
        WindowRegistry.setExitWhenFinished(true);
        if (args != null && args.length > 0) {
            for (int i = 0; i < args.length; i++)
                try {
                    new WaveFrame(new File(args[i])).showAndRegister();
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (WaveFormatException e) {
                    e.printStackTrace();
                }
        } else
            new WaveFrame().showAndRegister();
    }
}
