/*
 * Copyright (C) 1996 Emanuel Borsboom <manny@zerius.victoria.bc.ca>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package sound.zrs.synth;

import futils.Futil;
import gui.ClosableJFrame;
import gui.In;
import gui.run.RunButton;
import gui.run.RunMenuItem;
import sound.zrs.misc.WindowRegistry;
import sound.zrs.misc.WindowRegistryAble;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import java.awt.*;
import java.io.*;
import java.net.URL;

public class SynthFrame extends ClosableJFrame
        implements WindowRegistryAble {


    static int untitledCounter = 0;


    SynthComponent synthComponent = new SynthComponent(this);


    String name;
    File file = null;
    URL url = null;

    public SynthFrame() {
        initGUI();
        file = null;
        untitledCounter++;
        name = "Untitled " + untitledCounter;
        setTitle();
        pack();
    }

    public SynthFrame(InputStream is, String name)
            throws IOException, SynthIfException, FileFormatException {
        initGUI();
        synthComponent.load(is);
        this.name = name;
        setTitle();
        pack();
    }

    public SynthFrame(File file)
            throws IOException, SynthIfException, FileFormatException {
        initGUI();
        FileInputStream fin = null;
        try {
            fin = new FileInputStream(file);
            synthComponent.load(fin);
        } finally {
            try {
                if (fin != null) fin.close();
            } catch (IOException e) {
                In.message(e);
            }
        }
        this.file = file;
        name = file.getName();
        setTitle();
        pack();
    }

    public SynthFrame(URL url)
            throws IOException, SynthIfException, FileFormatException {
        initGUI();
        InputStream i = null;
        try {
            i = url.openStream();
            synthComponent.load(i);
        } finally {
            try {
                if (i != null) i.close();
            } catch (IOException e) {
                In.message(e);
            }
        }
        this.url = url;
        name = url.toString();
        setTitle();

        pack();
    }

    private void initGUI() {
        Container c = getContentPane();
        initMenuBar();
        c.add("North", getButtonPanel());
        c.add("Center", synthComponent);
    }

    private void initMenuBar() {
        JMenuBar menuBar = new JMenuBar();
        JMenu m = new JMenu("File");
        m.add(new RunMenuItem("new") {
            public void run() {
                new SynthFrame().showAndRegister();
            }
        });
        m.add(new RunMenuItem("open url") {
            public void run() {
                try {
                    final URL url = In.getUrl("select a synth url");
                    if (url == null) return;
                    openUrl(url);
                } catch (SynthIfException e) {
                    In.message(e);

                } catch (IOException e) {
                    In.message(e);

                }
            }
        });


        m.add(new RunMenuItem("save") {
            public void run() {
                save(file);
            }
        });
        m.add(new RunMenuItem("Save As ...") {
            public void run() {
                saveAs();
            }
        });


        m.addSeparator();
        m.add(new RunMenuItem("close") {
            public void run() {
                quit();
            }
        });


        menuBar.add(m);
        menuBar.add(synthComponent.getSynthesizerMenu());
        m = new JMenu("Edit");
        m.add(new RunMenuItem("select all") {
            public void run() {
                synthComponent.selectAll();
            }
        });
        m.add(new RunMenuItem("Unselect all") {
            public void run() {
                synthComponent.unselectAll();
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("Delete") {
            public void run() {
                synthComponent.deleteSelected();
                synthComponent.repaint();
            }
        });
        m.addSeparator();
        m.add(new RunMenuItem("Properties...") {
            public void run() {
                synthComponent.showProperties();
            }
        });

        menuBar.add(m);
        m = synthComponent.getGeneratorMenu();

        menuBar.add(m);

        m = new JMenu("Help");


        m.add(new RunMenuItem("Help") {
            public void run() {
                In.message("zrs modified by DocJava, Inc.");
            }
        });
        menuBar.add(m);

        setJMenuBar(menuBar);
    }

    private JPanel getButtonPanel() {
        JPanel jp = new JPanel();
        jp.setBorder(new EtchedBorder());
        jp.setLayout(new FlowLayout());
        jp.add(synthComponent.getSynthesizeButton());
        jp.add(synthComponent.getDeleteButton());
        jp.add(synthComponent.getPropertiesButton());
        jp.add(new RunButton("quit") {
            public void run() {
                quit();
            }
        });
        jp.add(new RunButton("open file") {
            public void run() {
                file = Futil.getReadFile("select a synth file");
                if (file == null) return;
                try {
                    new SynthFrame(file).showAndRegister();
                } catch (IOException e) {
                    e.printStackTrace();

                } catch (SynthIfException e) {
                    e.printStackTrace();

                }
            }
        });
        jp.add(new RunButton("save") {
            public void run() {
                if (file == null){
                    saveAs();
                    return;
                }
                save(file);
            }
        });
        return jp;
    }

    void setTitle() {
        setTitle("Synthesizer - " + name);
    }

    public String getWindowIdentifier() {
        return "Synthesizer - " + name;
    }

    public void showAndRegister() {
        setVisible(true);
        WindowRegistry.add(this);
    }

    private void quit() {
        setVisible(false);
        dispose();
        WindowRegistry.remove(this);
    }

    private void openUrl(URL s) throws SynthIfException, IOException {
        new SynthFrame(s).showAndRegister();
    }

    private void save(File file) {
        WindowRegistry.saveCursors();
        OutputStream out = null;
        try {
            out = new FileOutputStream(file);
            synthComponent.save(out);
        } catch (Exception e) {
            In.message(e);
        } finally {
            try {
                if (out != null) out.close();
            } catch (IOException e) {
                In.message(e);
            }
            WindowRegistry.restoreCursors();
        }
    }

    private void saveAs() {
        file = Futil.getWriteFile("select output synth name");
        save(file);
    }


}
