/*
 * Copyright (C) 1996 Emanuel Borsboom <manny@zerius.victoria.bc.ca>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package sound.zrs.synth;

import gui.In;
import gui.run.RunButton;
import gui.run.RunMenuItem;
import sound.zrs.synthgen.Synthesizer;
import sound.zrs.ui.*;

import javax.swing.*;
import javax.swing.border.BevelBorder;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.*;
import java.util.Enumeration;
import java.util.Vector;

class SynthComponent extends JComponent
        implements PropertiesObserver, MouseMotionListener, MouseListener {

    SynthFrame pa;

    static final Color BACKGROUND_COLOR = Color.white;
    static final int dragThreshold = 1;

    int samplingRate = 8000;
    double time = 1.0;

    Dimension prefSize = new Dimension(384, 512);

    Vector generators = new Vector();
    Vector connections = new Vector();

    Point dragStart;
    boolean dragged;

    boolean newGeneratorFlag = false;
    GeneratorBox newGeneratorGen;

    boolean moveOriginFlag = false;

    boolean selectBoxFlag = false;
    boolean selectBoxAddFlag;

    boolean moveGeneratorFlag = false;
    GeneratorBox moveGeneratorGens[];
    GeneratorBox moveGeneratorGen;
    boolean moveGeneratorSelectedFlag;

    boolean selectDestFlag = false;
    GeneratorBox selectDestSource;

    RubberBand rubbers[];

    SynthPropertiesDialog propertiesDialogDialog;

    SynthComponent(SynthFrame pa) {
        this.pa = pa;
        setBackground(BACKGROUND_COLOR);
        addMouseMotionListener(this);
        addMouseListener(this);
        setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
    }

    void synthesize() {
        int sampleCount = (int) (samplingRate * time);
        Synthesizer synth = new Synthesizer(samplingRate);
        GeneratorBox output = null;

        try {

            Enumeration e = generators.elements();
            while (e.hasMoreElements()) {
                GeneratorBox gen = (GeneratorBox) e.nextElement();
                gen.newGenerator(synth);
                if (gen instanceof OutputBox)
                    output = gen;
            }

            if (output == null)  {
                In.message("No output specified.");
                return;
            }

            e = generators.elements();
            while (e.hasMoreElements()) {
                GeneratorBox gen = (GeneratorBox) e.nextElement();
                gen.connectGenerator(synth);
            }

            synth.setOutput(
                    ((Input) output.inputs.elementAt(0)).getGenerator());

            new SynthesizeThread(pa, synth, samplingRate, sampleCount).start();
        } catch (Exception e) {
            In.message(e);
        }
    }

    public Dimension getMinimumSize() {
        return prefSize;
    }

    public Dimension getPreferredSize() {
        return getMinimumSize();
    }

    GeneratorBox inGenerator(int x, int y) {
        for (int i = generators.size() - 1; i >= 0; i--) {
            if (((GeneratorBox) generators.elementAt(i)).inside(x, y)) {
                return (GeneratorBox) generators.elementAt(i);
            }
        }
        return null;
    }

    Connection inConnection(int x, int y) {
        for (int i = connections.size() - 1; i >= 0; i--) {
            if (((Connection) connections.elementAt(i)).inside(x, y)) {
                return (Connection) connections.elementAt(i);
            }
        }
        return null;
    }

    void _selectAll() {
        Enumeration e = generators.elements();
        while (e.hasMoreElements())
            ((GeneratorBox) e.nextElement()).select();
        e = connections.elements();
        while (e.hasMoreElements())
            ((Connection) e.nextElement()).select();
    }

    void selectAll() {
        int count = countSelected();
        _selectAll();
        if (countSelected() != count)
            repaint();
    }

    void _unselectAll() {
        Enumeration e = generators.elements();
        while (e.hasMoreElements())
            ((GeneratorBox) e.nextElement()).unselect();
        e = connections.elements();
        while (e.hasMoreElements())
            ((Connection) e.nextElement()).unselect();
    }

    void unselectAll() {
        int count = countSelected();
        _unselectAll();
        if (countSelected() != count)
            repaint();
    }

    void selectInside(int x1, int y1, int x2, int y2) {
        int t;
        if (x2 < x1) {
            t = x1;
            x1 = x2;
            x2 = t;
        }
        if (y2 < y1) {
            t = y1;
            y1 = y2;
            y2 = t;
        }

        Rectangle r = new Rectangle(x1, y1,
                x2 - x1 + 1, y2 - y1 + 1);

        Enumeration genE = generators.elements();
        while (genE.hasMoreElements()) {
            GeneratorBox gen = (GeneratorBox) genE.nextElement();
            if (r.intersects(gen.getRect()))
                gen.select();
        }

        Enumeration conE = connections.elements();
        while (conE.hasMoreElements()) {
            Connection con = (Connection) conE.nextElement();
            if (r.intersects(con.getRect()))
                con.select();
        }

        repaint();
    }

    int countSelectedGenerators() {
        int count = 0;
        Enumeration e = generators.elements();
        while (e.hasMoreElements())
            if (((GeneratorBox) e.nextElement()).isSelected()) count++;
        return count;
    }

    int countSelectedConnections() {
        int count = 0;
        Enumeration e = connections.elements();
        while (e.hasMoreElements())
            if (((Connection) e.nextElement()).isSelected()) count++;
        return count;
    }

    int countSelected() {
        return countSelectedGenerators() + countSelectedConnections();
    }

    void selectOnly(GeneratorBox g) {
        if (countSelected() != 1 || !g.isSelected()) {
            _unselectAll();
            g.select();
            repaint();
        }
    }

    void selectOnly(Connection c) {
        if (countSelected() != 1 || !c.isSelected()) {
            _unselectAll();
            c.select();
            repaint();
        }
    }

    void deleteSelected() {
        int conNo;
        int genNo;
        int inNo;
        Connection con;
        GeneratorBox gen;
        Input in;

        // Remove selected connections
        conNo = 0;
        while (conNo < connections.size()) {
            con = (Connection) connections.elementAt(conNo);
            if (con.isSelected())
                connections.removeElement(con);
            else
                conNo++;
        }

        // Remove selected generators, any connections with them
        genNo = 0;
        while (genNo < generators.size()) {

            gen = (GeneratorBox) generators.elementAt(genNo);
            if (gen.isSelected()) {

                // Check for connections from this generator, and remove them
                conNo = 0;
                while (conNo < connections.size()) {
                    con = (Connection) connections.elementAt(conNo);
                    if (gen == con.source)
                        connections.removeElement(con);
                    else
                        conNo++;
                }

                // Check for connections to an input in this generator, and remove them
                conNo = 0;
                while (conNo < connections.size()) {
                    con = (Connection) connections.elementAt(conNo);
                    inNo = 0;
                    while (inNo < gen.inputs.size()) {
                        in = (Input) gen.inputs.elementAt(inNo);
                        if (in == con.dest) {
                            connections.removeElement(con);
                            break;
                        }
                        inNo++;
                    }
                    if (inNo == gen.inputs.size())
                        conNo++;
                }

                // finally, remove the generator
                gen.dispose();
                generators.removeElement(gen);

            } else
                genNo++;
        }

        repaint();

    }

    void showProperties() {
        if (countSelectedGenerators() == 0) {
            if (propertiesDialogDialog == null)
                propertiesDialogDialog =
                        new SynthPropertiesDialog( this);
            propertiesDialogDialog.showAndPack();
        } else {
            Enumeration e = generators.elements();
            GeneratorBox g;
            while (e.hasMoreElements()) {
                if ((g = (GeneratorBox) e.nextElement()).isSelected())
                    g.showProperties();
            }
        }
    }

    public void save(OutputStream os) throws IOException, SynthIfException {
        DataOutputStream dos = new DataOutputStream(os);
        dos.writeBytes("ZSyn");	// magic
        dos.writeInt(1);		// file format version

        dos.writeInt(getSize().width);
        dos.writeInt(getSize().height);

        dos.writeInt(samplingRate);
        dos.writeDouble(time);

        dos.writeInt(generators.size() + connections.size());

        Enumeration e = generators.elements();
        while (e.hasMoreElements()) {
            GeneratorBox gen = (GeneratorBox) e.nextElement();
            gen.write(dos);
        }

        e = connections.elements();
        while (e.hasMoreElements()) {
            Connection c = (Connection) e.nextElement();
            int sourceNo = -1, destGenNo = -1, destInNo = -1;
            for (int genNo = 0; genNo < generators.size(); genNo++) {
                GeneratorBox gen = (GeneratorBox) generators.elementAt(
                        genNo);
                if (c.source == gen)
                    sourceNo = genNo;
                for (int inNo = 0; inNo < gen.inputs.size(); inNo++) {
                    Input in = (Input) gen.inputs.elementAt(inNo);
                    if (c.dest == in) {
                        destGenNo = genNo;
                        destInNo = inNo;
                    }
                }
            }
            if (sourceNo < 0 || destGenNo < 0 || destInNo < 0)
                throw new SynthIfException(
                        "Invalid internal data structure");
            dos.writeBytes(Connection.IDENT);
            dos.writeInt(sourceNo);
            dos.writeInt(destGenNo);
            dos.writeInt(destInNo);
            dos.writeBoolean(c.isSelected());
        }
    }

    String readAsciiString(DataInputStream in, int n) throws IOException,
                                                             EOFException {
        byte bytes[] = new byte[n];
        in.readFully(bytes);
        return new String(bytes);
    }

    void load(InputStream in)
            throws IOException, EOFException, FileFormatException {
        DataInputStream din = new DataInputStream(in);
        GeneratorBox gen;
        String s;
        int i;

        try {
            if (!readAsciiString(din, 4).equals("ZSyn"))   {
                In.message("Incorrect magic number, bad file format");
                return;
            }
            if ((i = din.readInt()) != 1)
                throw new FileFormatException(
                        "Unknown version number " + i);
            i = din.readInt();
            prefSize = new Dimension(i, din.readInt());

            samplingRate = din.readInt();
            time = din.readDouble();

            generators = new Vector();
            connections = new Vector();

            int n = din.readInt();

            for (i = 0; i < n; i++) {
                s = readAsciiString(din, 4);

                gen = null;
                if (s.equals(AdderBox.IDENT)) {
                    gen = new AdderBox(this);
                } else if (s.equals(DividerBox.IDENT)) {
                    gen = new DividerBox(this);
                } else if (s.equals(MultiplierBox.IDENT)) {
                    gen = new MultiplierBox(this);
                } else if (s.equals(OscillatorBox.IDENT)) {
                    gen = new OscillatorBox(this);
                } else if (s.equals(OutputBox.IDENT)) {
                    gen = new OutputBox(this);
                } else if (s.equals(ConstantBox.IDENT)) {
                    gen = new ConstantBox(this);
                } else if (s.equals(EnvelopeBox.IDENT)) {
                    gen = new EnvelopeBox(this);
                } else if (s.equals(DelayBox.IDENT)) {
                    gen = new DelayBox(this);
                } else if (s.equals(AbsBox.IDENT)) {
                    gen = new AbsBox(this);
                } else if (s.equals(DistortBox.IDENT)) {
                    gen = new DistortBox(this);
                } else if (s.equals(FilterBox.IDENT)) {
                    gen = new FilterBox(this);
                } else if (s.equals(Connection.IDENT)) {
                    int sourceNo = din.readInt();
                    int destGenNo = din.readInt();
                    int destInNo = din.readInt();
                    GeneratorBox source = (GeneratorBox) generators.elementAt(
                            sourceNo);
                    Input dest = (Input) ((GeneratorBox) generators.elementAt(
                            destGenNo))
                            .inputs.elementAt(destInNo);
                    Connection con = new Connection(source, dest);
                    if (din.readBoolean()) con.select();

                    connections.addElement(con);
                } else
                    throw new FileFormatException("Unknown object: " + s);

                if (gen != null) {
                    gen.read(din);
                    generators.addElement(gen);
                }
            }
        } catch (EOFException e) {
            throw new FileFormatException("Premature End of File");
        }
    }

    public void hideProperties() {
        if (propertiesDialogDialog != null) {
            propertiesDialogDialog.setVisible(false);
            propertiesDialogDialog.dispose();
            propertiesDialogDialog = null;
        }
    }

    public RunButton getPropertiesButton() {
        return new RunButton("Properties") {
            public void run() {
                showProperties();
            }
        };
    }

    public JMenu getSynthesizerMenu() {
        JMenu jm = new JMenu();
        jm.add(new RunMenuItem("synthesize") {
            public void run() {
                synthesize();
            }
        });
        jm.add(new RunMenuItem("Synthesizer Properties ...") {
            public void run() {
                synthesizerProperties();
            }
        });
        return jm;
    }

    private void synthesizerProperties() {
        if (propertiesDialogDialog == null)
            propertiesDialogDialog =
                    new SynthPropertiesDialog( this);
        propertiesDialogDialog.showAndPack();
    }


    public RunButton getSynthesizeButton() {
        return new RunButton("Synthesize") {
            public void run() {
                synthesize();
            }
        };
    }

    public void paint(Graphics g) {
        for (int i = 0; i < generators.size(); i++) {
            ((GeneratorBox) generators.elementAt(i)).draw(g);
        }
        for (int i = 0; i < connections.size(); i++) {
            ((Connection) connections.elementAt(i)).draw(g);
        }
        super.paint(g);
    }

   

    public void mouseDragged(MouseEvent event) {
        int x = event.getX();
        int y = event.getY();
        if (dragStart==null) dragStart = event.getPoint();
        // only drag if mouse has moved a certain number of pixels
        // (mainly to work around another win32 awt bug)
        if (!dragged &&
                (x < dragStart.x - dragThreshold ||
                x > dragStart.x + dragThreshold ||
                y < dragStart.y - dragThreshold ||
                y > dragStart.y + dragThreshold)) {
            dragged = true;
        }
        if (!dragged)
            return;

        if (rubbers != null) {
            Graphics g = getGraphics();
            for (int i = 0; i < rubbers.length; i++)
                rubbers[i].move(g, x, y);
            g.dispose();
            return;
        }
    }

    public void mouseMoved(MouseEvent e) {
    }

    public void mouseClicked(MouseEvent e) {
    }

    public void mousePressed(MouseEvent event) {
        int x = event.getX();
        int y = event.getY();
        dragStart = new Point(x, y);
        dragged = false;

        if (newGeneratorFlag) {

            newGeneratorGen.move(x, y);
            generators.addElement(newGeneratorGen);
            selectOnly(newGeneratorGen);
            newGeneratorGen.showProperties();

            newGeneratorFlag = false;
            newGeneratorGen = null;
            pa.synthComponent.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));

        } else if (moveOriginFlag) {
        } else if (selectBoxFlag) {
        } else if (moveGeneratorFlag) {
        } else if ((event.getModifiers() & Event.META_MASK) != 0) {
            rubbers = new RubberBand[1];
            rubbers[0] = new LineRubberBand(x, y);
            moveOriginFlag = true;
        } else {
            Connection con;
            GeneratorBox gen;
            if ((con = inConnection(x, y)) != null) {
                if ((event.getModifiers() & Event.CTRL_MASK) != 0 ||
                        (event.getModifiers() & Event.ALT_MASK) != 0) {
                    if (con.isSelected())
                        con.unselect();
                    else
                        con.select();
                    repaint();
                } else
                    selectOnly(con);
            } else if ((gen = inGenerator(x, y)) != null) {
                if ((event.getModifiers() & Event.CTRL_MASK) != 0 ||
                        (event.getModifiers() & Event.ALT_MASK) != 0) {
                    if (gen.isSelected())
                        gen.unselect();
                    else
                        gen.select();
                    repaint();
                } else if (event.getClickCount() == 2) {
                    selectOnly(gen);
                    showProperties();
                } else {

                    if (gen.inBorder(x, y)) {
                        if (!gen.isSelected()) {
                            _unselectAll();
                            gen.select();
                            moveGeneratorSelectedFlag = true;
                        } else
                            moveGeneratorSelectedFlag = false;
                        rubbers =
                                new RubberBand[countSelectedGenerators()];
                        moveGeneratorGens =
                                new GeneratorBox[countSelectedGenerators()];
                        Enumeration genE = generators.elements();
                        int i = 0;
                        while (genE.hasMoreElements()) {
                            GeneratorBox g = (GeneratorBox) genE.nextElement();
                            if (g.isSelected()) {
                                rubbers[i] = new BoxRubberBand(
                                        g.getSize(),
                                        x - g.getPosition().x,
                                        y - g.getPosition().y);
                                moveGeneratorGens[i] = g;
                                i++;
                            }
                        }
                        moveGeneratorFlag = true;
                        moveGeneratorGen = gen;
                        return;
                    } else {
                        rubbers = new RubberBand[1];
                        rubbers[0] = new LineRubberBand(gen.getCenter());
                        selectDestFlag = true;
                        selectDestSource = gen;
                        return;
                    }
                }
            } else {
                if (event.getClickCount() == 2) {
                    unselectAll();
                    showProperties();
                } else {
                    rubbers = new RubberBand[1];
                    rubbers[0] = new BoxStretchRubberBand(x, y);

                    selectBoxFlag = true;
                }
            }
        }
    }

    public void mouseReleased(MouseEvent event) {
        int x = event.getX();
        int y = event.getY();
        RubberBand rubbers[] = this.rubbers;

        if (rubbers != null) {
            Graphics g = getGraphics();
            for (int i = 0; i < rubbers.length; i++)
                rubbers[i].erase(g);
            g.dispose();
            this.rubbers = null;
        }

        if (newGeneratorFlag) {
        } else if (moveOriginFlag) {
            if (dragged) {
                Point p = rubbers[0].getStart();
                for (int i = 0; i < generators.size(); i++) {
                    ((GeneratorBox) generators.elementAt(i)).translate(x -
                            p.x, y - p.y);
                }

                repaint();

            }

            moveOriginFlag = false;
            return;
        } else if (selectBoxFlag) {
            if (dragged) {
                if (!selectBoxAddFlag) _unselectAll();
                selectInside(rubbers[0].getStart().x,
                        rubbers[0].getStart().y,
                        x,
                        y);
            } else if (!selectBoxAddFlag)
                unselectAll();
            selectBoxFlag = false;
        } else if (moveGeneratorFlag) {
            if (dragged) {
                for (int i = 0; i < moveGeneratorGens.length; i++) {
                    Point p = ((BoxRubberBand) rubbers[i]).getOffset();
                    moveGeneratorGens[i].move(x - p.x, y - p.y);
                }

                repaint();
            } else {
                if (moveGeneratorSelectedFlag)
                    repaint();
                else
                    selectOnly(moveGeneratorGen);
            }
            moveGeneratorGen = null;
            moveGeneratorGens = null;
            moveGeneratorFlag = false;
            return;
        } else if (selectDestFlag) {
            if (dragged) {
                GeneratorBox gen = inGenerator(x, y);
                Input in;
                if (gen != null &&
                        gen != selectDestSource &&

                        (in = gen.inInput(x, y)) != null) {
                    boolean allowConnection = true;
                    if (!in.allowMultipleConnections) {
                        Enumeration e = connections.elements();

                        while (e.hasMoreElements())
                            if (((Connection) e.nextElement()).dest == in)
                                allowConnection = false;
                    }
                    if (!allowConnection)
                        In.message(
                                "This input already has a connection."
                                );
                    else {
                        Connection c = new Connection(selectDestSource,
                                in);
                        connections.addElement(c);
                        selectOnly(c);

                    }
                } else
                    In.message("Bad destination."
                           );
            } else
                selectOnly(selectDestSource);
            selectDestFlag = false;
            selectDestSource = null;
            return;
        }


    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public RunButton getDeleteButton() {
        return new RunButton("delete") {
            public void run() {
                deleteSelected();
                repaint();
            }
        };
    }

    /**
     * if (arg.equals("Constant")) {
     * <p/>
     * } else
     *
     * @return
     */
    public JMenu getGeneratorMenu() {
        JMenu m = new JMenu("Generators", true);
        m.add(new RunMenuItem("Constant") {
            public void run() {
                newGeneratorGen = new ConstantBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("oscillator") {
            public void run() {
                newGeneratorGen = new OscillatorBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("envelope") {
            public void run() {
                newGeneratorGen = new EnvelopeBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Adder") {
            public void run() {
                newGeneratorGen = new AdderBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Multiplier") {
            public void run() {
                newGeneratorGen = new MultiplierBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Divider") {
            public void run() {
                newGeneratorGen = new DividerBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Abs") {
            public void run() {
                newGeneratorGen = new AbsBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Distort") {
            public void run() {
                newGeneratorGen = new DistortBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Delay") {
            public void run() {
                newGeneratorGen = new DelayBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Filter") {
            public void run() {
                newGeneratorGen = new FilterBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        m.add(new RunMenuItem("Output") {
            public void run() {
                newGeneratorGen = new OutputBox(SynthComponent.this);
                newGeneratorFlag = true;
            }
        });
        return m;
    }

}
