package sound.recorder;

import sound.SoundUtils;

import javax.sound.sampled.*;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

/**
 * Reads data from the input channel and writes to the output stream
 */
public class Capture implements Runnable {

    TargetDataLine line;
    Thread thread;
    private CapturePlaybackPanel capturePlaybackPanel;

    public Capture(CapturePlaybackPanel capturePlaybackPanel) {
        this.capturePlaybackPanel = capturePlaybackPanel;
    }

    public void start() {
        capturePlaybackPanel.errStr = null;
        thread = new Thread(this);
        thread.setName("Capture");
        thread.start();
    }

    public void stop() {
        thread = null;
    }

    private void shutDown(String message) {
        if ((capturePlaybackPanel.errStr = message) != null && thread != null) {
            thread = null;
            capturePlaybackPanel.samplingGraph.stop();
            capturePlaybackPanel.loadB.setEnabled(true);
            capturePlaybackPanel.playB.setEnabled(true);
            capturePlaybackPanel.pausB.setEnabled(false);
            capturePlaybackPanel.auB.setEnabled(true);
            capturePlaybackPanel.aiffB.setEnabled(true);
            capturePlaybackPanel.waveB.setEnabled(true);
            capturePlaybackPanel.captB.setText("Record");
            System.err.println(capturePlaybackPanel.errStr);
            capturePlaybackPanel.samplingGraph.repaint();
        }
    }

    public void run() {

        capturePlaybackPanel.duration = 0;
        capturePlaybackPanel.audioInputStream = null;

        // define the required attributes for our line,
        // and make sure a compatible line is supported.

        AudioFormat format = capturePlaybackPanel.formatControls.getFormat();
        DataLine.Info info = new DataLine.Info(TargetDataLine.class,
                format);

        if (!AudioSystem.isLineSupported(info)) {
            shutDown("Line matching " + info + " not supported.");
            return;
        }

        // get and open the target data line for capture.

        try {
            line = (TargetDataLine) AudioSystem.getLine(info);
            line.open(format, line.getBufferSize());
        } catch (LineUnavailableException ex) {
            shutDown("Unable to open the line: " + ex);
            return;
        } catch (SecurityException ex) {
            shutDown(ex.toString());
            SoundUtils.showInfoDialog();
            return;
        } catch (Exception ex) {
            shutDown(ex.toString());
            return;
        }

        // play back the captured audio data
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int frameSizeInBytes = format.getFrameSize();
        int bufferLengthInFrames = line.getBufferSize() / 8;
        int bufferLengthInBytes = bufferLengthInFrames * frameSizeInBytes;
        byte[] data = new byte[bufferLengthInBytes];
        int numBytesRead;

        line.start();

        while (thread != null) {
            if ((numBytesRead = line.read(data, 0, bufferLengthInBytes)) == -1) {
                break;
            }
            out.write(data, 0, numBytesRead);
        }

        // we reached the end of the stream.  stop and close the line.
        line.stop();
        line.close();
        line = null;

        // stop and close the output stream
        try {
            out.flush();
            out.close();
        } catch (IOException ex) {
            ex.printStackTrace();
        }

        // load bytes into the audio input stream for playback

        byte audioBytes[] = out.toByteArray();
        ByteArrayInputStream bais = new ByteArrayInputStream(audioBytes);
        capturePlaybackPanel.audioInputStream = new AudioInputStream(bais, format, audioBytes.length / frameSizeInBytes);

        long milliseconds = (long) (capturePlaybackPanel.audioInputStream.getFrameLength() * 1000 / format.getFrameRate());
        capturePlaybackPanel.duration = milliseconds / 1000.0;

        try {
            capturePlaybackPanel.audioInputStream.reset();
        } catch (Exception ex) {
            ex.printStackTrace();
            return;
        }

        capturePlaybackPanel.samplingGraph.createWaveForm(audioBytes);
    }
} // End class Capture
