package sound;

import graphics.grapher.Graph;
import gui.ClosableJFrame;
import sound.recorder.CapturePlaybackPanel;
import sound.ulaw.UlawCodec;

import javax.sound.midi.MidiChannel;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Synthesizer;
import javax.sound.sampled.*;
import javax.swing.*;
import java.awt.*;
import java.util.Vector;

/**
 * Created by IntelliJ IDEA.
 * User: Douglas Lyon
 * Date: Dec 20, 2004
 * Time: 6:06:44 PM
 * Copyright DocJava, Inc.
 */
public class SoundUtils {
    private static Synthesizer synth = null;
    public static MidiChannel[] channels = null;

    public static void main(String[] args) {
        testHanningOverlappedWindowsSpectrogram();
    }

    public static void testMic() {
        Port p = getMicrophonePort();
        Line.Info inf = p.getLineInfo();
        System.out.println(inf);
        p.addLineListener(getLineListener());
    }

    /**
     * Converts a byte array into an array of samples (a bouble).
     * Each consecutive bytes in the byte array are converted into a double,
     * and becomes the next element in the double array. The size of the
     * returned array is (length/bytesPerDouble).
     * Currently, only 1 byte (8-bit) or 2 bytes (16-bit)
     * samples are supported.
     *
     * @param byteArray      a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signedData     whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws java.lang.ArrayIndexOutOfBoundsException
     *
     */
    public static double[] bytesToSamples(byte[] byteArray,
                                          int offset,
                                          int length,
                                          int bytesPerSample,
                                          boolean signedData)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= byteArray.length) {
            int doubleLength = length / bytesPerSample;
            double[] doubleArray = new double[doubleLength];

            if (bytesPerSample == 2) {
                if (!signedData) {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        int temp = (int) byteArray[i++];
                        temp = (temp << 8);
                        temp |= (int) (0x000000FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        short temp = (short) (byteArray[i++] << 8);
                        temp |= (short) (0x00FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = offset; i < doubleLength; i++) {
                    doubleArray[i] = (double) byteArray[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytes per sample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + byteArray.length);
        }
    }


    /**
     * Converts a little-endian byte array into an array of samples (double).
     * Each consecutive bytes of a float are converted into a double, and
     * becomes the next element in the double array. The number of bytes
     * in the double is specified as an argument. The size of
     * the returned array is (data.length/bytesPerSample).
     *
     * @param data           a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signed         whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws java.lang.ArrayIndexOutOfBoundsException
     *
     */
    public static double[] littleEndianBytesToSamples(byte[] data,
                                                      int offset,
                                                      int length,
                                                      int bytesPerSample,
                                                      boolean signed)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= data.length) {
            double[] doubleArray = new double[length / bytesPerSample];

            if (bytesPerSample == 2) {
                if (signed) {
                    for (int i = offset, j = 0; i < length; j++) {
                        short temp = (short) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; i < length; j++) {
                        int temp = (int) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = 0; i < doubleArray.length; i++) {
                    doubleArray[i] = data[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytesPerSample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + data.length);
        }
    }

    public static LineListener getLineListener() {
        return new LineListener() {
            public void update(LineEvent e) {
                System.out.println("event:" + e);
            }
        };
    }

    public static Port getMicrophonePort() {
        if (AudioSystem.isLineSupported(Port.Info.MICROPHONE)) {
            try {
                return (Port) AudioSystem.getLine(Port.Info.MICROPHONE);
            } catch (LineUnavailableException e) {
                e.printStackTrace();

            }
        }
        return null;
    }

    public static void listMixersAndExit() {
        System.out.println("Available Mixers:");
        Mixer.Info[] aInfos = AudioSystem.getMixerInfo();
        for (int i = 0; i < aInfos.length; i++) {
            System.out.println(aInfos[i].getName());
        }
        if (aInfos.length == 0) {
            System.out.println("[No mixers available]");
        }
        System.exit(0);
    }

    public static void listSupportedTargetTypes() {
        String strMessage = "Supported target types:";
        AudioFileFormat.Type[] aTypes = AudioSystem.getAudioFileTypes();
        for (int i = 0; i < aTypes.length; i++) {
            strMessage += " " + aTypes[i].getExtension();
        }
        System.out.println(strMessage);
    }

    public static void getAuFilePlayAndDisplay() {
        UlawCodec ulc = new UlawCodec();
        ulc.play();
        ulc.showSampleData();
    }

    public static void showInfoDialog() {
        final String msg =
                "When running the Java Sound demo as an applet these permissions\n" +
                "are necessary in order to load/save files and record audio :  \n\n" +
                "grant { \n" +
                "  permission java.io.FilePermission \"<<ALL FILES>>\", \"read, write\";\n" +
                "  permission javax.sound.sampled.AudioPermission \"record\"; \n" +
                "  permission java.util.PropertyPermission \"user.dir\", \"read\";\n" +
                "}; \n\n" +
                "The permissions need to be added to the .java.policy file.";
        new Thread(new Runnable() {
            public void run() {
                JOptionPane.showMessageDialog(null, msg, "Applet Info", JOptionPane.INFORMATION_MESSAGE);
            }
        }).start();
    }

    public static void checkAudio() {
        try {
            if (MidiSystem.getSequencer() == null) {
                System.err.println("MidiSystem Sequencer Unavailable, exiting!");
                System.exit(1);
            } else if (AudioSystem.getMixer(null) == null) {
                System.err.println("AudioSystem Unavailable, exiting!");
                System.exit(1);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            System.exit(1);
        }
    }

    public static void play(int ia[], int vel, int dur) {
        for (int i = 0; i < ia.length; i++)
            play(ia[i], vel, dur);
    }

    public static void print(double ia[]) {
        for (int i = 0; i < ia.length; i++)
            System.out.println(ia[i]);
    }

    public static void print(int ia[]) {
        for (int i = 0; i < ia.length; i++)
            System.out.println(ia[i]);
    }

    public static void play(int nn) {
        play(nn, 127, 20);
    }

    public static void play(int nn, int dur) {
        play(nn, 127, dur);
    }

    public static void play(int nn, int vel, int dur) {
        play(getSynthesizer(), nn, vel, dur);
    }

    public static void playThread(final int ia[],
                                  final int vel,
                                  final int dur) {
        Runnable r = new Runnable() {
            public void run() {
                for (int i = 0; i < ia.length; i++)
                    play(ia[i], vel, dur);
            }
        };
        Thread t = new Thread(r);
        t.start();
    }

    public static void play(int ia[], int dur) {
        for (int i = 0; i < ia.length; i++)
            play(ia[i], 127, dur);
    }

    public static int[] transpose(int ia[], int bias) {
        int ans[] = new int[ia.length];
        for (int i = 0; i < ans.length; i++)
            ans[i] = ia[i] + bias;
        return ans;
    }

    public static Synthesizer getSynthesizer() {
        Synthesizer synth = initSynth();
        openSynth(synth);
        channels = initChannel(synth);
        return synth;
    }

    private static Synthesizer initSynth() {
        if (synth != null) return synth;
        try {
            synth = MidiSystem.getSynthesizer();
        } catch (MidiUnavailableException e) {
        }
        return synth;
    }

    private static void openSynth(Synthesizer synth) {
        try {
            synth.open();
        } catch (MidiUnavailableException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static MidiChannel[] initChannel(Synthesizer synth) {
        MidiChannel[] channels = synth.getChannels();
        return channels;
    }

    public static void testRecorder() {
        ClosableJFrame cf = new ClosableJFrame();
        Container c = cf.getContentPane();
        c.setLayout(new BorderLayout());
        CapturePlaybackPanel cpp = new CapturePlaybackPanel();
        c.add(cpp, BorderLayout.CENTER);
        cf.setSize(600, 400);
        cf.setVisible(true);
        //Vector graphicLineData = cpp.getGraphicLineData();
    }

    public static double[] getDoubleData(AudioFormat format,
                                         byte[] audioBytes) {
        int data16[] = get16BitAudioData(format, audioBytes);
        double d[] = new double[data16.length];
        for (int i = 0; i < d.length; i++)
            d[i] = data16[i] / 32768.0;
        return d;
    }

    public static int[] get16BitAudioData(AudioFormat format,
                                          byte[] audioBytes) {
        int[] audioData = new int[audioBytes.length];
        if (format.getSampleSizeInBits() == 16) {
            int nlengthInSamples = audioBytes.length / 2;
            audioData = new int[nlengthInSamples];
            if (format.isBigEndian()) {
                for (int i = 0; i < nlengthInSamples; i++) {
                    /* First byte is MSB (high order) */
                    int MSB = (int) audioBytes[2 * i];
                    /* Second byte is LSB (low order) */
                    int LSB = (int) audioBytes[2 * i + 1];
                    audioData[i] = MSB << 8 | (255 & LSB);
                }
            } else {
                for (int i = 0; i < nlengthInSamples; i++) {
                    /* First byte is LSB (low order) */
                    int LSB = (int) audioBytes[2 * i];
                    /* Second byte is MSB (high order) */
                    int MSB = (int) audioBytes[2 * i + 1];
                    audioData[i] = MSB << 8 | (255 & LSB);
                }
            }
        } else if (format.getSampleSizeInBits() == 8) {
            int nlengthInSamples = audioBytes.length;
            audioData = new int[nlengthInSamples];
            if (format.getEncoding().toString().startsWith("PCM_SIGN")) {
                for (int i = 0; i < audioBytes.length; i++) {
                    audioData[i] = audioBytes[i];
                }
            } else {
                for (int i = 0; i < audioBytes.length; i++) {
                    audioData[i] = audioBytes[i] - 128;
                }
            }
        }
        return audioData;
    }

    public static void echo(double[] linearSamples) {
        int ECHO_NUMBER = 4;
        double DECAY = 0.5;
        echo(ECHO_NUMBER, DECAY, linearSamples);
    }

    public static void echo(int numberOfEchos, double decay, double[] linearSamples) {
        int n = linearSamples.length;
        int numTimes = numberOfEchos + 1;
        double currDecay = 1.0;
        double[] newSamples = new double[n * numTimes];
        for (int j = 0; j < numTimes; j++) {
            for (int i = 0; i < n; i++) // copy the sound's bytes
                newSamples[i + (n * j)] =
                        linearSamples[i] * currDecay;
            currDecay *= decay;
        }
        linearSamples = newSamples;
        UlawCodec ulc = new UlawCodec(linearSamples);
        ulc.play();
    }

    /**
     * @return little endian 8khz linear 8bit format. Unsigned.
     */
    public static AudioFormat get8khzMono8Format() {
        float sampleRate = 8000.0F;
//8000,11025,16000,22050,44100
        int sampleSizeInBits = 8;
//8,16
        int channels = 1;
//1,2
        // use signed if 16 bit, otherwise use
        // unsigned
        boolean signed = false;
//true,false
        boolean bigEndian = true;
//true,false
        return new AudioFormat(sampleRate,
                sampleSizeInBits,
                channels,
                signed,
                bigEndian);
    }//end get8khzMono8Format

    public static void printMicrophoneInfo() {
        Port p = getMicrophonePort2();
        Line.Info inf = p.getLineInfo();
        System.out.println(inf);
        p.addLineListener(getLineListener2());
    }

    public static double[] makeBartlett(int n) {

        double window[] = new double[n];


        double a = 2.0 / (n - 1);


        for (int i = 0; i < n / 2; i++)

            window[i] = i * a;


        for (int i = n / 2; i < n; i++)

            window[i] = 2.0 - i * a;


        return window;


    }

    public static void makeLyon() {

        double window[];

        window = makeLyon(256);

        Graph.graph(window,

                "The Lyon window", "f");

    }

    public static double y5(double y0, double y1, double u) {

        double t2 = u * u;

        double t3 = t2 * t2;

        return

                (6 * y1 - 6 * y0) * t3 * u +

                (-15 * y1 + 15 * y0) * t3 +

                (10 * y1 - 10 * y0) * t2 * u + y0;

    }

    public static double[] makeLyon(int n) {

        double window[] = new double[n];


        double u = 0.0;

        double du = 2.0 / n;

        for (int i = 0; i < n / 2; i++) {

            window[i] = y5(0, 1.0, u);

            u += du;

        }

        u = 0;


        for (int i = n / 2; i < n; i++) {

            window[i] = y5(1.0, 0.0, u);

            u += du;

        }

        return window;


    }

    public static void makeHanning() {
        Graph.graph(makeHanning(256),
                "The Hanning window", "f");
    }

    public static double[] makeHanning(int n) {
        double window[] = new double[n];
        double arg = 2.0 * Math.PI / (n - 1);
        for (int i = 0; i < n; i++)
            window[i] = 0.5 - 0.5 * Math.cos(arg * i);
        return window;
    }
    public static void testHanningOverlappedWindowsSpectrogram() {
         double d[] = new double[2048];
         for (int i = 0; i < d.length; i++)
             d[i] = i * Math.sin(i / Math.PI);
         Vector v = getOverlappedWindows(d);
         System.out.println("v.size=" + v.size());
         double h[] = makeHanning(256);
         for (int i = 0; i < v.size(); i++) {
             double[] ia = (double[]) v.elementAt(i);
             windowArray(h, ia);
             Graph.graph(ia, "w" + i, "hanning");
         }

     }

    public static void testHanningOverlappedWindows() {
        double d[] = new double[2048];
        for (int i = 0; i < d.length; i++)
            d[i] = i * Math.sin(i / Math.PI);
        Vector v = getOverlappedWindows(d);
        System.out.println("v.size=" + v.size());
        double h[] = makeHanning(256);
        for (int i = 0; i < v.size(); i++) {
            double[] ia = (double[]) v.elementAt(i);
            windowArray(h, ia);
            Graph.graph(ia, "w" + i, "hanning");
        }

    }

    public static void testGetOverlappedWindows() {
        double d[] = new double[511];
        for (int i = 0; i < d.length; i++)
            d[i] = i;
        Vector v = getOverlappedWindows(d);
        System.out.println("v.size=" + v.size());
        for (int i = 0; i < v.size(); i++) {
            print((double[]) v.elementAt(i));
        }

    }

    /**
     * return a vector of arrays, each array is only
     * 256 elements long, each array overlaps by 50% with
     * each other array. This is done before windowing is performed
     *
     * @param data
     * @return The Vector of double arrays of data
     */
    public static Vector getOverlappedWindows(double data[]) {
        Vector v = new Vector();
        int n = data.length;
        int ws = 256;
        int overlap = ws / 2;
        for (int i = 0; i < (2 * n / ws) - 1; i++) {
            double window[] = new double[ws];
            for (int j = 0; j < window.length; j++) {
                window[j] = data[i * overlap + j];
            }
            v.addElement(window);
        }
        return v;
    }

    public static void windowArray(double window[], double r_d[]) {
        for (int i = 0; i < window.length; i++) {
            r_d[i] *= window[i];
        }
    }

    /**
     * This will apply the windows to each of the arrays.
     * The length of all the arrays must be the same.
     *
     * @param window
     * @param r_d
     * @param i_d
     */
    public static void windowArrays(double window[], double r_d[], double i_d[]) {

        for (int i = 0; i < window.length; i++) {

            r_d[i] *= window[i];

            i_d[i] *= window[i];

        }

    }

    public static void logArray(double a[]) {

        for (int i = 0; i < a.length; i++) {

            a[i] = 10 * Math.log(a[i]);

        }

    }

    public static Port getMicrophonePort2() {
        if (AudioSystem.isLineSupported(Port.Info.MICROPHONE)) {
            try {
                return (Port) AudioSystem.getLine(Port.Info.MICROPHONE);
            } catch (LineUnavailableException e) {
                e.printStackTrace();

            }
        }
        return null;
    }

    public static LineListener getLineListener2() {
        return new LineListener() {
            public void update(LineEvent e) {
                System.out.println("event:" + e);
            }
        };
    }

    public static void play(Synthesizer synth, int nNoteNumber,
                            int nVelocity, int nDuration) {
        noteOn(nNoteNumber, nVelocity);
        if (nDuration < 0) return;
        sleep(nDuration);
        noteOff(nNoteNumber);
        System.out.println("synthesizer=" + synth);
    }

    public static void noteOff(int nNoteNumber) {
        channels[0].noteOff(nNoteNumber);
    }

    public static void noteOn(int nNoteNumber, int nVelocity) {
        channels[0].noteOn(nNoteNumber, nVelocity);
    }

    public static void sleep(int nDuration) {
        try {
            Thread.sleep(nDuration);
        } catch (InterruptedException e) {
        }
    }
}
