/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package serialPort;

/**
 * Copyright DocJava, inc. User: lyon
 * <p/>
 * Date: Jul 12, 2005
 * <p/>
 * Time: 7:45:20 AM
 */
import gnu.io.*;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.HashMap;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * Subclasses CommPortOpen and adds send/expect handling for dealing with
 * Hayes-type modems.
 *
 * @author Ian F. Darwin, http://www.darwinsys.com/
 */
class CommPortModem extends CommPortOpen {

    /**
     * The last line read from the serial port.
     */
    protected String response;

    /**
     * A flag to control debugging output.
     */
    protected boolean debug = true;

    public CommPortModem(JFrame f)
            throws IOException, PortInUseException,
                   UnsupportedCommOperationException {
        super(f);
    }

    /**
     * Send a line to a PC-style modem. Send \r\n, regardless of what
     * platform we're on, instead of using println().
     */
    protected void send(String s) throws IOException {
        if (debug) {
            System.out.print(">>> ");
            System.out.print(s);
            System.out.println();
        }
        os.print(s);
        os.print("\r\n");

        // Expect the modem to echo the command.
        if (!expect(s)) {
            System.err.println("WARNING: Modem did not echo command.");
        }

        // The modem sends an extra blank line by way of a prompt.
        // Here we read and discard it.
        String junk = is.readLine();
        if (junk.length() != 0) {
            System.err.print("Warning: unexpected response: ");
            System.err.println(junk);
        }
    }

    /**
     * Read a line, saving it in "response".
     *
     * @return true if the expected String is contained in the response,
     *         false if not.
     */
    protected boolean expect(String exp) throws IOException {
        response = is.readLine();
        if (debug) {
            System.out.print("<<< ");
            System.out.print(response);
            System.out.println();
        }
        return response.indexOf(exp) >= 0;
    }
}

class CommPortOpen {
    /**
     * How long to wait for the open to finish up.
     */
    public static final int TIMEOUTSECONDS = 30;
    /**
     * The baud rate to use.
     */
    public static final int BAUD = 19200;
    /**
     * The parent JFrame, for the chooser.
     */
    protected JFrame parent;
    /**
     * The input stream
     */
    protected BufferedReader is;
    /**
     * The output stream
     */
    protected PrintStream os;
    /**
     * The chosen Port Identifier
     */
    CommPortIdentifier thePortID;
    /**
     * The chosen Port itself
     */
    CommPort thePort;

    public static void main(String[] argv)
            throws IOException, PortInUseException,
                   UnsupportedCommOperationException {

        new CommPortOpen(null).converse();

        System.exit(0);
    }

    /* Constructor */
    public CommPortOpen(JFrame f)
            throws IOException, PortInUseException,
                   UnsupportedCommOperationException {
        // Use the PortChooser from before. Pop up the JDialog.
        PortChooser chooser = new PortChooser(null);

        String portName = null;
        do {
            chooser.setVisible(true);
            // Dialog done. Get the port name.
            portName = chooser.getSelectedName();

            if (portName == null)
                System.out.println("No port selected. Try again.\n");
        } while (portName == null);

        // Get the CommPortIdentifier.
        thePortID = chooser.getSelectedIdentifier();

        // Now actually open the port.
        // This form of openPort takes an Application Name and a timeout.
        //
        System.out.println("Trying to open " +
                thePortID.getName() +
                "...");

        switch (thePortID.getPortType()) {
            case CommPortIdentifier.PORT_SERIAL:
                thePort = thePortID.open("DarwinSys DataComm",
                        TIMEOUTSECONDS * 1000);
                SerialPort myPort = (SerialPort) thePort;

                // set up the serial port
                myPort.setSerialPortParams(BAUD, SerialPort.DATABITS_8,
                        SerialPort.STOPBITS_1, SerialPort.PARITY_NONE);
                break;

            case CommPortIdentifier.PORT_PARALLEL:
                thePort = thePortID.open("DarwinSys Printing",
                        TIMEOUTSECONDS * 1000);
                ParallelPort pPort = (ParallelPort) thePort;

                // Tell API to pick "best available mode" - can fail!
                // myPort.setMode(ParallelPort.LPT_MODE_ANY);

                // Print what the mode is
                int mode = pPort.getMode();
                switch (mode) {
                    case ParallelPort.LPT_MODE_ECP:
                        System.out.println("Mode is: ECP");
                        break;
                    case ParallelPort.LPT_MODE_EPP:
                        System.out.println("Mode is: EPP");
                        break;
                    case ParallelPort.LPT_MODE_NIBBLE:
                        System.out.println("Mode is: Nibble Mode.");
                        break;
                    case ParallelPort.LPT_MODE_PS2:
                        System.out.println("Mode is: Byte mode.");
                        break;
                    case ParallelPort.LPT_MODE_SPP:
                        System.out.println("Mode is: Compatibility mode.");
                        break;
                        // ParallelPort.LPT_MODE_ANY is a "set only" mode;
                        // tells the API to pick "best mode"; will report the
                        // actual mode it selected.
                    default:
                        throw new IllegalStateException("Parallel mode " +
                                mode +
                                " invalid.");
                }
                break;
            default:  // Neither parallel nor serial??
                throw new IllegalStateException("Unknown port type " +
                        thePortID);
        }

        // Get the input and output streams
        // Printers can be write-only
        try {
            is
                    = new BufferedReader(new InputStreamReader(thePort.getInputStream()));
        } catch (IOException e) {
            System.err.println("Can't open input stream: write-only");
            is = null;
        }
        os = new PrintStream(thePort.getOutputStream(), true);
    }

    /**
     * This method will be overridden by non-trivial subclasses to hold a
     * conversation.
     */
    protected void converse() throws IOException {

        System.out.println("Ready to read and write port.");

        // Input/Output code not written -- must subclass.

        // Finally, clean up.
        if (is != null)
            is.close();
        os.close();
    }
}

