/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * Title:        EulerRichardson
 * Description:  An Euler-Richardson method ODE solver.
 *
 * The Euler-Richardson method uses the state at the beginning of the interval
 * to estimate the state at the midpoint.
 *
 * x(midpoint) = x(n) + v(n)*dt/2
 * v(midpoint) = v(n) + a(n)*dt/2
 * t(midpoint) = t(n) + dt/2
 *
 * The midpoint state is then used to calculate the final state.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class EulerRichardson implements ODESolver {

  private double   stepSize = 0.1;
  private int      numEqn   = 0;
  private double[] rate;
  private double[] midstate;  // midpoint
  private ODE      ode;

  /**
   * Constructs the EulerRichardson ODESolver for a system of ordinary  differential equations.
   *
   * @param _ode the system of differential equations.
   */
  public EulerRichardson(ODE _ode) {
    ode = _ode;
    initialize(stepSize);
  }

  /**
   * Initializes the ODE solver.
   *
   * The rate and midstate arrays are allocated.
   * The number of differential equations is determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
  public void initialize(double _stepSize) {
    stepSize = _stepSize;
    double state[] = ode.getState();
    if(state == null) {  // state vector not defined.
      numEqn = 0;
    } else {
      numEqn = state.length;
    }
    rate     = new double[numEqn];
    midstate = new double[numEqn];
  }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
  public double step() {
    double[] state = ode.getState();
    ode.getRate(state, rate);  // get the rate at the start
    for(int i = 0; i < numEqn; i++) {
      // estimate the state at the midpoint
      midstate[i] = state[i] + stepSize * rate[i] / 2;
    }
    ode.getRate(midstate, rate);  // get the rate at the midpoint
    for(int i = 0; i < numEqn; i++) {
      state[i] = state[i] + stepSize * rate[i];
    }
    return stepSize;
  }

  /**
   * Sets the step size.
   *
   * The step size remains fixed in this algorithm.
   *
   * @param _stepSize
   */
  public void setStepSize(double _stepSize) {
    stepSize = _stepSize;
  }

  /**
   * Gets the step size.
   *
   * The stepsize is constant in this algorithm.
   *
   * @return the step size
   */
  public double getStepSize() {
    return stepSize;
  }
}
