package math.numerics;

/**
 * Class Integral defines various derivative algorithms.
 * This class cannot be subclassed or instantiated because all methods are static.
 *
 * @author Wolfgang Christian
 */

public class Derivative {
  private Derivative() { }  // prohibit instantiation because all methods are static

  /** Calculates the derivative using the Romberg scheme for Richardson extrapolation.
 *
 * This method runs until all Romberg rows are filled or until
 * the step size drops below defaultNumericalPrecision or if the desired
 * tolerance is reached.
 *
 * @param f   the function
 * @param x0   where derivative is to be calculated
 * @param h   initial step size
 * @param tol desired accuracy
 * @return
 */
static public double romberg(Function f,double x0, double h, double tol){
  int        n=6;  //max. number of columns in the Romberg scheme
  double[]   d= new double[n];
  d [0] = (f.evaluate (x0 + h) - f.evaluate(x0 - h)) / h/2.0;

  int error_code=1;
  for( int j=1; j <= n - 1;j++){
    d [j] = 0.0;
    double d1    = d [0];
    double h2  = h;
    h    *= 0.5;
    if (h < Util.defaultNumericalPrecision){
      error_code = 2;  /* step size less than  defaultNumericalPrecision */
      break;
    }
    d [0] = (f.evaluate (x0 + h) - f.evaluate (x0 - h)) / h2;
    for (int m = 4, i = 1; i <= j; i++, m *= 4){
      double d2 = d [i];
      d [i] = (m * d [i-1] - d1) / (m-1);
      d1 = d2;
    }
    if (Math.abs(d [j] - d [j-1]) < tol){  /* desired accuracy reached */
      return d[j];
    }
  }
  throw new NumericMethodException("Derivative did not converge.",error_code,d[0]);
}

/**
 * Calculates the derivative of a function using the centered difference approximation.
 * @param f  the function
 * @param x  the x value
 * @param h
 * @return
 */
 static public double centered(Function f,double x, double h){
   return (f.evaluate (x + h) - f.evaluate(x - h)) / h/2.0;
 }

 static public double centeredPartial(MultiVarFunction f, double[] x, int n, double tol){
     double[] tempPlus= new double[x.length];
     System.arraycopy(x,0,tempPlus,0,x.length);
     tempPlus[n]+= tol;
     double[] tempMinus= new double[x.length];
     System.arraycopy(x,0,tempMinus,0,x.length);
     tempMinus[n]-= tol;
     return (f.evaluate(tempPlus)-f.evaluate(tempMinus))/2.0/tol;
 }


 static public double second(Function f,double x, double tol){
     if(tol==0) tol=1.0e-9;
     double dx=tol;
     double dx2=tol*tol;
     double count=10;
     double d2fdx2=0;  // second order finite difference result
     double d2fdx4=0;  // fourth order finite difference result
     do{
       double fP2=f.evaluate(x+2*dx);
       double fP1=f.evaluate(x+dx);
       double f0=f.evaluate(x);
       double fM1=f.evaluate(x-dx);
       double fM2=f.evaluate(x-2*dx);
       d2fdx2 =(fP1-2*f0+fM1)/dx2;
       d2fdx4 = (-fP2+16*fP1-30*f0+16*fM1-fM2)/dx2/12.0;
       dx /= 10;  // decrease the dx in preparation for next try.
       dx2=dx*dx;
       count--;
     } while(count>0 && Math.abs(d2fdx2-d2fdx4)>tol );
     return d2fdx4;
 }

 /**
 * Fills an array with the second derivative of a function.
 * @param f
 * @param start
 * @param stop
 * @param tol
 * @param n
 * @return
 */
static public double[][] secondFillArray (Function f, double start, double stop, double tol, int n) {
    double[][] data = new double[2][n];
    return secondFillArray ( f,  start, stop,tol, data);
}

  /**
   * Fills an array with the second derivative of a function.
   * @param f Function
   * @param start double
   * @param stop double
   * @param tol double
   * @param data double[][]
   * @return double[][]
   */
  static public double[][] secondFillArray (Function f, double start, double stop, double tol, double[][] data) {
    double dx=1;
    int n=data[0].length;
    if(n>1)dx=(stop-start)/(n-1);
    double x=start;
    for(int i=0; i<n; i++){
       data[0][i]=x;
       data[1][i] =romberg(f,x,dx,tol);
       x+=dx;
    }
    return data;
}

  /**
 * Fills an array with the derivatives of a function.
 * @param f
 * @param start
 * @param stop
 * @param tol
 * @param data
 * @return
 */
static public double[][] fillArray (Function f, double start, double stop, double tol, double[][] data) {
    double dx=1;
    int n=data[0].length;
    if(n>1)dx=(stop-start)/(n-1);
    double x=start;
    for(int i=0; i<n; i++){
       data[0][i]=x;
       data[1][i] =romberg(f,x,dx,tol);
       x+=dx;
    }
    return data;
}

  static public double[][] fillArray (Function f, double start, double stop, double tol, int n) {
    double[][] data = new double[2][n];
    return fillArray ( f,  start, stop,tol, data);
}





}
