/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package math;

import utils.StopWatch;

/**
 * Mat1 is a class that uses 1-D arrays and contains static methods.
 * @author Douglas Lyon
 */
public abstract class Mat1 {

    public static void main(final String[] args) {
        //testAdd1();

        //testAdd2();
        for (int i=0; i < 10; i++)
            testSum();
    }


    /**
     * Add a vector times a scalar to a vector.
     * The first vector contains the sum.
     *
     * @param a the first  vector
     * @param b the second vector
     * @param c the scalar multiplier
     */
    public static void add(final double[] a,
                           final double[] b,
                           final double c) {
        final int aLength = a.length;
        if (aLength != b.length) {
            System.out.println(
                    "ERROR: Vectors must be of equal length to add.");
            return;
        }
        for (int i = 0; i < aLength; i++) {
            a[i] += c * b[i];
        }
    }

    /**
     * Add two vectors. The first vector contains
     * the sum.
     * <b>This is not optimized for <i>altivec</i>
     * </b>
     *
     * @param a the first  vector  <code>a = a + b; </code>
     * @param b the second vector
     */
    public static void add(final double[] a,
                           final double[] b) {
        final int aLength = a.length;
        if (aLength != b.length) {
            System.out.println(
                    "ERROR: Vectors must be of equal length to add.");
            return;
        }
        for (int i = 0; i < aLength; i++) {
            a[i] += b[i];
        }
    }

    /**
     * Add two vectors. The first vector contains
     * the sum.
     *
     * @param a the first  vector  a = a + b;
     * @param b the second vector
     */
    public static void add(final float[] a,
                           final float[] b) {
        final int aLength = a.length;
        if (aLength != b.length) {
            System.out.println(
                    "ERROR: Vectors must be of equal length to add.");
            return;
        }
        for (int i = 0; i < aLength; i++) {
            a[i] += b[i];
        }
    }

    public static void testSum() {
        final int n = 1 << 19;
        System.out.println("n=" + n);
        final float[] a = ramp(0, 100, n);
        final StopWatch t = new StopWatch();
        double ans;
        ans = sum(a);
        ans = sum(a);
        ans = sum(a);
        t.start();
        ans = sum(a);
        t.stop();

        t.print(1, "ans=" + ans);
        ans = sum4(a);
        ans = sum4(a);
        ans = sum4(a);
        t.start();
        ans = sum4(a);
        t.stop();
        t.print(1, "ans4=" + ans);
        ans = sum8(a);
        ans = sum8(a);
        ans = sum8(a);
        t.start();
        ans = sum8(a);
        t.stop();
        t.print(1, "ans8=" + ans);
        ans = sumV(a);
        ans = sumV(a);
        ans = sumV(a);
        t.start();
        ans = sumV(a);
        t.stop();
        t.print(1, "sumV=" + ans);
    }

    public static void testAdd2() {
        final int n = 1 << 12;
        System.out.println("n=" + n);
        final float[] a = ramp(0, 100, n);
        final float[] b = ramp(0, 100, n);
        final StopWatch t = new StopWatch();
        double ans;
        t.start();
        add(a, b);
        ans = add(a);
        t.stop();

        t.print(1, "ans=" + ans);

        ans = sum8(a);
        t.stop();
        t.print(1, "ans=" + ans);
    }

    /**
     * Calculate the dot product of two vectors.
     *
     * @param a the first  vector
     * @param b the second vector
     * @return the dot product
     */
    static public double dot(final double[] a,
                             final double[] b) {
        final int aLength = a.length;
        if (aLength != b.length) {
            System.out.println(
                    "ERROR: Vectors must be of equal length in dot product.");
            return 0;
        }
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i] * b[i];
        }
        return sum;
    }


    /**
     * Calculate the dot product of a vector with
     * itself.
     *
     * @param a the  vector
     * @return the dot product
     */
    static public double dot(final double[] a) {
        final int aLength = a.length;
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i] * a[i];
        }
        return sum;
    }

    /**
     * Calculate the magnitude a vector.
     *
     * @param a the  vector
     * @return the magnitude
     */
    static public double magnitude(
            final double[] a) {
        final int aLength = a.length;
        double sum = 0;
        for (int i = 0; i < aLength; i++) {
            sum += a[i] * a[i];
        }
        return Math.sqrt(sum);
    }

    public static void print(final double[] a) {
        for (int i = 0; i < a.length; i++) {

            System.out.print(a[i] + " ");
            System.out.println();
        }
    }

    public static void printStats(
            final String title,
            final float[] a) {
        System.out.println(title);
        printStats(a);
    }

    public static void printStats(
            final float[] a) {
        float min = Float.MAX_VALUE;
        float max = Float.MIN_VALUE;
        float aBar = 0;


        final double N = a.length;
        for (int x = 0; x < a.length; x++) {

            aBar += a[x];

            min = Math.min(a[x], min);
            max = Math.max(a[x], max);
        }


        aBar /= N;

        System.out.println(" aBar=" + aBar +
                           " a min=" +
                           min +
                           " a max=" +
                           max +
                           " a.length=" +
                           a.length +
                           " a[0].length=" +
                           a.length);

    }

    public static void print(final float[] a) {
        for (int i = 0; i < a.length; i++) {
            System.out.print(a[i] + " ");
            System.out.println();
        }
    }

    public static final float sum(
            final float[] a) {
        float s = 0;
        for (int i = 0; i < a.length; i++)
            s += a[i];
        return s;
    }

    public static final double sum(
            final double[] a) {
        double s = 0.0;
        for (int i = 0; i < a.length; i++)
            s += a[i];
        return s;
    }

    public static float[] ramp(final float start,
                               final float end,
                               final int n) {
        final float[] f = new float[n];
        float t = 0f;
        for (int i = 0; i < f.length; i++) {

            f[i] = t * end + (1 - t) * start;
            // t*p1 + (1-t)*p0
            t = (1f * i) / n;
        }
        return f;

    }

    public static void testAdd1() {
        final int n = 1 << 18;
        System.out.println("n=" + n);
        final float[] a = ramp(0, 100, n);
        //final float[] b = ramp(0, 200, n);
        //final float[] c = new float[a.length];
        //final Timer t = new Timer();


        testAdd1(a);


    }

    /**
     * Show how loop unwinding can improve speed
     * for a vector sum, but only to a point.
     */
    public static void testAdd1(final float a[]) {

        int numberOfRuns = 100;
        System.out.println(
                "each run is executed " +
                numberOfRuns +
                " times, averages are reported in ms");
        Runnable r = new Runnable() {
            public void run() {
                add(a);
            }
        };
        System.out.println(" add: " +
                           StopWatch.benchMark(r,
                                           numberOfRuns));

        r = new Runnable() {
            public void run() {
                sumV(a);
            }
        };
        System.out.println(" SumV: " +
                           StopWatch.benchMark(r,
                                           numberOfRuns));


        r = new Runnable() {
            public void run() {
                add16(a);
            }
        };
        System.out.println(" add16:" +
                           StopWatch.benchMark(r,
                                           numberOfRuns));

        r = new Runnable() {
            public void run() {
                add8Depend(a);
            }
        };
        System.out.println("  add8Depend:" +
                           StopWatch.benchMark(r,
                                           numberOfRuns));

        r = new Runnable() {
            public void run() {
                add8NoDepend(a);
            }
        };
        System.out.println(" add8NoDepend:" +
                           StopWatch.benchMark(r,
                                           numberOfRuns));


        //add(a, b, c);
        // ans = add8(c);
        // t.print(1, "add c = a + b, ans=" + ans);


        // add8(a, b, c);
        // ans = add8(c);
        // t.print(1,
        //         "add c = a + b, p; ans=" + ans);
    }

    public static void add(final float[] a,
                           final float[] b,
                           final float[] c) {
        for (int i = 0; i < a.length; i++)
            c[i] = a[i] + b[i];
    }

    public static void add8(final float[] a,
                            final float[] b,
                            final float[] c) {
        int i1, i2, i3, i4, i5, i6, i7;
        for (int i = 0; i < a.length; i += 8) {
            i1 = i + 1;
            i2 = i + 2;
            i3 = i + 3;
            i4 = i + 4;
            i5 = i + 5;
            i6 = i + 6;
            i7 = i + 7;
            c[i] = a[i] + b[i];
            c[i1] = a[i1] + b[i1];
            c[i2] = a[i2] + b[i2];
            c[i3] = a[i3] + b[i3];
            c[i4] = a[i4] + b[i4];
            c[i5] = a[i5] + b[i5];
            c[i6] = a[i6] + b[i6];
            c[i7] = a[i7] + b[i7];
        }
    }

    public static final float sum4(final float[] a) {
         float s = 0;
         for (int i = 0; i < a.length; i = i + 4) {
             s = a[i] +
                 a[i + 1] +
                 a[i + 2] +
                 a[i + 3] +
                 s;
         }
         return s;
     }

    public static final float sum8(final float[] a) {
        float s = 0;
        for (int i = 0; i < a.length; i = i + 8) {
            s = a[i] +
                a[i + 1] +
                a[i + 2] +
                a[i + 3] +
                a[i + 4] +
                a[i + 5] +
                a[i + 6] +
                a[i + 7] +
                s;
        }
        return s;
    }

    public static double add8Depend(
            final float[] a) {
        double s = 0;
        for (int i = 0; i < a.length; i = i + 8) {
            s = s + a[i];
            s = s + a[i + 1];
            s = s + a[i + 2];
            s = s + a[i + 3];
            s = s + a[i + 4];
            s = s + a[i + 5];
            s = s + a[i + 6];
            s = s + a[i + 7];
        }
        return s;
    }

    public static double add8NoDepend(
            final float[] a) {
        final float[] s = new float[8];
        for (int i = 0; i < a.length; i = i + 8) {
            s[0] = s[0] + a[i];
            s[1] = s[1] + a[i + 1];
            s[2] = s[2] + a[i + 2];
            s[3] = s[3] + a[i + 3];
            s[4] = s[4] + a[i + 4];
            s[5] = s[5] + a[i + 5];
            s[6] = s[6] + a[i + 6];
            s[7] = s[7] + a[i + 7];

        }
        // compacting phase.
        return
                s[0] +
                s[1] +
                s[2] +
                s[3] +
                s[4] +
                s[5] +
                s[6] +
                s[7];
    }

    public static double add16(final float[] a) {
        double s = 0;
        for (int i = 0; i < a.length; i = i + 16) {
            s = a[i] +
                a[i + 1] +
                a[i + 2] +
                a[i + 3] +
                a[i + 4] +
                a[i + 5] +
                a[i + 6] +
                a[i + 7] +
                a[i + 8] +
                a[i + 9] +
                a[i + 10] +
                a[i + 11] +
                a[i + 12] +
                a[i + 13] +
                a[i + 14] +
                a[i + 15] +
                s;
        }
        return s;
    }


    public static double add(final float[] a) {
        double s = 0.0;
        for (int i = 0; i < a.length; i++)
            s = s + a[i];
        return s;
    }

    public static double add(final short[] a) {
        double s = 0.0;
        if (a == null) return 0;
        for (int i = 0; i < a.length; i++)
            s += a[i];
        return s;
    }

    public static short clip(final short i) {
        if (i < 0)
            return 0;

        if (i > 255)
            return 255;

        return i;
    }

    public static final float sumV(
            final float a[]) {
        int i = 0;
        int n = a.length;
        float s[] = {0, 0, 0, 0};
        for (i = 0; i < n; i = i + 4) {
            s[0] = s[0] + a[i];
            s[1] = s[1] + a[i + 1];
            s[2] = s[2] + a[i + 2];
            s[3] = s[3] + a[i + 3];
        }// compaction phase...
        return s[0] + s[1] + s[2] + s[3];
    }

    public static short[][] clip(
            final short[][] i) {
        final short[][] i2 = new short[i.length][i[0].length];
        for (int x = 0; x < i2.length; x++)
            for (int y = 0; y < i2[0].length; y++)
                i2[x][y] = clip(i[x][y]);
        return i2;
    }
    /**
     * Sum all the elements in an array, then divide by the total.
     * for example a[]={1,0,10}, sum is 11, now divide
     * by the total and the array is 1/11,0,10/11 and the array
     * sums to one.
     * @param a
     */
    public static void normalize(
            final double[] a) {
        scale(a, 1.0 / sum(a));
    }

    public static float[] normalize(
            final short[] a) {
        return scale(a, (float) (1.0f / add(a)));
    }

    public static void normalize(final float[] a) {
        scale(a, 1.0 / add(a));
    }

    public static double average(
            final double[] a) {
        final int n = a.length * a.length;
        return sum(a) / n;
    }

    public static double average(final float[] a) {
        final int n = a.length * a.length;
        return add(a) / n;
    }

    public static double average(final short[] a) {
        final int n = a.length * a.length;
        return add(a) / n;
    }

    public static void threshold(final short[] a,
                                 final short thresh) {
        for (int i = 0; i < a.length; i++)
            if (a[i] < thresh)
                a[i] = 0;
            else
                a[i] = 255;
    }

    public static void threshold(final short[] a) {
        threshold(a, (short) average(a));
    }


    public static float[] scale(final short[] a,
                                final float k) {
        if (a == null) return null;
        final float[] f = new float[a.length];
        for (int i = 0; i < a.length; i++) {
            f[i] = k * a[i];
        }
        return f;

    }

    public static void scale(final double[] a,
                             final double k) {
        System.out.println(
                "scale(double a[], double k)");
        for (int i = 0; i < a.length; i++) {

            a[i] *= k;
        }

    }

    public static void scale(final float[] a,
                             final float k) {
        if (a == null) return;
        for (int i = 0; i < a.length; i++) {
            a[i] = a[i] * k;
        }
    }

    public static void scale(final float[] a,
                             final double k) {
//System.out.println("scale(float a[][], double k)");
//	System.out.println("k="+k);
        scale(a, (float) k);
    }

    public static float[] shortToFloat(
            final short[] a) {
        final int w = a.length;

        final float[] c = new float[w];
        for (int i = 0; i < w; i++)
            c[i] = a[i];
        return c;
    }

    public static short[] copyArray(
            final short[] a) {
        final int w = a.length;

        final short[] c = new short[w];
        for (int i = 0; i < w; i++)
            c[i] = a[i];
        return c;
    }

    public static double variance(final int[] a) {
        final double xBar = mean(a);
        double sum = 0;
        double dx;
        for (int i = 0; i < a.length; i++) {
            dx = a[i] - xBar;
            sum += dx * dx;
        }
        return sum / a.length;
    }

    public static double mean(final int[] a) {
        double sum = 0;
        for (int i = 0; i < a.length; i++)
            sum += a[i];
        return sum / a.length;
    }

    public static double coefficientOfVariation(
            final int[] a) {
        final double aBar = mean(a);
        final double aBar2 = aBar * aBar;
        return Math.sqrt(variance(a) / aBar2);
    }

    public static void testVariance() {
        final int[] a = {1, 2, 3, 5, 4, 3, 2, 5,
                         6, 7};
        System.out.println(
                "The variance =" + variance(a));
    }

    public static void testCoefficientOfVariation() {
        final int[] a = {0, 85, 87, 90, 100};
        System.out.println("coefficientOfVariation({0,85,87,90,100}) ="
                           +
                           coefficientOfVariation(
                                   a));
        final int[] b = {95, 85, 87, 90, 100};
        System.out.println("The coefficientOfVariation({95,85,87,90,100}) ="
                           +
                           coefficientOfVariation(
                                   b));
    }

    public static boolean outlierHere(
            final int[] a) {
        return (coefficientOfVariation(a) > .1);
    }

    public static void testOutlier() {
        final int[] a = {0, 85, 87, 90, 100};
        final int[] b = {95, 85, 87, 90, 100};
        System.out.println("dog ate my homework ={0,85,87,90,100}"
                           + outlierHere(a));
        System.out.println("dog ate my homework ={95,85,87,90,100}"
                           + outlierHere(b));
    }

    public static void quickSort(final int[] a,
                                 final int lo0,
                                 final int hi0) {
// Based on the QuickSort method by
// James Gosling from Sun's SortDemo applet

        int lo = lo0;
        int hi = hi0;
        final int mid;
        int t;

        if (hi0 > lo0) {
            mid = a[(lo0 + hi0) / 2];
            while (lo <= hi) {
                while ((lo < hi0) &&
                       (a[lo] < mid))
                    ++lo;
                while ((hi > lo0) &&
                       (a[hi] > mid))
                    --hi;
                if (lo <= hi) {
                    t = a[lo];
                    a[lo] = a[hi];
                    a[hi] = t;
                    ++lo;
                    --hi;
                }
            }
            if (lo0 < hi)
                quickSort(a, lo0, hi);
            if (lo < hi0)
                quickSort(a, lo, hi0);

        }
    }

    public static void swap(final int[] x,
                            final int a,
                            final int b) {
        final int t = x[a];
        x[a] = x[b];
        x[b] = t;
    }

    public static void quickSort(final int[] a) {
        quickSort(a, 0, a.length - 1);
    }

    public static double mean(final short[][] a) {
        double sum = 0;
        for (int x = 0; x < a.length; x++)
            for (int y = 0; y < a[0].length; y++) {
                sum += a[x][y];
            }
        return sum / (a.length * a[0].length);
    }

    public static double variance(
            final short[][] a) {
        final double xBar = mean(a);
        double sum = 0;
        double dx;
        for (int x = 0; x < a.length; x++)
            for (int y = 0; y < a[0].length; y++) {
                dx = a[x][y] - xBar;
                sum += dx * dx;
            }
        return sum / (a.length * a[0].length);
    }

    public static double[] getAverage(
            final double[] a,
            final double[] b,
            final double[] c) {
        final double[] avg = new double[a.length];

        for (int i = 0; i < a.length; i++) {
            avg[i] = (a[i] + b[i] + c[i]) / 3.0;
        }
        return avg;
    }

    public static short[][] copy(
            final short[][] r) {
        final short[][] c = new short[r.length][r[0].length];
        final int w = r.length;
        final int h = r[0].length;
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++)
                c[x][y] = r[x][y];
        return c;
    }

    public static short getMin(final short[] a) {
        short min = 255;
        for (int i = 0; i < a.length; i++)
            if (a[i] < min)
                min = a[i];
        return min;
    }

    public static short getMax(final short[] a) {
        short max = -255;
        for (int i = 0; i < a.length; i++)
            if (a[i] > max)
                max = a[i];
        return max;
    }


    public static void testQuickSort() {
        final int[] a = {1, 2, 3, 5, 4, 3, 2, 5,
                         6, 7};
        quickSort(a);
        for (int i = 0; i < a.length; i++)
            System.out.println(a[i]);
    }

    public static int numberOfNonZeros(
            final short[][] k) {
        final int umax = k.length;
        final int vmax = k[0].length;
        int sum = 0;

        for (int x = 0; x < umax; x++)
            for (int y = 0; y < vmax; y++)
                if (k[x][y] != 0) sum++;
        return sum;
    }

    public static void printMedian(
            final short[][] k,
            final String name) {
//printMaple(k);
        System.out.println("\npublic void " +
                           name +
                           "(){\n"
                           + "\tfloat k[][] = {");
        final int w = k.length;
        final int h = k[0].length;

        for (int y = 0; y < h; y++) {
            System.out.print("\t{");
            for (int x = 0; x < w - 1; x++)
                System.out.print(k[x][y] + ", ");
            String s = k[w - 1][y] + "}";
            if (y < h - 1)
                s = s + ",";
            else
                s = s + "};";
            System.out.println(s);
        }

        final String s = "\n\tmedian(k);\n}";
        System.out.println(s);

    }


}
