package j3d.utils.sceneGraphBuilderUtils;

import com.sun.j3d.utils.geometry.Cone;
import com.sun.j3d.utils.geometry.Primitive;
import com.sun.j3d.utils.picking.behaviors.PickTranslateBehavior;
import com.sun.j3d.utils.universe.SimpleUniverse;

import javax.media.j3d.*;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;
import java.awt.*;
import java.util.Enumeration;

public class SGBuilder {

    public static Appearance appearance(int mode, int cull) {
        // Duplicate the strip geometry and set the
        // appearance of the new Shape3D object to line mode
        // without culling.  The PolygonOffset is set to prevent stitching.
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setCullFace(cull);
        polyAttrib.setPolygonMode(mode);
        polyAttrib.setPolygonOffset(0.001f);
        Appearance polyAppear = new Appearance();
        polyAppear.setPolygonAttributes(polyAttrib);
        return polyAppear;
    }

    public static BranchGroup axes(double radius) {
        BranchGroup bg = SGBuilder.newBranchGroup("Axes");
        bg.addChild(SGBuilder.line("X axis",
                new Point3d(-radius, 0, 0),
                new Point3d(radius, 0, 0),
                new Color3f(Color.magenta),
                new Color3f(Color.orange)));
        bg.addChild(SGBuilder.line("Y axis",
                new Point3d(0, -radius, 0),
                new Point3d(0, radius, 0),
                new Color3f(Color.white),
                new Color3f(Color.green)));
        bg.addChild(SGBuilder.line("Z axis",
                new Point3d(0, 0, -radius),
                new Point3d(0, 0, radius),
                new Color3f(Color.cyan),
                new Color3f(Color.blue)));
        return bg;
    }

    public static Background background(Color3f c) {
        return background(c, null, null);
    }

    public static Background background(Color3f c, BranchGroup bg,
                                        BoundingSphere bounds) {
        if (bounds == null) bounds = new BoundingSphere();
        Background background = new Background(bg);
        background.setCapability(background.ALLOW_GEOMETRY_READ);
        background.setCapability(background.ALLOW_COLOR_READ);
        background.setColor(c);
        background.setApplicationBounds(bounds);
        return background;
    }

    public static Primitive cone(String name, float base, float height) {
        Appearance ap = new Appearance();
        ap.setMaterial(new Material());
        Primitive p = new Cone(base, height, Cone.GENERATE_NORMALS, ap);
        p.setCapability(p.ALLOW_CHILDREN_READ);
        p.setUserData(name);
        return p;
    }

    public static DirectionalLight directionalLight(String name, Color3f c,
                                                    Vector3f direction,
                                                    BoundingSphere bounds) {
        DirectionalLight lightD = new DirectionalLight(true, c, direction);
        lightD.setInfluencingBounds(bounds);
        lightD.setUserData(name);
        return lightD;
    }

    public static Appearance frontSurface() {
        return appearance(PolygonAttributes.POLYGON_FILL,
                PolygonAttributes.CULL_BACK);
    }

    public static BranchGroup greatCircleAxes() {
        LineStripArray lsax = new LineStripArray(20,
                LineStripArray.COORDINATES |
                LineStripArray.COLOR_3,
                new int[]{20});
        LineStripArray lsay = new LineStripArray(20,
                LineStripArray.COORDINATES |
                LineStripArray.COLOR_3,
                new int[]{20});
        for (int i = 0; i < 20; i++) {
            double xOrY = i / 10.0 - 1;
            double zParam = i / 20.0 * Math.PI;
            double z = -Math.sin(zParam);
            Point3d px = new Point3d(xOrY, 0, z);
            Point3d py = new Point3d(0, xOrY, z);
            lsax.setCoordinate(i, px);
            lsay.setCoordinate(i, py);
            lsax.setColor(i, new Color3f(0.5f, 1f, 1f));
            lsay.setColor(i, new Color3f(0.5f, 1f, 1f));
        }

        Shape3D gcx = new Shape3D(lsax);
        Shape3D gcy = new Shape3D(lsay);

        BranchGroup bg = SGBuilder.newBranchGroup("Great circles");
        bg.addChild(gcx);
        bg.addChild(gcy);
        return bg;
    }

    public static Shape3D line(String name, Point3d start, Point3d end,
                               Color3f color1, Color3f color2) {
        LineArray la = new LineArray(2, LineArray.COORDINATES | LineArray.COLOR_3);
        la.setCoordinate(0, start);
        la.setCoordinate(1, end);
        la.setColor(0, color1);
        la.setColor(1, color2);
        Shape3D s = new Shape3D(la);
        s.setUserData(name);
        s.setCapability(s.ALLOW_APPEARANCE_READ);
        s.setCapability(s.ALLOW_GEOMETRY_READ);
        return s;
    }

    public static BranchGroup makePickable(String name, Node node,
                                           SimpleUniverse u, double radius) {
        if (name == null) name = "Pickable group";
        BranchGroup bg = newBranchGroup(name, node);

        node.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        node.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        node.setCapability(TransformGroup.ENABLE_PICK_REPORTING);

        PickTranslateBehavior pickTranslateBehavior =
                new PickTranslateBehavior(bg, u.getCanvas(),
                        new BoundingSphere(new Point3d(), radius));

        bg.addChild(pickTranslateBehavior);

        return bg;
    }

    public static BranchGroup newBranchGroup(String name) {
        BranchGroup bg = new BranchGroup();
        bg.setCapability(bg.ALLOW_CHILDREN_READ);
        bg.setUserData(name);
        return bg;
    }

    public static BranchGroup newBranchGroup(String name, Node n) {
        BranchGroup bg = newBranchGroup(name);
        bg.addChild(n);
        return bg;
    }

    public static void printBranchGroups(SimpleUniverse u) {
        Locale locale = u.getLocale();
        Enumeration branchGraphs = locale.getAllBranchGraphs();
        String indent = "";
        System.out.println("\n" + indent + u);
        printChildren(indent, branchGraphs);
    }

    public static void printChildren(String indent, Enumeration branchGraphs) {
        indent += "  ";
        while (branchGraphs.hasMoreElements()) {
            SceneGraphObject sgo = (SceneGraphObject) branchGraphs.nextElement();
            Object userData = sgo.getUserData();
            if (userData == null) userData = "<no label>";
            System.out.println(indent + userData + ": " + sgo);
            if (sgo instanceof Group) {
                Group group = (Group) sgo;
                if (group.getCapability(group.ALLOW_CHILDREN_READ))
                    printChildren(indent, group);
                else
                    System.out.println(indent + "  ALLOW_CHILDREN_READ capability not set");
            } else if (sgo instanceof Shape3D) {
                Shape3D s = (Shape3D) sgo;
                if (s.getCapability(s.ALLOW_APPEARANCE_READ))
                    System.out.println(indent + "  (appearance): " + s.getAppearance());
                else
                    System.out.println(indent + "  ALLOW_APPEARANCE_READ capability not set");
                if (s.getCapability(s.ALLOW_GEOMETRY_READ))
                    System.out.println(indent + "  (geometry): " + s.getAllGeometries().nextElement());
                else
                    System.out.println(indent + "  ALLOW_GEOMETRY_READ capability not set");
            }
        }
    }

    public static void printChildren(String indent, Group bg) {
        printChildren(indent, bg.getAllChildren());
    }

    public static Shape3D text3D(String text, String fontName, int style,
                                 int size, Color3f color, Point3f point) {
        Font3D font3d = new Font3D(new Font(fontName, style, size),
                new FontExtrusion());
        Text3D textGeom = new Text3D(font3d, new String(text), point);
        Shape3D textShape = new Shape3D(textGeom);
        Appearance ap = new Appearance();
        ap.setColoringAttributes(new ColoringAttributes(color, ColoringAttributes.SHADE_FLAT));
        textShape.setAppearance(ap);
        textShape.setCapability(textShape.ALLOW_APPEARANCE_READ);
        textShape.setCapability(textShape.ALLOW_GEOMETRY_READ);
        textShape.setUserData(text);
        return textShape;
    }

    public static Appearance wireFrame() {
        return appearance(PolygonAttributes.POLYGON_LINE,
                PolygonAttributes.CULL_NONE);
    }

}