/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;

import j3d.examples.particles.emitters.*;
import javax.vecmath.*;
import java.util.*;

/*
 * A circular motion influence that assumes it is very close
 * to emitted particles.  Particles that originate outside of the
 * vortex and then later enter it are not affected properly.
 */
public class Vortex implements IExternalInfluence, IParticleLifeCycleListener {
	private float angularVelocity;
	private float heightVelocity;
	private float innerRadius;
	private float innerRadiusSquared;
	private float outterRadius;
	private float outterRadiusSquared;
	private float spreadAngle;
	private float tangent;
	private Vector3f up;

	public Vortex(
		double spreadAngle,
		float innerRadius,
		float outterRadius,
		double angularVelocity,
		float heightVelocity) {
		this(
			(float) spreadAngle,
			innerRadius,
			outterRadius,
			(float) angularVelocity,
			heightVelocity);
	}

	public Vortex(
		float spreadAngle,
		float innerRadius,
		float outterRadius,
		float angularVelocity,
		float heightVelocity) {
		this.innerRadius = innerRadius;
		this.outterRadius = outterRadius;
		this.spreadAngle = spreadAngle;
		tangent = (float) Math.tan(spreadAngle);
		innerRadiusSquared = innerRadius * innerRadius;
		outterRadiusSquared = outterRadius * outterRadius;
		this.angularVelocity = angularVelocity;
		this.heightVelocity = heightVelocity;
		up = new Vector3f(0,1,0);
	}
	
	public void aboutToDie(List c, float dt) {
		//		Nothing to do
	}

	public void aboutToEmit(List aList, float dt) {
		// Assign a vortex velocity to the particles
		int aSize = aList.size();
		for(int i = 0; i < aSize; i++){	
			Particle aParticle = (Particle)aList.get(i);
			float[] position = aParticle.getLocalPosition();
			float x = position[0];
			float y = position[1];
			float z = position[2];
			float distanceSquared = x * x + z * z;
			float thisRadius = (float) Math.sqrt(distanceSquared);
			if (thisRadius >= outterRadius*1.05) {
				return;
			} else if (thisRadius < innerRadius) {
				return;
			}
			/* Vortex velocity has a magnitude of v = r*w
			 * tangential to the rotation.  The tangent can be 
			 * found by creating the radius vector and taking the 
			 * cross product with a vector pointing straight up.
			 * The result does not have the right magnitude, so 
			 * it is normalized and scaled by the product of the
			 * radius (based on the position) and the angular
			 * velocity of this vortex.
			 */ 
			Vector3f r = new Vector3f(x,y,z);
			r.normalize();
			Vector3f velocity = new Vector3f();
			velocity.cross(up, r);
			float tangentialVelocity = thisRadius * angularVelocity;
			velocity.scale(tangentialVelocity);
			velocity.y = heightVelocity;
			// I haven't found a good way to allow the combining of the 
			// initial velocity set by the emitter and the rotational
			// velocity of the vortex.
			//velocity.add(aParticle.getLocalVelocity());
			aParticle.setLocalVelocity(velocity);
		}
	}

	public void apply(Particle aParticle, float dt) {
		float[] position = aParticle.getLocalPosition();
		float x = position[0];
		//float y = position[1];
		float z = position[2];
		float distanceSquared = x * x + z * z;

		float thisRadius = (float) Math.sqrt(distanceSquared);
		if (thisRadius >= outterRadius*1.05) {
			// Consider a small region beyond the outter radius because
			// of small variations of the recalculated radius.
			return;
		} else if (thisRadius < innerRadius*0.95) {
			// Consider a small region beyond the inner radius because
			// of small variations of the recalculated radius.
			return;
		}
		/*
		 * Rotational particle movement always has an acceleration
		 * toward the center of the vortex due to the change in  
		 * direction of the motion.  This acceleration is called
		 * centripetal acceleration.  This influence assumes the rate
		 * of change in speed (magnitude of the tangential velocity)
		 * is zero making the tangential acceleration zero.  
		 * Centripetal acceleration is defined as the speed
		 * squared divided by the radius.  Refer to a good physics
		 * book for more details.
		 */
		float dRadius = heightVelocity * dt * tangent;
		Vector3f velocity = aParticle.getLocalVelocity();
		float velocity2 = velocity.x * velocity.x + velocity.z * velocity.z;
		float newCentripetal = velocity2/(thisRadius + dRadius);
		Vector3f centripetal = new Vector3f(-x, 0, -z);
		centripetal.normalize(); // get the direction
		centripetal.scale(newCentripetal);
		aParticle.addLocalAcceleration(centripetal);
	}

	public void initializeParticle(Particle aParticle) {
		// Nothing to do
	}

	public void updated(List c, float dt) {
		// Nothing to do	
	}

}
