/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;

import j3d.examples.particles.emitters.*;
import java.util.*;

import javax.vecmath.*;
import javax.media.j3d.*;
/*
 * Light the particle with a point light source.
 */
public class EmitShapeLight extends Shape3DInfluence implements IExternalInfluence, IParticleLifeCycleListener {
	private ParticleAgeAlpha alpha;
	private Color3f color;
	private float constant, linear, quadradic; // attenuation factors
	private Map lightMap;
	/**
	 * Emit a burst of white light and slowly decrease the intensity
	 * as the particle ages.  Use the attentuation parameters
	 * (constant, linear, quadradic) to control the attenuation
	 * of the point lights.  Note: This implementation uses a
	 * point light for each particle which can have significant
	 * performance issues with a large number or if the lifespan
	 * of the particle is short (less than 1 or 2 seconds).
	 */
	public EmitShapeLight(float aConstant, float aLinear, float aQuadradic){
		/* 
		 * The inverse sum of the attenuation parameters of a 
		 * point light are used to determine how much attentuation
		 * occurs.  Very small attenuation parameters yield a very
		 * bright point light.  
		 */
		
		this(new ParticleAgeAlpha(0,0,0,1),new Color3f(1,1,1), aConstant, aLinear, aQuadradic);
	}
	/**
	 * Emit a burst of colored light and slowly decrease the intensity
	 * as the particle ages.  Use the attentuation parameters
	 * (constant, linear, quadradic) to control the attenuation
	 * of the point lights.  Note: This implementation uses a
	 * point light for each particle which can have significant
	 * performance issues with a large number or if the lifespan
	 * of the particle is short (less than 1 or 2 seconds).
	 */
	public EmitShapeLight(Color3f aColor, float aConstant, float aLinear, float aQuadradic){
		/* 
		 * The inverse sum of the attenuation parameters of a 
		 * point light are used to determine how much attentuation
		 * occurs.  Very small attenuation parameters yield a very
		 * bright point light.  
		 */
		this(new ParticleAgeAlpha(0,0,0,1), aColor, aConstant, aLinear, aQuadradic);
	}
	/**
	 * Emit a point light based on the particle age alpha.  One minus the 
	 * alpha value is used for the attenuation parameters, so you can
	 * think of the alpha value as the brightness of the point light.
	 */
	public EmitShapeLight(ParticleAgeAlpha anAlpha, Color3f aColor, float aConstant, float aLinear, float aQuadradic){
		alpha = anAlpha;
		color = aColor;
		constant = aConstant;
		linear = aLinear;
		quadradic = aQuadradic;
	}
	
	private Map getLightMap(){
		if(lightMap == null){
			lightMap = new HashMap();
		}
		return lightMap;
	}
	
	private PointLight getLight(Particle aParticle){
		PointLight light = (PointLight) getLightMap().get(aParticle);
		return light;
	}
	
	public void initializeParticle(Particle aParticle){
		if(isCompatible(aParticle)){
			PointLight point = new PointLight();
			point.setColor(color);
			Shape3D shape = getShape3D(aParticle);
			point.setInfluencingBounds(shape.getBounds());
			point.setEnable(false);
			point.setCapability(PointLight.ALLOW_STATE_WRITE);
			point.setCapability(PointLight.ALLOW_POSITION_WRITE);
			point.setCapability(PointLight.ALLOW_ATTENUATION_WRITE);
			Map map = getLightMap();
			map.put(aParticle, point);
			BranchGroup parent = getBranchGroup(aParticle);
			parent.addChild(point);
		}
	}
	
	public void apply(Particle aParticle, float dt) {
		if(isCompatible(aParticle)){
			float[] position = aParticle.getWorldPosition();
			PointLight light = getLight(aParticle);
			if(light != null){
				light.setPosition(position[0], position[1], position[2]);
				float value = 1 - alpha.value(aParticle);
				light.setAttenuation(constant*value, linear*value, quadradic*value);
			}
		}
	}
	
	public void aboutToDie(List aList, float dt) {
		int aSize = aList.size();
		for(int i = 0; i < aSize; i++){	
			Particle aParticle = (Particle)aList.get(i);
			PointLight light = getLight(aParticle);
			if(light != null){
				light.setEnable(false);
				light.setAttenuation(1, 0, 0); // default J3D value
			}
		}
	}

	public void aboutToEmit(List aList, float dt) {
		int aSize = aList.size();
		for(int i = 0; i < aSize; i++){	
			Particle aParticle = (Particle)aList.get(i);
			PointLight light = getLight(aParticle);
			if(light != null){
				light.setEnable(true);
			}
		}
	}
	
	public void updated(List c, float dt) {
		// Nothing to do	
	}

}
