/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;
import javax.vecmath.*;
import javax.media.j3d.*;

import j3d.examples.particles.emitters.Particle;
/**
 * A simulation of Newton�s Law of Restitution for 
 * Instantaneous Collisions.  Assumes a mostly elastic impulse 
 * collision between the "ground" and the particle based on the 
 * coefficient of restitution in the world Y coordinate system.  
 * Also assumes a simple friction model to slow down the particle 
 * in the world X and Z planes. 
 */
// TODO: Find a way to eliminate the conflict with gravity for long living particles (continues to squirm)
public class Bounce implements IExternalInfluence {
	private static float COEFFICIENT_OF_RESTITUTION = 0.5f;
	private static float DRAG = 0.5f;
	private BoundingBox bounds;
	private float drag;
	private float restitution;

	public Bounce() {
		this(
			new BoundingBox(
				new Point3d(-500, -500, -500),
				new Point3d(500, 0, 500)));
	}

	public Bounce(BoundingBox aBounds) {
		this(aBounds, DRAG, COEFFICIENT_OF_RESTITUTION);
	}

	public Bounce(BoundingBox aBounds, float aDrag, float aRestitution) {
		Point3d upper = new Point3d();
		aBounds.getUpper(upper);
		bounds = aBounds;
		drag = aDrag;
		restitution = aRestitution;
	}

	public void initializeParticle(Particle aParticle) {
		// Nothing to do 
	}

	/* (non-Javadoc)
	 * @see particles.influences.IExternalInfluence#apply(particles.emitters.Particle, float)
	 */
	public void apply(Particle aParticle, float dt) {
		float[] position = aParticle.getWorldPosition();
		Vector3f velocity = aParticle.getWorldVelocity();
		float[] future = new float[3];
		future[0] = position[0] + velocity.x * dt;
		future[1] = position[1] + velocity.y * dt;
		future[2] = position[2] + velocity.z * dt;
		Point3d point = new Point3d(future[0], future[1], future[2]);
		if (bounds.intersect(point)) {
			/* 
			 * The point has interpenetration with the ground.				
			 * Back up the full time interval and make the
			 * position a reflection off of the ground.  To be
			 * more precise, we should calculate the 'exact'
			 * interpenetration time (+/- some buffer), back up
			 * with the old velocity to the 'exact' time and
			 * then move the particle forward for the left over
			 * time difference with the new velocity calculated
			 * below.  Because the animation is dealing with a
			 * dt of ~20 milliseconds based on the particle 
			 * system manager, and we are dealing with our eyes
			 * and not a physical simulation, our approach here
			 * is good enough.  
			 */
			float intersectX = position[0] + velocity.x * dt;
			float intersectY = position[1] - velocity.y * dt;
			float intersectZ = position[2] + velocity.z * dt;
			
			/* 
			 * Adjust the velocity due to the collision.  Because the
			 * ground is assumed to have infinite mass, is stationary
			 * and has a normal pointing straight up in the y direction, 
			 * the impulse affect in the linear velocity is simple.
			 * See Chris Hecker's March 1997 Game Developer Magazine
			 * article.  The lines below implement equation 5 while
			 * also simulating the drag of friction.
			 */
			velocity.x = DRAG * velocity.x;
			velocity.y = -COEFFICIENT_OF_RESTITUTION * velocity.y;
			velocity.z = DRAG * velocity.z;
			aParticle.setWorldVelocity(velocity);

			aParticle.setWorldPosition(intersectX, intersectY, intersectZ);
		}
	}

}
