/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.examples;
import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.GraphicsConfiguration;

import javax.media.j3d.*;
import javax.vecmath.*;

import j3d.examples.particles.*;
import j3d.examples.particles.emitters.*;
import j3d.examples.particles.generationshapes.*;
import j3d.examples.particles.influences.*;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.behaviors.mouse.*;
import com.sun.j3d.utils.universe.SimpleUniverse;
/*
 * This example demonstrates the fundamental use of a particle emitter,
 * generation shape, point particle system (no motion blur), and one
 * external influence (fading the particle as it ages).
 */
public class PointParticleSystemExample extends Applet {
	
	public static void main(String[] args) {
		new MainFrame(new PointParticleSystemExample(false), 500, 500);
	}
	private Canvas3D canvas3D;
	private ParticleSystem particleSystem;
	private boolean recording;
	
	public PointParticleSystemExample(boolean recording) {
		initialize(recording);
	}
	
	public BranchGroup createSceneGraph(SimpleUniverse su) {
		BranchGroup objRoot = new BranchGroup();
		XZPlane plane = new XZPlane(5, 10);
		objRoot.addChild(plane);
		BoundingSphere bounds = new BoundingSphere();
		bounds.setRadius(1000);
		// Allow the particle system to be rotated with the mouse
		MouseRotate mouseRotate = new MouseRotate();
		mouseRotate.setSchedulingBounds(bounds);
		
		ParticleEmitter pe =  new ParticleEmitter(
						new PointGenerationShape(Math.PI/4),
						400, 	// emission rate
						50, 	// emission rate variance
						8.5f, 	// velocity
						2, 		// velocity variance
						3, 		// lifetime
						1,		// lifetime variance
						1500	// emitter lifetime in seconds
						);
		pe.addInfluence(new FadePoint());
			
		particleSystem = new PointParticleSystem(pe, new Color3f(0,1,0));
		System.out.println("Move the mouse with the left button down to rotate the particle system");
		TransformGroup psGroup = new TransformGroup();
		psGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
		psGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3f(0, 5, 0));
		psGroup.setTransform(t3d);
		mouseRotate.setTransformGroup(psGroup);
		psGroup.addChild(mouseRotate);	
		psGroup.addChild(particleSystem);
		objRoot.addChild(psGroup);
		
		// Add ambient light
		AmbientLight ambient = new AmbientLight();
		ambient.setColor(new Color3f(1f, 1f, 1f));
		ambient.setInfluencingBounds(bounds);
		objRoot.addChild(ambient);

		// Add a directional light
		DirectionalLight directional = new DirectionalLight();
		directional.setDirection(1.0f, -0.4f, 1f);
		directional.setColor(new Color3f(0.7f, 0.7f, 0.7f));
		directional.setInfluencingBounds(bounds);
		objRoot.addChild(directional);		

		// Add a behavior to manage the particle system animation
		ParticleSystemManager manager = ParticleSystemManager.getCurrent();
		manager.setSchedulingBounds(bounds);
		objRoot.addChild(manager);
		
		// Add a keyboard navigator to move around
		TransformGroup vpTrans =
			su.getViewingPlatform().getViewPlatformTransform();
		KeyNavigatorBehavior keyNavBeh =
			new KeyNavigatorBehavior(vpTrans);
		keyNavBeh.setSchedulingBounds(bounds);
		objRoot.addChild(keyNavBeh);
		
		// Add an image capturing behavior
		if(recording){
			ImageCaptureBehavior capture = new ImageCaptureBehavior(canvas3D, 0, 200);
			capture.setSchedulingBounds(bounds);
			objRoot.addChild(capture);
		}
		
		// Optimize the scene graph
		objRoot.compile();
		return objRoot;
	}
	private void initialize(boolean recording) {
		this.recording = recording;
		setLayout(new BorderLayout());
		GraphicsConfiguration config =
			SimpleUniverse.getPreferredConfiguration();
		canvas3D = new Canvas3D(config);
		add("Center", canvas3D);
		SimpleUniverse simpleU = new SimpleUniverse(canvas3D);
		
		//Position the view
		TransformGroup viewingPlatformGroup =
			simpleU.getViewingPlatform().getViewPlatformTransform();
		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3f(0, 10, 60));
		viewingPlatformGroup.setTransform(t3d);

		BranchGroup scene = createSceneGraph(simpleU);
		simpleU.addBranchGraph(scene);

		canvas3D.getView().setBackClipDistance(300.0d);
		canvas3D.getView().setFrontClipDistance(0.1d);
		canvas3D.getView().setMinimumFrameCycleTime(20);
	}

}
