/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.examples;
import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.GraphicsConfiguration;

import javax.media.j3d.*;
import javax.vecmath.*;

import j3d.examples.particles.*;
import j3d.examples.particles.emitters.*;
import j3d.examples.particles.generationshapes.*;
import j3d.examples.particles.influences.*;
import j3d.examples.particles.shapes.*;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.universe.SimpleUniverse;

/**
 * An example showing the use of the Attract influence with
 * the Rock object to create an black hole sucking up an
 * asteriod field.  The blackhole shrinks the shapes as it
 * approaches the event horizon.  This one is strong allowing
 * no rocks to escape.  
 */
public class BlackHoleExample extends Applet {

	public static void main(String[] args) {
		new MainFrame(new BlackHoleExample(false), 600, 600);
	}
	private Canvas3D canvas3D;
	private boolean recording;
	
	public BlackHoleExample(boolean recording) {
		this.recording = recording;
		initialize();
	}
	
	public BranchGroup createSceneGraph(SimpleUniverse su) {
		BranchGroup objRoot = new BranchGroup();
		
		XZPlane plane = new XZPlane(5, 10); // 50 by 50 
		objRoot.addChild(plane);
		float rate = (float) Math.PI;
		Vector3f particleRotationRate = new Vector3f(rate, rate, rate);
		Vector3f particleRotationRateVariance = new Vector3f(rate, rate, rate);

		ParticleEmitter pe =  new ParticleEmitter(
						new RadialGenerationShape(80, 0),
						40f, 	// emission rate
						0f, 	// emission rate variance
						1, 	// velocity
						0.5f, 	// velocity variance
						particleRotationRate,
						particleRotationRateVariance,
						30f, 	// lifetime
						0f,	// lifetime variance
						120		// emitter lifetime in seconds
						);
		Point3d holeLocation = new Point3d(0, 0, 0);
		BoundingSphere holeBounds = new BoundingSphere(holeLocation, 40);// outter radius = 20 
		pe.addInfluence(new Attract(50000, holeLocation, 5)); // inner radius
		// shrink object within outter radius of black hole
		pe.addInfluence(new BoundedScale(1, -1, holeBounds));
		
		IShape3DFactory factory = new RockFactory(1, 0.7f);			
		Shape3DParticleSystem particleSystem = new Shape3DParticleSystem(pe, factory);

		TransformGroup psGroup = new TransformGroup();
		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3f(0, 35, 0));
		psGroup.setTransform(t3d);
		psGroup.addChild(particleSystem);
		objRoot.addChild(psGroup);

		BoundingSphere bounds = new BoundingSphere();
		bounds.setRadius(500);
		
		FuzzBall hole = new FuzzBall(10, new Color3f(0,0,0));
		TransformGroup holeGroup = new TransformGroup();
		Transform3D holeTransform = new Transform3D();
		holeTransform.setTranslation(new Vector3f(0, 0, 0));
		holeGroup.setTransform(holeTransform);
		holeGroup.addChild(hole);
		objRoot.addChild(holeGroup);
		
		Background background = new Background();
		background.setApplicationBounds(bounds);
		background.setColor(0.05f, 0.05f, 0.05f);
		objRoot.addChild(background);
		
		AmbientLight ambient = new AmbientLight();
		ambient.setColor(new Color3f(1f, 1f, 1f));
		ambient.setInfluencingBounds(bounds);
		objRoot.addChild(ambient);

		// Add a directional light
		DirectionalLight directional = new DirectionalLight();
		directional.setDirection(1.0f, -0.8f, 1f);
		directional.setColor(new Color3f(1f, 1f, 1f));
		directional.setInfluencingBounds(bounds);
		objRoot.addChild(directional);		

		// Add a behavior to manage the particle system animation
		ParticleSystemManager manager = ParticleSystemManager.getCurrent();
		manager.setSchedulingBounds(bounds);
		objRoot.addChild(manager);
		
		// Add a keyboard navigator to move around
		TransformGroup vpTrans =
			su.getViewingPlatform().getViewPlatformTransform();
		KeyNavigatorBehavior keyNavBeh =
			new KeyNavigatorBehavior(vpTrans);
		keyNavBeh.setSchedulingBounds(bounds);
		objRoot.addChild(keyNavBeh);
		
		if(recording){
			ImageCaptureBehavior capture = new ImageCaptureBehavior(canvas3D, 0, 300);
			capture.setSchedulingBounds(bounds);
			objRoot.addChild(capture);
		}
		
		// Optimize the scene graph
		objRoot.compile();
		return objRoot;
	}
	private void initialize() {
		setLayout(new BorderLayout());
		GraphicsConfiguration config =
			SimpleUniverse.getPreferredConfiguration();
		canvas3D = new Canvas3D(config);
		add("Center", canvas3D);
		SimpleUniverse simpleU = new SimpleUniverse(canvas3D);
		
		//Position the view
		TransformGroup viewingPlatformGroup =
			simpleU.getViewingPlatform().getViewPlatformTransform();
		Transform3D t3d = new Transform3D();
		t3d.rotX(-Math.PI/8);
		t3d.setTranslation(new Vector3f(0, 40, 80));
		viewingPlatformGroup.setTransform(t3d);

		BranchGroup scene = createSceneGraph(simpleU);
		simpleU.addBranchGraph(scene);

		canvas3D.getView().setBackClipDistance(300.0d);
		canvas3D.getView().setFrontClipDistance(0.1d);
		canvas3D.getView().setMinimumFrameCycleTime(20);	
		canvas3D.getView().setTransparencySortingPolicy(View.TRANSPARENCY_SORT_GEOMETRY);	
	}

}
