/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles;

import javax.vecmath.*;
import javax.media.j3d.*;
import java.util.*;

import j3d.examples.particles.emitters.*;
import j3d.examples.particles.shapes.*;

/**
 * A particle system that uses any <code>Shape3D</code> as a particle.
 * This particle system uses the user data of each Shape3D particle to
 * maintain scene graph information.  The collision bounds of the shape
 * is assigned by the shape factory based on the radius of the particle.
 * The collision bounds is a BoundingBox. 
 */

public class Shape3DParticleSystem extends Group implements IParticleSystem, IParticleLifeCycleListener {
	private ParticleEmitter emitter;
	private IShape3DFactory factory;
	private ArrayList shapes;
	Transform3D transform3D;
	// Kept around to reduce object creation
	Vector3f translation;
	
	public Shape3DParticleSystem(ParticleEmitter anEmitter, IShape3DFactory aFactory){
		emitter = anEmitter;
		emitter.addMonitor(this);
		emitter.setParticleSystem(this);
		factory = aFactory;
		translation = new Vector3f();
		transform3D = new Transform3D();
		getShapes();
		setCapabilities();
		ParticleSystemManager.getCurrent().register(this);
	}
	
	/* (non-Javadoc)
	 * @see particles.emitters.IParticleLifeCycleListener#aboutToDie(java.util.Collection, float)
	 */
	public void aboutToDie(List aList, float dt){
		int aSize = aList.size();
		for(int i = 0; i < aSize; i++){	
			Particle aParticle = (Particle)aList.get(i);
			removeChild(getBranchGroup(aParticle));		
		}
	}
	
	/* (non-Javadoc)
	 * @see particles.emitters.IParticleLifeCycleListener#aboutToEmit(java.util.Collection, float)
	 */
	public void aboutToEmit(List aList, float dt){
		int aSize = aList.size();
			for(int i = 0; i < aSize; i++){
			Particle aParticle = (Particle)aList.get(i);
			BranchGroup aGroup = refresh(aParticle);
			addChild(aGroup);		
		}
	}
	
	public BranchGroup getBranchGroup(Particle aParticle){
		TransformGroup aGroup = getTransformGroup(aParticle);
		BranchGroup aBG = (BranchGroup) aGroup.getUserData();
		return aBG;
	}
	
	protected int getMaximumParticles(){
		return emitter.getMaximumParticles();
	}

	public Shape3D getShape(Particle aParticle){
		int indice = aParticle.getIndice();
		Shape3D aShape = (Shape3D)getShapes().get(indice);
		TransformGroup aTransformGroup = (TransformGroup) aShape.getUserData();
		if(aTransformGroup == null){
			aTransformGroup = new TransformGroup();
			aTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
			aTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
			aTransformGroup.addChild(aShape);
			aShape.setUserData(aTransformGroup);
		}
		BranchGroup aBranchGroup = (BranchGroup) aTransformGroup.getUserData();
		if(aBranchGroup == null){
			aBranchGroup = new BranchGroup();
			aBranchGroup.setCapability(BranchGroup.ALLOW_DETACH);
			aBranchGroup.addChild(aTransformGroup);
			aTransformGroup.setUserData(aBranchGroup);
			aBranchGroup.setUserData(aParticle);
		}
		
		return aShape;
	}
	
	protected ArrayList getShapes(){
		if(shapes == null){
			shapes = new ArrayList(getMaximumParticles());
			initializeShapes();
		}
		return shapes;
	}
	
	
	public TransformGroup getTransformGroup(Particle aParticle){
		Shape3D aShape = getShape(aParticle);
		TransformGroup aTG = (TransformGroup) aShape.getUserData();
		return aTG;
	}
	
	protected void initializeShapes(){
		for(int i=0; i < getMaximumParticles(); i++){
			Shape3D aShape = factory.createShape();
			shapes.add(i,aShape);
		}
	}

	public boolean isAlive() {
		return emitter.isAlive();
	}

	public boolean isDead() {
		return emitter.isDead();
	}
	public void nextFrame(float dt) {
		emitter.update(dt);
	}
	
	private BranchGroup refresh(Particle aParticle){
		/*
		 * Refresh the Shape and transform from the 
		 * existing state of the particle.
		 */
		Shape3D aShape = getShape(aParticle);
		TransformGroup aTransformGroup = (TransformGroup) aShape.getUserData();
		translation.set(aParticle.getLocalPosition());

		// Change the transform (position, scale and rotation) of the shape.
		if(aParticle.isRotatable()){
			transform3D.set(aParticle.getOrientation(),translation,aParticle.getScale());
		}
		else{
			transform3D.set(aParticle.getScale(), translation);
		}
		aTransformGroup.setTransform(transform3D);
		
		return (BranchGroup)aTransformGroup.getUserData();
	}
	
	protected void setCapabilities() {
		setCapability(Group.ALLOW_CHILDREN_EXTEND);
		setCapability(Group.ALLOW_CHILDREN_WRITE);
		setCapability(Node.ALLOW_LOCAL_TO_VWORLD_READ);
		setPickable(true);
	}
	
	/* (non-Javadoc)
	 * @see particles.emitters.IParticleLifeCycleListener#updated(java.util.Collection, float)
	 */
	public void updated(List aList, float dt){
		/*
		 * The collection of particles have been updated
		 * by the influences and moved and rotated.  Map
		 * the change in the particle with the shape
		 * maintined by this particle system.
		 */

		int aSize = aList.size();
		for(int i = 0; i < aSize; i++){	
			Particle aParticle = (Particle)aList.get(i);
			refresh(aParticle);	
		}
	}

}
