/**********************************************************
 Copyright (C) 2001 	Daniel Selman

 First distributed with the book "Java 3D Programming"
 by Daniel Selman and published by Manning Publications.
 http://manning.com/selman

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 The license can be found on the WWW at:
 http://www.fsf.org/copyleft/gpl.html

 Or by writing to:
 Free Software Foundation, Inc.,
 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 Authors can be contacted at:
 Daniel Selman: daniel@selman.org

 If you make changes you think others would like, please
 contact one of the authors or someone at the
 www.j3d.org web site.
 **************************************************************/

package j3d.examples.avatar;

import com.sun.j3d.audioengines.javasound.JavaSoundMixer;
import com.sun.j3d.utils.geometry.Primitive;
import com.sun.j3d.utils.geometry.Sphere;
import com.sun.j3d.utils.image.TextureLoader;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.universe.ViewerAvatar;
import j3d.examples.common.ComplexBranchGroup;
import j3d.examples.common.Land;
import utils.ResourceManager;

import javax.media.j3d.*;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import java.applet.Applet;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * Displays a simple driving type game scene, using texture mapped cubes.
 * It assigns an Avatar to the viewer and incorporates simple sounds and
 * collision detection/notification/
 * <p/>
 * This example does not use the Java3dApplet base class but is based on a
 * SimpleUniverse construction instead. that way we can illustrate the
 * setPlatformGeometry call.
 */
public class Main extends Applet {

    public BranchGroup createSceneGraph() {
        BranchGroup bg = new BranchGroup();

        TransformGroup tgRoot = addBehaviors(bg);

        createBuildings(tgRoot);
        createRoad(tgRoot);
        createLand(tgRoot);
        createCars(tgRoot);
        createBackground(bg);

        return bg;
    }

    public void createBackground(Group bg) {
        // add the sky backdrop
        Background back = new Background();
        back.setApplicationBounds(getBoundingSphere());
        bg.addChild(back);

        BranchGroup bgGeometry = new BranchGroup();

        // create an appearance and assign the texture image
        Appearance app = new Appearance();
        Texture tex = new TextureLoader(
                ResourceManager.getResourceManager()
                .getImageFile("back.jpg")
                .getAbsolutePath(),
                this).getTexture();
        app.setTexture(tex);

        Sphere sphere = new Sphere(
                1.0f,
                Primitive.GENERATE_TEXTURE_COORDS |
                Primitive.GENERATE_NORMALS_INWARD,
                app);

        bgGeometry.addChild(sphere);
        back.setGeometry(bgGeometry);
    }


    public Group createLand(Group g) {
        Land land = new Land(
                this,
                g,
                ComplexBranchGroup.GEOMETRY | ComplexBranchGroup.TEXTURE);
        return land.createObject(
                new Appearance(),
                new Vector3d(0, 0, 0),
                new Vector3d(1, 1, 1),
                getImage("land.jpg"),
                null,
                null);
    }

    public static String getImage(String fn) {
        return
                ResourceManager.getResourceManager()
                .getImageFile(fn)
                .getAbsolutePath();
    }

    public Group createRoad(Group g) {
        Road road = new Road(
                this,
                g,
                ComplexBranchGroup.GEOMETRY | ComplexBranchGroup.TEXTURE);
        return road.createObject(
                new Appearance(),
                new Vector3d(0, 0, 0),
                new Vector3d(1, 1, 1),
                getImage("road.jpg"),
                null,
                null);
    }


    private float getRandomNumber(float basis, float random) {
        return basis + ((float) Math.random() * random * 2) - (random);
    }

    public Group createBuildings(Group g) {
        BranchGroup bg = new BranchGroup();

        for (int n = (int) Road.ROAD_LENGTH; n < 0; n = n + 10) {
            Building building = new Building(
                    this,
                    bg,
                    ComplexBranchGroup.GEOMETRY |
                    ComplexBranchGroup.TEXTURE |
                    ComplexBranchGroup.COLLISION);

            building.createObject(
                    new Appearance(),
                    new Vector3d(
                            getRandomNumber(-4.0f, 0.25f),
                            getRandomNumber(1.0f, 0.5f),
                            getRandomNumber(n, 0.5f)),
                    new Vector3d(1, 1, 1),
                    getImage("house.jpg"),
                    null,
                    null);

            building =
                    new Building(
                            this,
                            bg,
                            ComplexBranchGroup.GEOMETRY |
                    ComplexBranchGroup.TEXTURE |
                    ComplexBranchGroup.COLLISION);

            building.createObject(
                    new Appearance(),
                    new Vector3d(
                            getRandomNumber(4.0f, 0.25f),
                            getRandomNumber(1.0f, 0.5f),
                            getRandomNumber(n, 0.5f)),
                    new Vector3d(1, 1, 1),
                    getImage("house.jpg"),
                    null,
                    null);

        }

        g.addChild(bg);

        return bg;
    }


    public Group createCars(Group g) {
        BranchGroup bg = new BranchGroup();

        for (int n = (int) Road.ROAD_LENGTH; n < 0; n = n + 10) {
            Car car = new Car(
                    this, bg, ComplexBranchGroup.GEOMETRY |
                    ComplexBranchGroup.TEXTURE |
                    ComplexBranchGroup.SOUND);

            car.createObject(
                    new Appearance(),
                    new Vector3d(
                            getRandomNumber(0.0f, 2.0f),
                            Car.CAR_HEIGHT / 2.0f,
                            getRandomNumber(n, 5.0f)),
                    new Vector3d(1, 1, 1),
                    getImage("car0.jpg"),
                    getAudio("car.wav"),
                    getAudio("collide.wav"));
        }

        g.addChild(bg);
        return bg;
    }

    public String getAudio(String s) {
        try {
            URL url = ResourceManager.getResourceManager()
                    .getAudioFile(s)
                    .toURL();
            return url.toString();
        } catch (MalformedURLException e) {
            e.printStackTrace();

        }
        return null;
    }

    public TransformGroup addBehaviors(Group bgRoot) {
        // Create the transform group node and initialize it to the
        // identity.  Enable the TRANSFORM_WRITE capability so that
        // our behavior code can modify it at runtime.  Add it to the
        // root of the subgraph.
        TransformGroup objTrans = new TransformGroup();
        objTrans.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        Transform3D zAxis = new Transform3D();
        zAxis.rotY(Math.toRadians(90.0));

        Alpha zoomAlpha = new Alpha(
                -1, Alpha.INCREASING_ENABLE,
                0, 0,
                20000, 0, 0,
                0, 0, 0);

        PositionInterpolator posInt = new PositionInterpolator(
                zoomAlpha,
                objTrans,
                zAxis,
                0,
                -160);

        posInt.setSchedulingBounds(getBoundingSphere());
        objTrans.addChild(posInt);

        bgRoot.addChild(objTrans);

        return objTrans;
    }

    BoundingSphere getBoundingSphere() {
        return new BoundingSphere(new Point3d(0.0, 0.0, 0.0), 400.0);
    }

    ViewerAvatar createAvatar() {
        ViewerAvatar va = new ViewerAvatar();
        TransformGroup tg = new TransformGroup();

        Car car = new Car(
                this, tg, ComplexBranchGroup.GEOMETRY |
                ComplexBranchGroup.TEXTURE |
                ComplexBranchGroup.COLLISION |
                ComplexBranchGroup.COLLISION_SOUND);

        car.createObject(
                new Appearance(),
                new Vector3d(0, -0.3, -0.3),
                new Vector3d(0.3, 0.3, 1),
                getImage("platform.jpg"),
                null,
                getAudio("collide.wav"));

        tg.addChild(car);
        va.addChild(tg);

        return va;
    }

    public static void main(String[] args) {
        Main main = new Main();

        // Create a simple scene and attach it to the virtual universe
        SimpleUniverse u = new SimpleUniverse();

        PhysicalEnvironment physicalEnv = u.getViewer()
                .getPhysicalEnvironment();

        TransformGroup tg = u.getViewer()
                .getViewingPlatform()
                .getViewPlatformTransform();

        Transform3D t3d = new Transform3D();
        t3d.set(new Vector3f(0, 0.5f, 0));
        tg.setTransform(t3d);

        CarSteering keys = new CarSteering(tg);
        keys.setSchedulingBounds(main.getBoundingSphere());

        u.getViewer().setAvatar(main.createAvatar());

        if (physicalEnv != null) {
            JavaSoundMixer javaSoundMixer = new JavaSoundMixer(
                    physicalEnv);

            if (javaSoundMixer == null)
                System.out.println("Unable to create AudioDevice.");

            javaSoundMixer.initialize();
        }

        // Add everthing to the scene graph - it will now be displayed.
        BranchGroup scene = main.createSceneGraph();
        scene.addChild(keys);


        u.addBranchGraph(scene);


        u.getViewingPlatform().getViewPlatform().setActivationRadius(2);
    }
}
	
