package j3d.examples.appearance.texture.noise;

import javax.media.j3d.*;

/*
 * RidgedFractalTerrain is a Surface subclass that demonstrates 
 * a generated terrain.  This implementation uses Perlin noise 
 * to create a geometry that appears to be terrain with a varying
 * degree of roughness (multifractal).
 */

public class RidgedFractalTerrain extends Surface {
    private static int NUMBER_OF_COLORS = 27;
    /* The noise zoom controls how much detail is to be
     * included.  A very high (>1000) zoom factor reveals
     * less variation and is suitable for large expanses.
     * Smaller zoom factors (in the 100's) has more detail
     * because more noise space is used.  This appears to
     * be the sweet spot for reasonable terrains.  Values
     * less than 100 result in so much noise space that
     * the result is not very natural for a terrain.
     */
    protected static float NOISE_ZOOM = 372.7f;
    private float minimumHeight = 100, maximumHeight = -100;

    public RidgedFractalTerrain(float sideLength, int divisions) {
        super(sideLength, divisions);
    }

    public void nextFrame(int ticks) {
        // noop
    }

    protected float calculateHeight(double x, double y, double z) {
        // Transform from world space to noise space.
        double s = x / NOISE_ZOOM;
        double t = y / NOISE_ZOOM;
        double r = z / NOISE_ZOOM;
        /* The offset and gain have a large impact on the resulting
         * terrain.  The offset should be around 1.0.
         */
        double height = ImprovedNoise.ridged(s, t, r, 8, 0.85, 1.9);
        if (height > maximumHeight)
            maximumHeight = (float) height;
        if (height < minimumHeight)
            minimumHeight = (float) height;
        return (float) height;
    }

    protected Appearance createAppearance() {
        Appearance appearance = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
        polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);
        // Setting back face normal flip to true allows backfacing
        // polygons to be lit (normal is facing wrong way) but only
        // if the normal is flipped.
        polyAttrib.setBackFaceNormalFlip(true);
        appearance.setPolygonAttributes(polyAttrib);
        Material material = new Material();
        material.setAmbientColor(0f, 0f, 0f);
        // Changing the specular color reduces the shine
        // that occurs with the default setting.
        material.setSpecularColor(0.1f, 0.1f, 0.1f);
        appearance.setMaterial(material);
        return appearance;
    }

    protected void customizeGeometry(IndexedTriangleStripArray itsa) {
        // Normalize the heights (y values)
        // System.out.println("Max: " + maximumHeight + " Min: " + minimumHeight);
        float range = maximumHeight - minimumHeight;
        float[] coordinates = itsa.getCoordRefFloat();
        for (int row = 0; row < (divisions + 1); row++) {
            int width = row * (divisions + 1);
            for (int col = 0; col < (divisions + 1); col++) {
                int ci = (col + width) * 3;
                float y = coordinates[ci + 1];
                coordinates[ci + 1] = 100f * (y - minimumHeight) / range;
            }
        }
        // Assign the colors and color coordinates
        itsa.setColorRefFloat(getElevationColors(coordinates.length));
        int[] colorIndices = new int[divisions * (divisions + 1) * 2];
        int index = 0;
        for (int row = 0; row < divisions; row++) {
            int width = row * (divisions + 1);
            for (int col = 0; col < (divisions + 1); col++) {
                // NW
                float nw = coordinates[(col + width + divisions + 1) * 3 + 1];
                colorIndices[index + 0] = getElevationColorIndex(nw, row + 1, col);
                ;
                // SW
                float sw = coordinates[(col + width) * 3 + 1];
                colorIndices[index + 1] = getElevationColorIndex(sw, row, col);
                ;
                index = index + 2;
            }
        }
        itsa.setColorIndices(0, colorIndices);
    }

    protected int getVertexFormat() {
        return super.getVertexFormat() | GeometryArray.COLOR_3;
    }

    private int getElevationColorIndex(float elevation, int row, int col) {
        // Normalize the height value to a
        // color index between 0 and NUMBER_OF_COLORS - 1
        float index = (NUMBER_OF_COLORS - 1) * ((100f - elevation) / 100f);
        // Vary the color index with a bit of noise.
        float delta =
                1.5f
                * (float) ImprovedNoise.noise(row / 3.7,
                        elevation / 7.4,
                        col / 3.7);

        int answer = Math.round(index + delta);
        // Clamp the index value
        if (answer < 0)
            answer = 0;
        if (answer > NUMBER_OF_COLORS - 1)
            answer = NUMBER_OF_COLORS - 1;
        return answer;
    }

    private float[] getElevationColors(int numberOfVertices) {
        /* These colors were arrived at through experimentation.
         * A color utility I found very useful is called
         * 'La boite a couleurs' by Benjamin Chartier
         */

        /* Even though there are only NUMBER_OF_COLORS colors
         * used, Java3D requires the color array to be based on
         * the number of vertices.  The rest of the array goes
         * unused.
         */
        float[] colors = new float[numberOfVertices];
        int i = 0;
        //
        colors[i++] = 0.72f;
        colors[i++] = 0.59f;
        colors[i++] = 0.44f;
        //
        colors[i++] = 0.64f;
        colors[i++] = 0.49f;
        colors[i++] = 0.32f;
        //
        colors[i++] = 0.51f;
        colors[i++] = 0.39f;
        colors[i++] = 0.25f;

        colors[i++] = 0.43f;
        colors[i++] = 0.33f;
        colors[i++] = 0.21f;
        //
        colors[i++] = 0.38f;
        colors[i++] = 0.29f;
        colors[i++] = 0.18f;
        //
        colors[i++] = 0.31f;
        colors[i++] = 0.25f;
        colors[i++] = 0.15f;

        colors[i++] = 0.27f;
        colors[i++] = 0.21f;
        colors[i++] = 0.13f;

        colors[i++] = 0.23f;
        colors[i++] = 0.28f;
        colors[i++] = 0.14f;

        //
        colors[i++] = 0.28f;
        colors[i++] = 0.36f;
        colors[i++] = 0.14f;
        //
        colors[i++] = 0.23f;
        colors[i++] = 0.35f;
        colors[i++] = 0.11f;
        //
        colors[i++] = 0.28f;
        colors[i++] = 0.43f;
        colors[i++] = 0.13f;

        colors[i++] = 0.30f;
        colors[i++] = 0.46f;
        colors[i++] = 0.14f;
        //
        colors[i++] = 0.33f;
        colors[i++] = 0.50f;
        colors[i++] = 0.16f;
        //
        colors[i++] = 0.35f;
        colors[i++] = 0.53f;
        colors[i++] = 0.17f;

        colors[i++] = 0.38f;
        colors[i++] = 0.58f;
        colors[i++] = 0.18f;

        colors[i++] = 0.43f;
        colors[i++] = 0.66f;
        colors[i++] = 0.20f;
        // 		sandy
        colors[i++] = 0.62f;
        colors[i++] = 0.58f;
        colors[i++] = 0.38f;
        //
        colors[i++] = 0.66f;
        colors[i++] = 0.62f;
        colors[i++] = 0.44f;
        //
        colors[i++] = 0.70f;
        colors[i++] = 0.67f;
        colors[i++] = 0.50f;
        //
        colors[i++] = 0.74f;
        colors[i++] = 0.71f;
        colors[i++] = 0.56f;
        //
        colors[i++] = 0.77f;
        colors[i++] = 0.75f;
        colors[i++] = 0.63f;
        //		blue
        colors[i++] = 0.0f;
        colors[i++] = 0.56f;
        colors[i++] = 0.57f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.38f;
        colors[i++] = 0.54f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.24f;
        colors[i++] = 0.35f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.14f;
        colors[i++] = 0.20f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.07f;
        colors[i++] = 0.10f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.03f;
        colors[i++] = 0.04f;

        return colors;
    }
}
