package j3d.examples.appearance.texture.noise;

import javax.media.j3d.*;
import java.awt.image.BufferedImage;

/**
 * 
 * 
 */

public class Lava extends Surface implements ImageComponent2D.Updater {
    /* The number of octaves and image size are directly related.  The
     * number of octaves determines the number of noise iterations to
     * compute the turbulance value.  There is no reason to calculate
     * turbulance effects that are sub-pixel within the image so the
     * image size is 2 to the power of the octaves.
     */
    private static final int OCTAVES = 6;
    /*
     * While the image size is determined by the octaves, you can also
     * get some pretty interesting effects with 2 octaves and an image
     * size of 64 or more.
     */
    private static final int IMAGE_SIZE = 1 << OCTAVES;

    public Lava(float sideLength, int divisions) {
        super(sideLength, divisions);
    }

    public void nextFrame(int ticks) {
        /* Called during animation.  Because
         * this object is an image updated, get
         * the image and pass this object to
         * the updateData method on the image.
         */
        this.ticks = ticks;
        ImageComponent2D image = (ImageComponent2D) getAppearance().getTexture().getImage(0);
        image.updateData(this, 0, 0, IMAGE_SIZE, IMAGE_SIZE);
    }

    public void updateData(ImageComponent2D image, int x, int y, int width, int height) {
        // This example alters the texture in realtime.
        BufferedImage bi = image.getImage();
        generateImage(bi);
    }

    protected float calculateHeight(double x, double y, double z) {
        // This example uses a flat geometry.
        return 0f;
    }

    protected Appearance createAppearance() {
        Appearance appearance = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
        polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);
        // Setting back face normal flip to true allows backfacing
        // polygons to be lit (normal is facing wrong way) but only
        // if the normal is flipped.
        polyAttrib.setBackFaceNormalFlip(true);
        appearance.setPolygonAttributes(polyAttrib);

        Material material = new Material();
        material.setAmbientColor(0f, 0f, 0f);
        appearance.setMaterial(material);

        Texture2D texture =
                new Texture2D(Texture2D.BASE_LEVEL,
                        Texture2D.RGBA,
                        IMAGE_SIZE,
                        IMAGE_SIZE);
        texture.setImage(0, getImage());
        texture.setEnable(true);
        texture.setMagFilter(Texture2D.NICEST);
        texture.setMinFilter(Texture2D.NICEST);
        appearance.setTexture(texture);

        return appearance;
    }

    private ImageComponent2D getImage() {
        BufferedImage bi =
                new BufferedImage(IMAGE_SIZE,
                        IMAGE_SIZE,
                        BufferedImage.TYPE_INT_RGB);
        generateImage(bi);

        // For debugging the image creation
//		 		try {
//		 			File outfile = new File("c:\\lava.jpg");
//		 			ImageIO.write(bi, "jpg", outfile);
//		 		} catch (Exception e) {
//		 			e.printStackTrace();
//		 		}

        ImageComponent2D image =
                new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, bi, true, false);
        return image;
    }

    protected void generateImage(BufferedImage bi) {
        for (int column = 0; column < IMAGE_SIZE; column++) {
            for (int row = 0; row < IMAGE_SIZE; row++) {
                // To speed things up, you could precompute the x and y values
                double x = column / NOISE_ZOOM;
                double y = row / NOISE_ZOOM;
                double z = ticks / 32.56;
                double turbulance = ImprovedNoise.turbulance(x, y, z, OCTAVES);
                // A color must be in the range of 0 to 255.
                double color = Math.min(192 * turbulance, 192);
                int red = 255 - (int) (0.3 * color);
                int green = 192 - (int) (color);
                int blue = 0;
                bi.setRGB(column, row, (red << 16) + (green << 8) + blue);
            }
        }
    }

    protected void customizeGeometry(IndexedTriangleStripArray itsa) {
        // Assign the texture coordinates
        float[] coordinates = itsa.getCoordRefFloat();
        int[] indices = (int[]) itsa.getUserData();

        float[] textureCoordinates = itsa.getTexCoordRefFloat(0);
        if (textureCoordinates == null) {
            textureCoordinates = new float[2 * coordinates.length / 3];
            itsa.setTexCoordRefFloat(0, textureCoordinates);
            itsa.setTextureCoordinateIndices(0, 0, indices);
        }
        // Generate the texture coordinates
        float textureIncrement = 1.0f / (float) divisions;
        float textureRowOffset = 0;
        float textureColumnOffset = 0;
        for (int row = 0, i = 0; row < (divisions + 1); row++) {
            textureColumnOffset = 0;
            for (int col = 0; col < (divisions + 1); col++, i++) {
                textureCoordinates[2 * i + 0] = textureColumnOffset;
                textureCoordinates[2 * i + 1] = textureRowOffset;
                textureColumnOffset = textureColumnOffset + textureIncrement;
            }
            textureRowOffset = textureRowOffset + textureIncrement;
        }
    }

    protected int getVertexFormat() {
        return super.getVertexFormat() | GeometryArray.TEXTURE_COORDINATE_2;
    }

    protected void setCapabilities() {
        super.setCapabilities();
        setCapability(Shape3D.ALLOW_APPEARANCE_READ);
        getAppearance().setCapability(Appearance.ALLOW_TEXTURE_READ);
        getAppearance().getTexture().setCapability(Texture.ALLOW_IMAGE_READ);
        getAppearance().getTexture().getImage(0).setCapability(ImageComponent.ALLOW_IMAGE_READ);
        getAppearance().getTexture().getImage(0).setCapability(ImageComponent.ALLOW_IMAGE_WRITE);
    }
}
