package j3d.examples.appearance;

/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */

import com.sun.j3d.loaders.Scene;
import com.sun.j3d.loaders.objectfile.ObjectFile;
import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.vp.OrbitBehavior;
import com.sun.j3d.utils.geometry.GeometryInfo;
import com.sun.j3d.utils.geometry.NormalGenerator;
import com.sun.j3d.utils.geometry.Sphere;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.universe.ViewingPlatform;
import gui.run.IntChooser;
import gui.run.IntEvent;
import gui.run.IntListener;
import j3d.utils.*;
import utils.ResourceManager;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.text.NumberFormat;
import java.util.Enumeration;
import java.util.Hashtable;


public class AppearanceExplorer extends JApplet implements
        Java3DExplorerConstants {

    private ResourceManager rm = ResourceManager.getResourceManager();
    // Scene graph items
    private SimpleUniverse su;
    private Switch sceneSwitch;
    private Group beethoven = null;
    private Group galleon = null;
    private Switch bgSwitch;
    private IntChooser bgChooser;
    private Appearance appearance;


    // image grabber
    private boolean isApplication;
    private Canvas3D canvas;
    private OffScreenCanvas3D offScreenCanvas;


    // ColoringAttributes
    private ColoringAttributes coloringAttr;
    private ColoringAttributesEditor coloringAttrEditor;
    private Color3f coloringColor;
    private int coloringShadeModel = ColoringAttributes.SHADE_GOURAUD;

    // PointAttributes
    private PointAttributes pointAttr;
    private PointAttributesEditor pointAttrEditor;
    private float pointSize = 4.0f;
    private boolean pointAAEnable = false;

    // LineAttributes
    private LineAttributes lineAttr;
    private float lineWidth = 1.0f;
    private LineAttributesEditor lineAttrEditor;
    private boolean lineAAEnable = false;
    private int linePattern = LineAttributes.PATTERN_SOLID;

    // PolygonAttributes
    private PolygonAttributes polygonAttr;
    private PolygonAttributesEditor polygonAttrEditor;
    private int polygonMode = PolygonAttributes.POLYGON_FILL;
    private int polygonCull = PolygonAttributes.CULL_NONE;
    private float polygonOffsetBias = 1.0f;
    private float polygonOffsetFactor = 1.0f;


    // RenderingAttributes
    private RenderingAttributes renderAttr;
    private RenderingAttributesEditor renderAttrEditor;
    private boolean renderVisible = true;
    private boolean renderDepthBuffer = true;
    private boolean renderDepthBufferWrite = true;
    private boolean renderIgnoreVertexColor = false;
    private boolean renderRasterOpEnable = false;
    private int renderRasterOp = RenderingAttributes.ROP_COPY;

    // TransparencyAttributes
    private TransparencyAttributes transpAttr;
    private TransparencyAttributesEditor transpAttrEditor;
    private int transpMode = TransparencyAttributes.NONE;
    private float transpValue = 0.5f;

    // Material
    private Material material;
    private MaterialEditor materialEditor;

    // Texture2D
    private Texture2DEditor texture2DEditor;
    private boolean texEnable;
    //String texImageFile;
    private int texBoundaryModeS;
    private int texBoundaryModeT;
    private Color4f texBoundaryColor;
    private int texMinFilter;
    private int texMagFilter;
    private int texMipMapMode;

    // TextureAttributes
    private TextureAttributes textureAttr;
    private TextureAttributesEditor textureAttrEditor;
    private int texMode;
    private Color4f texBlendColor;
    private Transform3D texTransform;
    private int texPerspCorrect;


    // TexCoordGeneration

    private TexCoordGenerationEditor texGenEditor;
    private boolean texGenEnable;
    private int texGenMode;
    private Vector4f texGenPlaneS;
    private Vector4f texGenPlaneT;

    // GUI helpers to allow galleon and beethoven to be loaded as needed
    // to reduce the startup time
    private int galleonIndex;
    private int beethovenIndex;
    private String galleonString = "Obj File: Galleon";
    private String beethovenString = "Obj File: Beethoven";
    private BranchGroup beethovenPlaceholder;
    private BranchGroup galleonPlaceholder;

    // Config items
    private Switch lightSwitch;
    private String snapImageString = "Snap Image";
    private String outFileBase = "appear";
    private int outFileSeq = 0;
    private float offScreenScale = 1.5f;

    // Temporaries that are reused
    private Transform3D tmpTrans = new Transform3D();
    private Vector3f tmpVector = new Vector3f();


    // NumberFormat to print out floats with only two digits
    private NumberFormat nf;

    // create the appearance and it's components
    void setupAppearance() {
        appearance = new Appearance();

        // ColoringAttributes
        getColorAttributes();
        appearance.setColoringAttributes(coloringAttr);

        // set up the editor
        coloringAttrEditor = new ColoringAttributesEditor(coloringAttr);

        // PointAttributes
        pointAttr = new PointAttributes(pointSize, pointAAEnable);
        pointAttr.setCapability(PointAttributes.ALLOW_SIZE_WRITE);
        pointAttr.setCapability(PointAttributes.ALLOW_ANTIALIASING_WRITE);
        appearance.setPointAttributes(pointAttr);

        // set up the editor
        pointAttrEditor = new PointAttributesEditor(pointAttr);

        // LineAttributes
        lineAttr = new LineAttributes(lineWidth, linePattern, lineAAEnable);
        lineAttr.setCapability(LineAttributes.ALLOW_WIDTH_WRITE);
        lineAttr.setCapability(LineAttributes.ALLOW_PATTERN_WRITE);
        lineAttr.setCapability(LineAttributes.ALLOW_ANTIALIASING_WRITE);
        appearance.setLineAttributes(lineAttr);

        // set up the editor
        lineAttrEditor = new LineAttributesEditor(lineAttr);

        // PolygonAttributes
        polygonAttr = new PolygonAttributes(polygonMode, polygonCull,
                0.0f);
        polygonAttr.setPolygonOffset(polygonOffsetBias);
        polygonAttr.setPolygonOffsetFactor(polygonOffsetFactor);
        polygonAttr.setCapability(PolygonAttributes.ALLOW_MODE_WRITE);
        polygonAttr.setCapability(PolygonAttributes.ALLOW_CULL_FACE_WRITE);
        polygonAttr.setCapability(PolygonAttributes.ALLOW_OFFSET_WRITE);
        appearance.setPolygonAttributes(polygonAttr);

        // set up the editor
        polygonAttrEditor = new PolygonAttributesEditor(polygonAttr);

        // Rendering attributes
        renderAttr = new RenderingAttributes(renderDepthBuffer, renderDepthBufferWrite,
                0.0f, RenderingAttributes.ALWAYS,
                renderVisible, renderIgnoreVertexColor,
                renderRasterOpEnable, renderRasterOp);
        renderAttr.setCapability(RenderingAttributes.ALLOW_IGNORE_VERTEX_COLORS_WRITE);
        renderAttr.setCapability(RenderingAttributes.ALLOW_VISIBLE_WRITE);
        renderAttr.setCapability(RenderingAttributes.ALLOW_RASTER_OP_WRITE);
        renderAttr.setCapability(RenderingAttributes.ALLOW_ALPHA_TEST_FUNCTION_WRITE);
        renderAttr.setCapability(RenderingAttributes.ALLOW_ALPHA_TEST_VALUE_WRITE);
        appearance.setRenderingAttributes(renderAttr);
        appearance.setCapability(Appearance.ALLOW_RENDERING_ATTRIBUTES_WRITE);

        // set up the editor
        renderAttrEditor = new RenderingAttributesEditor(renderAttr);

        // TransparencyAttributes
        transpAttr = new TransparencyAttributes(transpMode, transpValue);
        transpAttr.setCapability(TransparencyAttributes.ALLOW_MODE_WRITE);
        transpAttr.setCapability(TransparencyAttributes.ALLOW_VALUE_WRITE);
        transpAttr.setCapability(TransparencyAttributes.ALLOW_BLEND_FUNCTION_WRITE);
        appearance.setTransparencyAttributes(transpAttr);

        // set up the editor
        transpAttrEditor = new TransparencyAttributesEditor(transpAttr);

        // Material
        material = new Material(red, black, red, white, 20.f);
        material.setLightingEnable(false);
        material.setCapability(Material.ALLOW_COMPONENT_WRITE);
        appearance.setMaterial(material);


        // material presets
        String[] materialNames = {"Red", "White", "Red Ambient",
                                  "Red Diffuse", "Grey Emissive", "White Specular",
                                  "Aluminium", "Blue Plastic", "Copper", "Gold", "Red Alloy",
                                  "Black Onyx"};
        Material[] materialPresets = new Material[materialNames.length];
        materialPresets[0] = new Material(red, black, red, white, 20.0f);
        materialPresets[1] = new Material(white, black, white, white, 20.0f);
        materialPresets[2] = new Material(red, black, black, black, 20.0f);
        materialPresets[3] = new Material(black, black, red, black, 20.0f);
        materialPresets[4] = new Material(black, grey, black, black, 20.0f);
        materialPresets[5] = new Material(black, black, black, white, 20.0f);
        Color3f alum = new Color3f(0.37f, 0.37f, 0.37f);
        Color3f alumSpec = new Color3f(0.89f, 0.89f, 0.89f);
        materialPresets[6] = new Material(alum, black, alum, alumSpec, 17);
        Color3f bluePlastic = new Color3f(0.20f, 0.20f, 0.70f);
        Color3f bluePlasticSpec = new Color3f(0.85f, 0.85f, 0.85f);
        materialPresets[7] = new Material(bluePlastic, black, bluePlastic,
                bluePlasticSpec, 22);
        Color3f copper = new Color3f(0.30f, 0.10f, 0.00f);
        ;
        Color3f copperSpec = new Color3f(0.75f, 0.30f, 0.00f);
        materialPresets[8] = new Material(copper, black, copper, copperSpec, 10);
        Color3f gold = new Color3f(0.49f, 0.34f, 0.00f);
        Color3f goldSpec = new Color3f(0.89f, 0.79f, 0.00f);
        materialPresets[9] = new Material(gold, black, gold, goldSpec, 15);
        Color3f redAlloy = new Color3f(0.34f, 0.00f, 0.34f);
        Color3f redAlloySpec = new Color3f(0.84f, 0.00f, 0.00f);
        materialPresets[10] = new Material(redAlloy, black, redAlloy,
                redAlloySpec, 15);
        Color3f blackOnyxSpec = new Color3f(0.72f, 0.72f, 0.72f);
        materialPresets[11] = new Material(black, black, black, blackOnyxSpec, 23);

        // set up the editor
        materialEditor = new MaterialPresetEditor(material, materialNames,
                materialPresets);

        // Texture2D

        // set the values to the defaults
        texEnable = false;
        texMipMapMode = Texture.BASE_LEVEL;
        texBoundaryModeS = Texture.WRAP;
        texBoundaryModeT = Texture.WRAP;
        texMinFilter = Texture.BASE_LEVEL_POINT;
        texMagFilter = Texture.BASE_LEVEL_POINT;
        texBoundaryColor = new Color4f(0.0f, 0.0f, 0.0f, 0.0f);
        ResourceManager rm = ResourceManager.getResourceManager();
        File fa[] = rm.getImages("gif");

        // set up the image choices
        String[] texImageNames = getFileNames(fa);
        String[] texImageFileNames = getFileNames(fa);
        int texImageFileIndex = 0;

        // set up the appearance to allow the texture to be changed
        appearance.setCapability(Appearance.ALLOW_TEXTURE_WRITE);

        // set up the editor (this will create the initial Texture2D and
        // assign it to the appearance)
        texture2DEditor = new Texture2DEditor(appearance,
                texImageNames, texImageFileNames,
                texImageFileIndex, texEnable, texBoundaryModeS,
                texBoundaryModeT, texMinFilter, texMagFilter,
                texMipMapMode, texBoundaryColor);


        // TextureAttributes
        texMode = TextureAttributes.REPLACE;
        texBlendColor = new Color4f(1.0f, 1.0f, 1.0f, 1.0f);
        texTransform = new Transform3D();
        texPerspCorrect = TextureAttributes.NICEST;
        textureAttr = new TextureAttributes(texMode, texTransform,
                texBlendColor, texPerspCorrect);

        // set the capabilities to allow run time changes
        textureAttr.setCapability(TextureAttributes.ALLOW_MODE_WRITE);
        textureAttr.setCapability(TextureAttributes.ALLOW_BLEND_COLOR_WRITE);
        textureAttr.setCapability(TextureAttributes.ALLOW_TRANSFORM_WRITE);

        // connect it to the appearance
        appearance.setTextureAttributes(textureAttr);

        // setup the editor
        textureAttrEditor = new TextureAttributesEditor(textureAttr);

        // set up the tex coordinate generation
        texGenEnable = false;
        texGenMode = TexCoordGeneration.OBJECT_LINEAR;
        texGenPlaneS = new Vector4f(1.0f, 0.0f, 0.0f, 0.0f);
        texGenPlaneT = new Vector4f(0.0f, 1.0f, 0.0f, 0.0f);

        // set the appearance so that we can replace the tex gen when live
        appearance.setCapability(Appearance.ALLOW_TEXGEN_WRITE);

        // setup the editor
        texGenEditor = new TexCoordGenerationEditor(appearance, texGenEnable,
                texGenMode, texGenPlaneS, texGenPlaneT);
        //texGenEditor = new RunTexCoordGenerationEditor(){
        //    public void run(){
        //        appearance.setTexCoordGeneration(getValue());
        //    }
       // };

    }

    private String[] getFileNames(File[] fa) {
        String s[] = new String  [fa.length];
        for (int i=0; i < fa.length;i++)
            s[i] = fa[i].getName();
        return s;
    }

    private String[] getAbsoluteFileNames(File[] fa) {
        String s[] = new String [fa.length];
        for (int i=0; i < fa.length;i++)
            s[i] = fa[i].getAbsolutePath();
        return s;
    }

    private void getColorAttributes() {
        coloringColor = new Color3f(red);
        coloringAttr = new ColoringAttributes(coloringColor,
                coloringShadeModel);
        coloringAttr.setCapability(ColoringAttributes.ALLOW_COLOR_WRITE);
        coloringAttr.setCapability(ColoringAttributes.ALLOW_SHADE_MODEL_WRITE);
    }

    int powerOfTwo(int value) {
        int retval = 2;
        while (retval < value) {
            retval *= 2;
        }
        return retval;
    }

    // Point Array with three points
    Shape3D createPointArray() {

        Point3f pnt[] = new Point3f[3];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);

        PointArray pa = new PointArray(3, GeometryArray.COORDINATES);
        pa.setCoordinates(0, pnt);

        return new Shape3D(pa, appearance);
    }

    // Line Array with two lines with vertex colors
    Shape3D createLineArray() {

        Point3f pnt[] = new Point3f[4];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        pnt[3] = new Point3f(-1.0f, 1.0f, 0.0f);
        Color3f colrs[] = new Color3f[4];
        colrs[0] = black;
        colrs[1] = white;
        colrs[2] = red;
        colrs[3] = green;

        LineArray la = new LineArray(4,
                GeometryArray.COORDINATES | GeometryArray.COLOR_3);
        la.setCoordinates(0, pnt);
        la.setColors(0, colrs);

        return new Shape3D(la, appearance);
    }

    // Triangle Array with one triangle with vertex colors and a facet normal
    Shape3D createTriangleArray() {

        Point3f pnt[] = new Point3f[3];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        Color3f colrs[] = new Color3f[3];
        colrs[0] = red;
        colrs[1] = green;
        colrs[2] = blue;
        Vector3f norms[] = new Vector3f[3];
        Vector3f triNormal = new Vector3f(0.0f, 0.0f, 1.0f);
        norms[0] = triNormal;
        norms[1] = triNormal;
        norms[2] = triNormal;

        TriangleArray ta = new TriangleArray(3,
                GeometryArray.COORDINATES | GeometryArray.COLOR_3 |
                GeometryArray.NORMALS);
        ta.setCoordinates(0, pnt);
        ta.setColors(0, colrs);
        ta.setNormals(0, norms);

        return new Shape3D(ta, appearance);
    }

    // Line Strip Array with two lines with 3 and 2 vertices each making
    // a two segment line and a one segment line
    Shape3D createLineStripArray() {

        int[] stripLengths = new int[2];
        stripLengths[0] = 3;
        stripLengths[1] = 2;
        Point3f pnt[] = new Point3f[5];
        // first line
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        // second line
        pnt[3] = new Point3f(0.5f, 0.5f, 0.0f);
        pnt[4] = new Point3f(-0.5f, -0.5f, 0.0f);

        LineStripArray lsa = new LineStripArray(5, GeometryArray.COORDINATES,
                stripLengths);
        lsa.setCoordinates(0, pnt);

        return new Shape3D(lsa, appearance);
    }

    Shape3D createTriangleStripArray() {

        int[] stripLengths = new int[1];
        stripLengths[0] = 5;
        Point3f pnt[] = new Point3f[5];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(-1.0f, 0.0f, 0.0f);
        pnt[3] = new Point3f(1.0f, 0.0f, 0.0f);
        pnt[4] = new Point3f(1.0f, 1.0f, 0.0f);

        TriangleStripArray tsa = new TriangleStripArray(5,
                GeometryArray.COORDINATES, stripLengths);
        tsa.setCoordinates(0, pnt);

        return new Shape3D(tsa, appearance);
    }

    Shape3D createTriangleFanArray() {

        int[] stripLengths = new int[1];
        stripLengths[0] = 5;
        Point3f pnt[] = new Point3f[5];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 0.0f, 0.0f);
        pnt[3] = new Point3f(0.0f, 1.0f, 0.0f);
        pnt[4] = new Point3f(-1.0f, 1.0f, 0.0f);

        TriangleFanArray tfa = new TriangleFanArray(5,
                GeometryArray.COORDINATES, stripLengths);
        tfa.setCoordinates(0, pnt);

        return new Shape3D(tfa, appearance);
    }

    Shape3D createTexTris() {

        Point3f pnt[] = new Point3f[9];
        pnt[0] = new Point3f(-0.8f, -0.8f, 0.0f);
        pnt[1] = new Point3f(-0.5f, -0.7f, 0.0f);
        pnt[2] = new Point3f(-0.7f, 0.7f, 0.0f);

        pnt[3] = new Point3f(-0.4f, 0.7f, 0.0f);
        pnt[4] = new Point3f(0.0f, -0.7f, 0.0f);
        pnt[5] = new Point3f(0.4f, 0.7f, 0.0f);

        pnt[6] = new Point3f(0.5f, 0.7f, 0.0f);
        pnt[7] = new Point3f(0.5f, -0.7f, 0.0f);
        pnt[8] = new Point3f(0.9f, 0.0f, 0.0f);

        TexCoord2f texCoord[] = new TexCoord2f[9];
        texCoord[0] = new TexCoord2f(0.05f, 0.90f);
        texCoord[1] = new TexCoord2f(0.25f, 0.10f);
        texCoord[2] = new TexCoord2f(1.00f, 0.60f);

        texCoord[3] = texCoord[0];
        texCoord[4] = texCoord[1];
        texCoord[5] = texCoord[2];

        texCoord[6] = texCoord[0];
        texCoord[7] = texCoord[1];
        texCoord[8] = texCoord[2];

        TriangleArray ta = new TriangleArray(9,
                GeometryArray.COORDINATES | GeometryArray.TEXTURE_COORDINATE_2);
        ta.setCoordinates(0, pnt);
        ta.setTextureCoordinates(0, 0, texCoord);

        return new Shape3D(ta, appearance);
    }

    Shape3D createTexSquare() {

        // color cube
        Point3f pnt[] = new Point3f[4];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        pnt[3] = new Point3f(-1.0f, 1.0f, 0.0f);
        TexCoord2f texCoord[] = new TexCoord2f[4];
        texCoord[0] = new TexCoord2f(0.0f, 0.0f);
        texCoord[1] = new TexCoord2f(1.0f, 0.0f);
        texCoord[2] = new TexCoord2f(1.0f, 1.0f);
        texCoord[3] = new TexCoord2f(0.0f, 1.0f);

        QuadArray qa = new QuadArray(4,
                GeometryArray.COORDINATES | GeometryArray.TEXTURE_COORDINATE_2);
        qa.setCoordinates(0, pnt);
        qa.setTextureCoordinates(0, 0, texCoord);

        return new Shape3D(qa, appearance);
    }

    Shape3D createLargeTexSquare() {

        // color cube
        Point3f pnt[] = new Point3f[4];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        pnt[3] = new Point3f(-1.0f, 1.0f, 0.0f);
        TexCoord2f texCoord[] = new TexCoord2f[4];
        texCoord[0] = new TexCoord2f(-1.0f, -1.0f);
        texCoord[1] = new TexCoord2f(2.0f, -1.0f);
        texCoord[2] = new TexCoord2f(2.0f, 2.0f);
        texCoord[3] = new TexCoord2f(-1.0f, 2.0f);

        QuadArray qa = new QuadArray(4,
                GeometryArray.COORDINATES | GeometryArray.TEXTURE_COORDINATE_2);
        qa.setCoordinates(0, pnt);
        qa.setTextureCoordinates(0, 0, texCoord);

        return new Shape3D(qa, appearance);
    }

    Shape3D createColorCube() {

        // color cube
        int[] indices = {
            0, 3, 4, 2, // left face   x = -1
            0, 1, 5, 3, // bottom face y = -1
            0, 2, 6, 1, // back face   z = -1
            7, 5, 1, 6, // right face  x =  1
            7, 6, 2, 4, // top face    y =  1
            7, 4, 3, 5  // front face  z =  1
        };

        Point3f pts[] = new Point3f[8];
        pts[0] = new Point3f(-1.0f, -1.0f, -1.0f);
        pts[1] = new Point3f(1.0f, -1.0f, -1.0f);
        pts[2] = new Point3f(-1.0f, 1.0f, -1.0f);
        pts[3] = new Point3f(-1.0f, -1.0f, 1.0f);
        pts[4] = new Point3f(-1.0f, 1.0f, 1.0f);
        pts[5] = new Point3f(1.0f, -1.0f, 1.0f);
        pts[6] = new Point3f(1.0f, 1.0f, -1.0f);
        pts[7] = new Point3f(1.0f, 1.0f, 1.0f);
        Color3f colr[] = new Color3f[8];
        colr[0] = black;
        colr[1] = red;
        colr[2] = green;
        colr[3] = blue;
        colr[4] = cyan;
        colr[5] = magenta;
        colr[6] = yellow;
        colr[7] = white;
        // The normals point out from 0,0,0, through the verticies of the
        // cube.  These can be calculated by copying the coordinates to
        // a Vector3f and normalizing.
        Vector3f norm[] = new Vector3f[8];
        for (int i = 0; i < 8; i++) {
            norm[i] = new Vector3f(pts[i]);
            norm[i].normalize();
        }


        IndexedQuadArray iqa = new IndexedQuadArray(8,
                GeometryArray.COORDINATES | GeometryArray.COLOR_3 |
                GeometryArray.NORMALS, 24);
        iqa.setCoordinates(0, pts);
        iqa.setColors(0, colr);
        iqa.setNormals(0, norm);
        iqa.setCoordinateIndices(0, indices);
        iqa.setColorIndices(0, indices);
        iqa.setNormalIndices(0, indices);
        return new Shape3D(iqa, appearance);
    }

    Shape3D createNGCube(float creaseAngle) {

        // color cube
        int[] indices = {
            0, 3, 4, 2, // left face   x = -1
            0, 1, 5, 3, // bottom face y = -1
            0, 2, 6, 1, // back face   z = -1
            7, 5, 1, 6, // right face  x =  1
            7, 6, 2, 4, // top face    y =  1
            7, 4, 3, 5  // front face  z =  1
        };

        Point3f pts[] = new Point3f[8];
        pts[0] = new Point3f(-1.0f, -1.0f, -1.0f);
        pts[1] = new Point3f(1.0f, -1.0f, -1.0f);
        pts[2] = new Point3f(-1.0f, 1.0f, -1.0f);
        pts[3] = new Point3f(-1.0f, -1.0f, 1.0f);
        pts[4] = new Point3f(-1.0f, 1.0f, 1.0f);
        pts[5] = new Point3f(1.0f, -1.0f, 1.0f);
        pts[6] = new Point3f(1.0f, 1.0f, -1.0f);
        pts[7] = new Point3f(1.0f, 1.0f, 1.0f);

        GeometryInfo gi = new GeometryInfo(GeometryInfo.QUAD_ARRAY);
        gi.setCoordinates(pts);
        gi.setCoordinateIndices(indices);
        NormalGenerator ng = new NormalGenerator();
        ng.setCreaseAngle((float) Math.toRadians(creaseAngle));
        ng.generateNormals(gi);
        GeometryArray cube = gi.getGeometryArray();
        return new Shape3D(cube, appearance);
    }

    Shape3D createTriWithHole() {

        int[] stripCounts = new int[2];
        stripCounts[0] = 3;
        stripCounts[1] = 3;
        int[] contourCounts = new int[1];
        contourCounts[0] = 2;
        Point3f pnt[] = new Point3f[6];
        pnt[0] = new Point3f(-1.0f, -1.0f, 0.0f);
        pnt[1] = new Point3f(1.0f, -1.0f, 0.0f);
        pnt[2] = new Point3f(1.0f, 1.0f, 0.0f);
        pnt[3] = new Point3f(-0.6f, -0.8f, 0.0f);
        pnt[4] = new Point3f(0.8f, 0.6f, 0.0f);
        pnt[5] = new Point3f(0.8f, -0.8f, 0.0f);

        GeometryInfo gi = new GeometryInfo(GeometryInfo.POLYGON_ARRAY);
        gi.setCoordinates(pnt);
        gi.setStripCounts(stripCounts);
        gi.setContourCounts(contourCounts);

        //Triangulator tr = new Triangulator();
        //tr.triangulate(gi);
        GeometryArray triWithHole = gi.getGeometryArray();

        return new Shape3D(triWithHole, appearance);
    }

    Shape3D createText3D() {
        Font3D f3d = new Font3D(new Font(null, Font.PLAIN, 2), new
                FontExtrusion());
        Text3D t3d = new Text3D(f3d, "Text3D", new Point3f(-3.0f, -1.0f, 0.0f));
        Shape3D textShape = new Shape3D(t3d, appearance);
        return textShape;
    }


    BranchGroup createGalleon() {
        java.net.URL galleonURL = null;
        try {
            galleonURL = rm.get3dURL("galleon.obj");
        } catch (Exception e) {
            System.err.println("Exception: " + e);
            System.exit(1);
        }

        int flags = ObjectFile.RESIZE;
        ObjectFile f = new ObjectFile(flags);
        Scene s = null;
        try {
            s = f.load(galleonURL);
        } catch (Exception e) {
            System.err.println(e);
            System.exit(1);
        }


        Hashtable namedObjects = s.getNamedObjects();
        Enumeration e = namedObjects.keys();
        while (e.hasMoreElements()) {
            String name = (String) e.nextElement();
            //System.out.println("name = " + name);
            Shape3D shape = (Shape3D) namedObjects.get(name);
            shape.setAppearance(appearance);
        }

        BranchGroup retVal = new BranchGroup();
        retVal.addChild(s.getSceneGroup());
        return retVal;
    }

    BranchGroup createBeethoven() {
        java.net.URL beethovenURL = null;
        try {
            beethovenURL = rm.get3dURL("beethoven.obj");
        } catch (Exception e) {
            System.err.println("Exception: " + e);
            System.exit(1);
        }

        int flags = ObjectFile.RESIZE;
        ObjectFile f = new ObjectFile(flags);
        Scene s = null;
        try {
            s = f.load(beethovenURL);
        } catch (Exception e) {
            System.err.println(e);
            System.exit(1);
        }


        Hashtable namedObjects = s.getNamedObjects();
        Enumeration e = namedObjects.keys();
        while (e.hasMoreElements()) {
            String name = (String) e.nextElement();
            Shape3D shape = (Shape3D) namedObjects.get(name);
            shape.setAppearance(appearance);
        }

        BranchGroup retVal = new BranchGroup();
        retVal.addChild(s.getSceneGroup());
        return retVal;
    }


    // sets up the scene switch
    void setupSceneSwitch() {

        // create a Switch for the scene, allow switch changes
        sceneSwitch = new Switch();
        sceneSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
        sceneSwitch.setCapability(Switch.ALLOW_CHILDREN_READ);
        sceneSwitch.setCapability(Switch.ALLOW_CHILDREN_WRITE);
        sceneSwitch.setCapability(Switch.ALLOW_CHILDREN_EXTEND);

        Shape3D pointArray = createPointArray();
        sceneSwitch.addChild(pointArray);

        Shape3D lineArray = createLineArray();
        sceneSwitch.addChild(lineArray);

        Shape3D triangleArray = createTriangleArray();
        sceneSwitch.addChild(triangleArray);

        Shape3D lineStripArray = createLineStripArray();
        sceneSwitch.addChild(lineStripArray);

        Shape3D triangleStripArray = createTriangleStripArray();
        sceneSwitch.addChild(triangleStripArray);

        Shape3D triangleFanArray = createTriangleFanArray();
        sceneSwitch.addChild(triangleFanArray);

        Shape3D texTris = createTexTris();
        sceneSwitch.addChild(texTris);

        Shape3D texSquare = createTexSquare();
        sceneSwitch.addChild(texSquare);

        Shape3D largeTexSquare = createLargeTexSquare();
        sceneSwitch.addChild(largeTexSquare);

        Shape3D colorCube = createColorCube();
        sceneSwitch.addChild(colorCube);

        Shape3D ngCreaseCube = createNGCube(45);
        sceneSwitch.addChild(ngCreaseCube);

        Shape3D ngSmoothCube = createNGCube(100);
        sceneSwitch.addChild(ngSmoothCube);

        Shape3D triWithHole = createTriWithHole();
        sceneSwitch.addChild(triWithHole);


        // create a sphere with the shared appearance
        Sphere sphere = new Sphere(1.0f, Sphere.GENERATE_NORMALS |
                Sphere.GENERATE_TEXTURE_COORDS, appearance);
        sceneSwitch.addChild(sphere);

        // create a sphere with the shared appearance
        Sphere lrSphere = new Sphere(1.0f, Sphere.GENERATE_NORMALS |
                Sphere.GENERATE_TEXTURE_COORDS, 10, appearance);
        sceneSwitch.addChild(lrSphere);

        // create a sphere with the shared appearance
        Sphere hrSphere = new Sphere(1.0f, Sphere.GENERATE_NORMALS |
                Sphere.GENERATE_TEXTURE_COORDS, 45, appearance);
        sceneSwitch.addChild(hrSphere);


        // Text3D
        Shape3D text3D = createText3D();
        sceneSwitch.addChild(text3D);

        // galleon -- use a placeholder to indicate it hasn't been loaded yet
        // then load it the first time it gets asked for
        //was:
        //Group galleon = createGalleon();
        //sceneSwitch.addChild(galleon);
        galleonIndex = sceneSwitch.numChildren();
        galleonPlaceholder = new BranchGroup();
        galleonPlaceholder.setCapability(BranchGroup.ALLOW_DETACH);
        sceneSwitch.addChild(galleonPlaceholder);

        // beethoven -- use a placeholder to indicate it hasn't been loaded yet
        // then load it the first time it gets asked for
        //was:
        //Group beethoven = createBeethoven();
        //sceneSwitch.addChild(beethoven);
        beethovenIndex = sceneSwitch.numChildren();
        beethovenPlaceholder = new BranchGroup();
        beethovenPlaceholder.setCapability(BranchGroup.ALLOW_DETACH);
        sceneSwitch.addChild(beethovenPlaceholder);
    }


    /* Set up the lights.  This is a group which contains the
       ambient light and a switch for the other lights.
	directional   : white light pointing along Z axis
	point	      : white light near upper left corner of spheres
	spot	      : white light near upper left corner of spheres,
			      pointing towards center.
     */
    Group setupLights() {

        Group group = new Group();

        // set up the BoundingSphere for all the lights
        BoundingSphere bounds = new BoundingSphere(new Point3d(), 100.0);

        // Set up the ambient light
        AmbientLight lightAmbient = new AmbientLight(medGrey);
        lightAmbient.setInfluencingBounds(bounds);
        lightAmbient.setCapability(Light.ALLOW_STATE_WRITE);
        group.addChild(lightAmbient);

        lightSwitch = new Switch();
        lightSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
        group.addChild(lightSwitch);

        // Set up the directional light
        Vector3f lightDirection1 = new Vector3f(0.0f, 0.0f, -1.0f);
        DirectionalLight lightDirectional1 =
                new DirectionalLight(white, lightDirection1);
        lightDirectional1.setInfluencingBounds(bounds);
        lightDirectional1.setCapability(Light.ALLOW_STATE_WRITE);
        lightSwitch.addChild(lightDirectional1);

        Point3f lightPos1 = new Point3f(-4.0f, 8.0f, 16.0f);
        Point3f lightAttenuation1 = new Point3f(1.0f, 0.0f, 0.0f);
        PointLight pointLight1 = new PointLight(brightWhite, lightPos1,
                lightAttenuation1);
        pointLight1.setInfluencingBounds(bounds);
        lightSwitch.addChild(pointLight1);

        Point3f lightPos2 = new Point3f(-16.0f, 8.0f, 4.0f);
        //Point3f lightPos = new Point3f(-4.0f, 2.0f,  1.0f);
        Point3f lightAttenuation2 = new Point3f(1.0f, 0.0f, 0.0f);
        PointLight pointLight2 = new PointLight(white, lightPos2,
                lightAttenuation2);
        pointLight2.setInfluencingBounds(bounds);
        lightSwitch.addChild(pointLight2);

        return group;
    }

    BranchGroup createSceneGraph() {
        // Create the root of the branch graph
        BranchGroup objRoot = new BranchGroup();

        // Add the primitives to the scene
        setupAppearance();
        setupSceneSwitch();
        objRoot.addChild(sceneSwitch);
        objRoot.addChild(bgSwitch);
        Group lightGroup = setupLights();
        objRoot.addChild(lightGroup);

        return objRoot;
    }


    public AppearanceExplorer() {
        this(false, 1.0f);
    }

    public AppearanceExplorer(boolean isApplication, float initOffScreenScale) {
        this.isApplication = isApplication;
        this.offScreenScale = initOffScreenScale;
    }

    public void init() {


        // set up a NumFormat object to print out float with only 3 fraction
        // digits
        nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(3);

        Container contentPane = getContentPane();
        contentPane.setLayout(new BorderLayout());
        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();

        canvas = new Canvas3D(config);
        canvas.setSize(600, 600);

        su = new SimpleUniverse(canvas);

        if (isApplication) {
             offScreenCanvas = new OffScreenCanvas3D(config, true);
            // set the size of the off-screen canvas based on a scale
            // of the on-screen size
            Screen3D sOn = canvas.getScreen3D();
            Screen3D sOff = offScreenCanvas.getScreen3D();
            Dimension dim = sOn.getSize();
            dim.width *= offScreenScale;
            dim.height *= offScreenScale;
            sOff.setSize(dim);
            sOff.setPhysicalScreenWidth(sOn.getPhysicalScreenWidth() *
                    offScreenScale);
            sOff.setPhysicalScreenHeight(sOn.getPhysicalScreenHeight() *
                    offScreenScale);

// attach the offscreen canvas to the view
            su.getViewer().getView().addCanvas3D(offScreenCanvas);
        }
        contentPane.add("Center", canvas);

        BackgroundTool bgTool = new BackgroundTool();
        bgSwitch = bgTool.getSwitch();
        bgChooser = bgTool.getChooser();

        // Create a simple scene and attach it to the virtual universe
        BranchGroup scene = createSceneGraph();

        // set up sound
        su.getViewer().createAudioDevice();

        // get the view
        su.getViewer().getView();

        // Get the viewing platform
        ViewingPlatform viewingPlatform = su.getViewingPlatform();

        // Move the viewing platform back to enclose the -2 -> 2 range
        double viewRadius = 2.0; 	// want to be able to see circle
        // of viewRadius size around origin
        // get the field of view
        double fov = su.getViewer().getView().getFieldOfView();

        // calc view distance to make circle view in fov
        float viewDistance = (float) (viewRadius / Math.tan(fov / 2.0));
        tmpVector.set(0.0f, 0.0f, viewDistance);// setup offset
        tmpTrans.set(tmpVector);	  	// set trans to translate
        // move the view platform
        viewingPlatform.getViewPlatformTransform().setTransform(tmpTrans);

        // add an orbit behavior to move the viewing platform
        OrbitBehavior orbit = new OrbitBehavior(canvas,
                OrbitBehavior.PROPORTIONAL_ZOOM | OrbitBehavior.REVERSE_ROTATE |
                OrbitBehavior.REVERSE_TRANSLATE);
        orbit.setZoomFactor(0.25);
        BoundingSphere bounds =
                new BoundingSphere(new Point3d(0.0, 0.0, 0.0), 100.0);
        orbit.setSchedulingBounds(bounds);
        viewingPlatform.setViewPlatformBehavior(orbit);

        su.addBranchGraph(scene);

        contentPane.add("East", guiPanel());
    }

    public void destroy() {
        su.removeAllLocales();
    }

    // create a panel with a tabbed pane holding each of the edit panels
    JPanel guiPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        JTabbedPane tabbedPane = new JTabbedPane();
        tabbedPane.addTab("Setup", setupPanel());
        tabbedPane.addTab("ColoringAttributes", coloringAttrEditor);
        tabbedPane.addTab("PointAttributes", pointAttrEditor);
        tabbedPane.addTab("LineAttributes", lineAttrEditor);
        tabbedPane.addTab("PolygonAttributes", polygonAttrEditor);
        tabbedPane.addTab("RenderingAttributes", renderAttrEditor);
        tabbedPane.addTab("TransparencyAttributes", transpAttrEditor);
        tabbedPane.addTab("Material", materialEditor);
        tabbedPane.addTab("Texture2D", texture2DEditor);
        tabbedPane.addTab("TextureAttributes", textureAttrEditor);
        tabbedPane.addTab("TexCoordGeneration", texGenEditor);
        panel.add("Center", tabbedPane);

        return panel;
    }

    JPanel setupPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(0, 1));

        // This order of the names must match the cases in the Switch in
        // setupSceneSwitch
        String[] dataNames = {
            "Point Array",
            "Line Array",
            "Triangle Array",
            "Line Strip Array",
            "Triangle Strip Array",
            "Triangle Fan Array",
            "Textured Triangles",
            "Textured Square",
            "Large Texture Square",
            "Color Cube",
            "Norm Gen Cube - Crease",
            "Norm Gen Cube - Smooth",
            "Tri with hole",
            "Sphere",
            "Low-res Sphere",
            "High-res Sphere",
            "Text 3D",
            galleonString,
            beethovenString,
        };

        IntChooser dataChooser = new IntChooser("Data:", dataNames);
        dataChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                if (sceneSwitch.getChild(value) == beethovenPlaceholder) {
                    beethoven = createBeethoven();
                    sceneSwitch.setChild(beethoven, beethovenIndex);
                } else if (sceneSwitch.getChild(value) == galleonPlaceholder) {
                    galleon = createGalleon();
                    sceneSwitch.setChild(galleon, galleonIndex);
                }
                sceneSwitch.setWhichChild(value);
            }
        });
        dataChooser.setValueByName("Sphere");

        panel.add(dataChooser);

        panel.add(bgChooser);

        String[] lightNames = {
            "Ambient Only",
            "Directional",
            "Point Light 1",
            "Point Light 2",
        };
        int[] lightValues = {Switch.CHILD_NONE, 0, 1, 2};

        IntChooser lightChooser = new IntChooser("Light:", lightNames,
                lightValues, 0);
        lightChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                lightSwitch.setWhichChild(value);
            }
        });
        lightChooser.setValueByName("Point Light 1");

        panel.add(lightChooser);

        panel.add(new JLabel(""));

        if (isApplication) {
            JButton snapButton = new JButton(snapImageString);
            snapButton.setActionCommand(snapImageString);
            snapButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    doSnapshot(AppearanceExplorer.this);
                }
            });
            panel.add(snapButton);
        }

        return panel;
    }

    static void doSnapshot(AppearanceExplorer appearanceExplorer) {
        Canvas3D canvas = appearanceExplorer.canvas;


        float offScreenScale = appearanceExplorer.offScreenScale;
        OffScreenCanvas3D offScreenCanvas = appearanceExplorer.offScreenCanvas;
        snap(appearanceExplorer, canvas, offScreenCanvas, offScreenScale);

    }

    private static void snap(
            AppearanceExplorer appearanceExplorer,
            Canvas3D canvas,
            OffScreenCanvas3D offScreenCanvas,
            float offScreenScale) {
        offScreenCanvas.setOffScreenLocation(
                canvas.getLocationOnScreen());

        Dimension dim = canvas.getSize();
        dim.width *= offScreenScale;
        dim.height *= offScreenScale;
        appearanceExplorer.nf.setMinimumIntegerDigits(3);
        String outFileBase = appearanceExplorer.outFileBase;
        offScreenCanvas.snapImageFile(outFileBase +
                appearanceExplorer.nf.format(appearanceExplorer.outFileSeq++),
                dim.width, dim.height);
        appearanceExplorer.nf.setMinimumIntegerDigits(0);
    }


    // The following allows AppearanceExplorer to be run as an application
    // as well as an applet
    //
    public static void main(String[] args) {
        float initOffScreenScale = 2.5f;
        for (int i = 0; i < args.length; i++) {
            if (args[i].equals("-s")) {
                if (args.length >= (i + 1)) {
                    initOffScreenScale = Float.parseFloat(args[i + 1]);
                    i++;
                }
            }
        }
        new MainFrame(new AppearanceExplorer(true, initOffScreenScale),
                950, 600);
    }

}
