// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.io.file;

import j2d.io.ppm.Decoders.PpmDecoder;
import j2d.io.ppm.Encoders.PpmEncoder;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.PixelGrabber;
import java.io.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * Wrapper for Jef Poskanzer's PPM classes to create a simple toolkit
 * interface for reading and writing PPM files.
 * (http://www.acme.com/java/)
 *
 * Note that the PPM format supports ASCII or binary representation
 * of the pixel data.  Both of these formats are supported for input.
 * For output, only binary pixel data is supported.
 * 
 */
public class PPMToolkit {

	/**
	 * Read a PPM file (binary or ASCII formats 1-6)
	 *
	 * @param infile        The PPM image file.
	 * @param zipped        Flag to select ZIP decompression.
	 * @return              An AWT Image object containing the read data.
	 */
	public static Image getImage(File infile, boolean zipped) 
		throws FileNotFoundException, IOException {
		
		PpmDecoder decoder;
		GZIPInputStream gis = null;
		FileInputStream fis = null;
		// This next line makes up for the lousy API of PpmDecoder.
		// The decoder should already know its input stream and should
		// not need to be told again on readHeader() or readRow()!
		InputStream inputStreamAlias = null;
		
		fis = new FileInputStream(infile);
		if (zipped){
			gis = new GZIPInputStream(fis);
			inputStreamAlias = gis;
			decoder = new PpmDecoder(gis);
		}else{
			inputStreamAlias = fis;
			decoder = new PpmDecoder(fis);
		}
		
		BufferedImage outputImage = null;
		int width, height;
		int pixels[];
		int rowPixels[];
		
      	decoder.readHeader(inputStreamAlias);
      	width = decoder.getWidth();
      	height = decoder.getHeight();
      	if (width == -1 || height == -1)
        	throw new IOException();

        pixels = new int[width * height];
        rowPixels = new int[width];
        
        for (int row = 0; row < height; row++) {
        
        	decoder.readRow(inputStreamAlias, row, rowPixels);
        	
        	for (int i=0; i< width; i++){
        		pixels[row*width + i] = rowPixels[i];
        	}
	    }
	    
	    outputImage = new BufferedImage(width,height,
				BufferedImage.TYPE_INT_RGB);
		outputImage.setRGB(0,0,width,
				height,pixels,0,width);
	    return outputImage;
	}
	
	/**
	 * Write a PPM file (binary format 6)
	 *
	 * @param img           The AWT Image to be written.
	 * @param outfile       The file location to which to write.
	 * @param zipped        Flag to select ZIP compression.
	 */
	public static void saveImage(Image img, File outfile, boolean zipped)
		throws FileNotFoundException, IOException {
		
		PpmEncoder encoder;
		FileOutputStream fos = null;
		GZIPOutputStream gos = null;
		
		fos = new FileOutputStream(outfile);
		if (zipped){
			gos = new GZIPOutputStream(fos);
			encoder = new PpmEncoder(img, gos);
		}else{
			encoder = new PpmEncoder(img, fos);
		}
		
		int width = img.getWidth(null);
		int height = img.getHeight(null);
		int rowPixels[] = new int[width];
		int pixels[] = new int[width*height];

        try{
            PixelGrabber pg = new PixelGrabber(
                    img, 0, 0, width, height, pixels, 0, width);
            pg.grabPixels();
        }catch(InterruptedException ie){
            // do nothing for now
        }
        
        encoder.encodeStart(width, height);
        encoder.encodePixels(0,0,width,height, pixels, 0, width);
        encoder.encodeDone();
        
        if (gos != null){
        	gos.finish();
        	gos.close();
        }
        if (fos != null)
        	fos.close();
	}
}