package graphics.grapher;
import java.awt.*;
import java.applet.*;
import java.util.Vector;
import java.util.Enumeration;



/*************************************************************************
**
**    Class  Axis            
**                                              Version 1.0   October  1995
**                                              Version 1.1   December 1995
**                                              Version 1.2   January  1996
**
**************************************************************************
**    Copyright (C) 1995, 1996 Leigh Brookshaw
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**************************************************************************
**
**    This class is designed to be used in conjunction with 
**    the Graph2D class and DataSet class for plotting 2D graphs.
**
*************************************************************************/





public class Axis extends Object {
    
/**************************
** Public Static Values     
**************************/

/*
**    Horizontal Axis or Vertical ?
*/
      public static final int  HORIZONTAL = 0;
      public static final int  VERTICAL   = 1;
/*
**    The axis will be on the lleft/right border or top/bottom border?
*/
     public static final int  LEFT       = 2;
     public static final int  RIGHT      = 3;
     public  static final int  TOP        = 4;
     public static final int  BOTTOM     = 5;
/*
**    The first guess on the number of Major tic marks with labels
*/
      static final int  NUMBER_OF_TICS = 4;

/**********************
** Public Variables      
**********************/

      public boolean  drawgrid        = false;
      public boolean  drawzero        = false;
      public Color   gridcolor        = null;
      public Color   zerocolor        = null;
      public boolean redraw           = true;
      public boolean force_end_labels = false;

      public int     major_tic_size = 10;

      public int     minor_tic_size  = 5;
      public int     minor_tic_count = 1;

      public Color   axiscolor;

      public double minimum;
      public double maximum;

      public Dimension data_window = new Dimension(0,0);


      public TextLine      title = new TextLine();

/**********************
** Protected Variables      
**********************/
      protected Point amin;
      protected Point amax;
      protected int orientation;
      protected int position;
      protected int width = 0;


      protected TextLine label    = new TextLine("0");
      protected TextLine exponent = new TextLine();      

      protected int max_label_width     = 0;


      protected Vector dataset = new Vector();
       
      protected String label_string[]     = null;
      protected double  label_value[]      = null;
      protected double label_start        = 0.0;
      protected double label_step         = 0.0;
      protected int    label_exponent     = 0;
      protected int    label_count        = 0;
      protected int    guess_label_number = 4;

/********************
** Constructors
********************/

      public Axis() {
                 orientation = HORIZONTAL;
                 position    = BOTTOM;
      }           


      public Axis(int p) {

           setPosition(p);
      }

/******************
** Public Methods
******************/

      public void setPosition(int p) {
           position = p;

           switch (position) {
              case LEFT:
                          orientation = VERTICAL;
                          break;
              case RIGHT:
                          orientation = VERTICAL;
                          break;

              case TOP:
                          orientation = HORIZONTAL;
                          break;
              case BOTTOM:
                          orientation = HORIZONTAL;
                          break;
              case HORIZONTAL:
                          orientation = HORIZONTAL;
                          position    = BOTTOM;
                          break;
              case VERTICAL:
                          orientation = VERTICAL;
                          position    = LEFT;
                          break;
              default:
                          orientation = HORIZONTAL;
                          position    = BOTTOM;
                          break;
           }
      }
      

      public void attachDataSet( DataSet d ) {
            if( orientation == HORIZONTAL )   attachXdata( d );
            else                              attachYdata( d );
      }

      public void detachDataSet( DataSet d ) {
           int i = 0;

           if( d == null ) return;

           if( orientation == HORIZONTAL ) {
               d.xaxis = null;
           } else {
               d.yaxis = null;
           }
           dataset.removeElement(d);

           resetRange();
      }

      public void detachAll() {
            int i;
            DataSet d;

            if( dataset.isEmpty() ) return;


            if( orientation == HORIZONTAL ) {
                 for (i=0; i<dataset.size(); i++) {
                     d = (DataSet)(dataset.elementAt(i));
                     d.xaxis = null;
                 }
            } else {
                 for (i=0; i<dataset.size(); i++) {
                     d = (DataSet)(dataset.elementAt(i));
                     d.yaxis = null;
                 }
            }

            dataset.removeAllElements();

            minimum = 0.0;
            maximum = 0.0;
      }
      public double getDataMin() {
            double m;
            Enumeration e;
            DataSet d;

            if( dataset.isEmpty() ) return 0.0;

            d = (DataSet)(dataset.firstElement());
            if(d == null) return 0.0;

            if( orientation == HORIZONTAL ) {
                 m = d.getXmin();
                 for (e = dataset.elements() ; e.hasMoreElements() ;) {

                     d = (DataSet)e.nextElement();
                     m = Math.min(d.getXmin(),m);
                     
                 }
            } else {
                 m = d.getYmin();
                 for (e = dataset.elements() ; e.hasMoreElements() ;) {

                     d = (DataSet)e.nextElement();
                     m = Math.min(d.getYmin(),m);
                     
                 }
            }

            return m;
      }
      public double getDataMax() {
            double m;
            Enumeration e;
            DataSet d;

            if( dataset.isEmpty() ) return 0.0;

            d = (DataSet)(dataset.firstElement());

            if(d == null) return 0.0;

          
            if( orientation == HORIZONTAL ) {
                 m = d.getXmax();
                 for (e = dataset.elements() ; e.hasMoreElements() ;) {

                     d = (DataSet)e.nextElement();
                     m = Math.max(d.getXmax(),m);
                 }
            } else {
                 m = d.getYmax();
                 for (e = dataset.elements() ; e.hasMoreElements() ;) {

                     d = (DataSet)e.nextElement();

                     m = Math.max(d.getYmax(),m);
                 }
            }

            return m;
      }


      public int getInteger(double v) {
          double scale;
          
          if( orientation == HORIZONTAL ) {
               scale  = (double)(amax.x - amin.x)/(maximum - minimum);
               return amin.x + (int)( ( v - minimum ) * scale);
          } else {
               scale  = (double)(amax.y - amin.y)/(maximum - minimum);
               return amin.y + (int)( (maximum-v)*scale );

          }

      }

      public double getDouble(int i) {
            double scale;

          if( orientation == HORIZONTAL ) {
               scale  = (maximum - minimum)/(double)(amax.x - amin.x);
               return minimum + (i - amin.x)*scale;
          } else {
               scale  = (maximum - minimum)/(double)(amax.y - amin.y);
               return maximum - (i - amin.y)*scale;
          }
      }

     public void resetRange() {
             minimum = getDataMin();
             maximum = getDataMax();
     }


     public int getAxisPos() { return position; }      

     public int getAxisWidth(Graphics g) {
          int i;
          width = 0;

          if( minimum == maximum )    return 0;
          if( dataset.size() == 0 )   return 0;


          calculateGridLabels();


          exponent.setText(null);

          if(label_exponent != 0) {
              exponent.copyState(label);
              exponent.setText("x10^"+String.valueOf(label_exponent));
          }

           if( orientation == HORIZONTAL ) {

               width = label.getHeight(g) + label.getLeading(g); 
               width += Math.max(title.getHeight(g),exponent.getHeight(g));

           } else {
               for(i=0; i<label_string.length; i++) {
                  label.setText(" "+label_string[i]);
                  width = Math.max(label.getWidth(g),width);
               }
               max_label_width = width;
               width = 0;

               if(!title.isNull() ) {
                  width = Math.max(width,title.getWidth(g)+
                                         title.charWidth(g,' '));
               }

               if ( !exponent.isNull() ) {
                  width = Math.max(width,exponent.getWidth(g)+
                                         exponent.charWidth(g,' '));
               }
               width += max_label_width;
           }


           return width;
      }

      public boolean positionAxis(int xmin, int xmax, int ymin, int ymax ){
           amin = null;
           amax = null;

           if( orientation == HORIZONTAL && ymin != ymax ) return false;
           if( orientation == VERTICAL   && xmin != xmax ) return false;

           amin = new Point(xmin,ymin);
           amax = new Point(xmax,ymax);


           return true;
      }         



      public void drawAxis(Graphics g) {
          Graphics lg;

          if( !redraw            ) return;
          if( minimum == maximum ) {
                 resetRange();
                 if( minimum == maximum ) {
                    System.out.println(
                      "Axis: data minimum==maximum Axis not drawn!");
                    return;
		  }
	  }
          if( amin.equals(amax) ) return;
          if( width == 0 ) width = getAxisWidth(g);

          lg = g.create();

          if( force_end_labels ) {
              minimum = label_start;
              maximum = minimum + (label_count-1)*label_step;
          }

          if( orientation == HORIZONTAL) {
               drawHAxis(lg);
          } else {
               drawVAxis(lg);
          }


     }


/******************
** Protected Methods
******************/

     protected void drawHAxis(Graphics g) {
          Graphics lg;
          int i;
          int j;
          int x0,y0,x1,y1;
          int direction;
          int offset;
          double minor_step;
          
          Color c;

          double vmin = minimum*1.001;
          double vmax = maximum*1.001;

          double scale  = (amax.x - amin.x)/(maximum - minimum);
          double val;
          double minor;

//          System.out.println("Drawing Horizontal Axis!");


          if( axiscolor != null) g.setColor(axiscolor);

          g.drawLine(amin.x,amin.y,amax.x,amax.y);
          
          if(position == TOP )     direction =  1;
          else                     direction = -1;

          minor_step = label_step/(minor_tic_count+1);
          val = label_start;
          for(i=0; i<label_count; i++) {
              if( val >= vmin && val <= vmax ) {
                 y0 = amin.y;
                 x0 = amin.x + (int)( ( val - minimum ) * scale);
                 if( Math.abs(label_value[i]) <= 0.0001 && drawzero ) {
                      c = g.getColor();
                      if(zerocolor != null) g.setColor(zerocolor);
                      g.drawLine(x0,y0,x0,y0+data_window.height*direction);
                      g.setColor(c);
                      
                 } else 
                 if( drawgrid ) {
                      c = g.getColor();
                      if(gridcolor != null) g.setColor(gridcolor);
                      g.drawLine(x0,y0,x0,y0+data_window.height*direction);
                      g.setColor(c);
                 }
                 x1 = x0;
                 y1 = y0 + major_tic_size*direction;
                 g.drawLine(x0,y0,x1,y1);
              }

              minor = val + minor_step;
              for(j=0; j<minor_tic_count; j++) {
                 if( minor >= vmin && minor <= vmax ) {
                    y0 = amin.y;
                    x0 = amin.x + (int)( ( minor - minimum ) * scale);
                    if( drawgrid ) {
                      c = g.getColor();
                      if(gridcolor != null) g.setColor(gridcolor);
                      g.drawLine(x0,y0,x0,y0+data_window.height*direction);
                      g.setColor(c);
                    }
                    x1 = x0;
                    y1 = y0 + minor_tic_size*direction;
                    g.drawLine(x0,y0,x1,y1);
                 }
                minor += minor_step;
              }

              val += label_step;
          }


          if(position == TOP ) {
             offset = - label.getLeading(g) - label.getDescent(g);
          } else {
             offset = + label.getLeading(g) + label.getAscent(g);
          }


          val = label_start;
          for(i=0; i<label_count; i++) {
              if( val >= vmin && val <= vmax ) {
                 y0 = amin.y + offset;
                 x0 = amin.x + (int)(( val - minimum ) * scale);
                 label.setText(label_string[i]);
                 label.draw(g,x0,y0,TextLine.CENTER);
              }
              val += label_step;
          }


          if( !exponent.isNull() ) {
             if(position == TOP ) {
                y0 = amin.y - label.getLeading(g) 
                            - label.getDescent(g)
                            - exponent.getLeading(g)
                            - exponent.getDescent(g);
             } else {
                y0 = amax.y + label.getLeading(g) 
                            + label.getAscent(g)
                            + exponent.getLeading(g)
                            + exponent.getAscent(g);
             }

              x0 = amax.x;
              
              exponent.draw(g,x0,y0,TextLine.LEFT);

          }

          if( !title.isNull() ) {
             if(position == TOP ) {
                y0 = amin.y - label.getLeading(g) 
                            - label.getDescent(g)
                            - title.getLeading(g)
                            - title.getDescent(g);
             } else {
                y0 = amax.y + label.getLeading(g) 
                            + label.getAscent(g)
                            + title.getLeading(g)
                            + title.getAscent(g);
             }

              x0 = amin.x + ( amax.x - amin.x)/2;
              
              title.draw(g,x0,y0,TextLine.CENTER);

          }

     }


     protected void drawVAxis(Graphics g) {
          Graphics lg;
          int i;
          int j;
          int x0,y0,x1,y1;
          int direction;
          int offset = 0;
          double minor_step;
          double minor;
          Color c;
          
          FontMetrics fm;
          Color gc = g.getColor();
          Font  gf = g.getFont();

          double vmin = minimum*1.001;
          double vmax = maximum*1.001;

          double scale  = (amax.y - amin.y)/(maximum - minimum);
          double val;

//          System.out.println("Drawing Vertical Axis!");


          if( axiscolor != null) g.setColor(axiscolor);

          g.drawLine(amin.x,amin.y,amax.x,amax.y);

          if(position == RIGHT )     direction = -1;
          else                       direction =  1;

          minor_step = label_step/(minor_tic_count+1);
          val = label_start;
          for(i=0; i<label_count; i++) {
              if( val >= vmin && val <= vmax ) {
                 x0 = amin.x;
                 y0 = amax.y - (int)( ( val - minimum ) * scale);
                 if( Math.abs(label_value[i]) <= 0.0001 && drawzero ) {
                      c = g.getColor();
                      if(zerocolor != null) g.setColor(zerocolor);
                      g.drawLine(x0,y0,x0+data_window.width*direction,y0);
                      g.setColor(c);
                 } else
                 if( drawgrid ) {
                      c = g.getColor();
                      if(gridcolor != null) g.setColor(gridcolor);
                      g.drawLine(x0,y0,x0+data_window.width*direction,y0);
                      g.setColor(c);
                 }
                 x1 = x0 + major_tic_size*direction;
                 y1 = y0;
                 g.drawLine(x0,y0,x1,y1);
              }

              minor = val + minor_step;
              for(j=0; j<minor_tic_count; j++) {
                 if( minor >= vmin && minor <= vmax ) {
                    x0 = amin.x;
                    y0 = amax.y - (int)( ( minor - minimum ) * scale);
                    if( drawgrid ) {
                      c = g.getColor();
                      if(gridcolor != null) g.setColor(gridcolor);
                      g.drawLine(x0,y0,x0+ data_window.width*direction,y0);
                      g.setColor(c);
                    }
                    x1 = x0 + minor_tic_size*direction;
                    y1 = y0;
                    g.drawLine(x0,y0,x1,y1);
                 }
                minor += minor_step;
              }
              val += label_step;
          }


          val = label_start;
          for(i=0; i<label_count; i++) {
              if( val >= vmin && val <= vmax ) {
                 x0 = amin.x + offset;
                 y0 = amax.y - (int)(( val - minimum ) * scale) + 
                             label.getAscent(g)/2;
                 
                 if(position == RIGHT ) {
                    label.setText(" "+label_string[i]);
                    label.draw(g,x0,y0,TextLine.LEFT);
                 } else {
                    label.setText(label_string[i]+" ");
                    label.draw(g,x0,y0,TextLine.RIGHT);
		  }
              }
              val += label_step;
          }



          if( !exponent.isNull() ) {

             y0 = amin.y;

             if(position == RIGHT ) {
                 x0  =  amin.x + max_label_width + exponent.charWidth(g,' ');
                 exponent.draw(g,x0,y0,TextLine.LEFT);
             } else {
                 x0 =  amin.x - max_label_width - exponent.charWidth(g,' ');
                 exponent.draw(g,x0,y0,TextLine.RIGHT);
             }
 
          }

          if( !title.isNull() ) {

             y0 = amin.y + (amax.y-amin.y)/2;

             if(position == RIGHT ) {
                 x0 =  amin.x + max_label_width + title.charWidth(g,' ');
                 title.draw(g,x0,y0,TextLine.LEFT);
             } else {
                 x0 =  amin.x - max_label_width - title.charWidth(g,' ');
                 title.draw(g,x0,y0,TextLine.RIGHT);
             }
 
          }


 }








      protected void attachXdata( DataSet d ) {

            dataset.addElement(d);
            d.xaxis = this;

            if( dataset.size() == 1 ) {
                  minimum = d.xmin;
                  maximum = d.xmax;
            } else {
               if(minimum > d.xmin) minimum = d.xmin;
               if(maximum < d.xmax) maximum = d.xmax;
            }

      }

      protected void attachYdata( DataSet d ) {

            dataset.addElement(d);
            d.yaxis = this;

            if( dataset.size() == 1 ) {
                  minimum = d.ymin;
                  maximum = d.ymax;
            } else {
               if(minimum > d.ymin) minimum = d.ymin;
               if(maximum < d.ymax) maximum = d.ymax;
            }

      }


      protected void calculateGridLabels() {
        double val;
        int i;
        int j;

        
        if (Math.abs(minimum) > Math.abs(maximum) ) 
         label_exponent = ((int)Math.floor(log10(Math.abs(minimum))/3.0) )*3;
        else
         label_exponent = ((int)Math.floor(log10(Math.abs(maximum))/3.0) )*3;

        label_step = RoundUp( (maximum-minimum)/guess_label_number );
        label_start = Math.floor( minimum/label_step )*label_step;

        val = label_start;
        label_count = 1;
        while(val < maximum) { val += label_step; label_count++; }

        label_string = new String[label_count];
        label_value  = new double[label_count];


//      System.out.println("label_step="+label_step);
//      System.out.println("label_start="+label_start);
//      System.out.println("label_count="+label_count);
//      System.out.println("label_exponent"+label_exponent);

           
        for(i=0; i<label_count; i++) {
            val = label_start + i*label_step;

            if( label_exponent< 0 ) {
                  for(j=label_exponent; j<0;j++) { val *= 10; }
            } else {
                  for(j=0; j<label_exponent;j++) { val /= 10; }
            }

            label_string[i] = String.valueOf(val);
            label_value[i] = (double)val;
        }

      }

/******************
** Private Methods
******************/

      private double RoundUp( double val ) {
          int exponent;
          int i;

          exponent = (int)(Math.floor( log10(val) ) );

          if( exponent < 0 ) {
             for(i=exponent; i<0; i++) { val *= 10.0; }
          } else {
             for(i=0; i<exponent; i++) { val /= 10.0; }
          }

          if( val > 5.0 )     val = 10.0;
          else
          if( val > 2.0 )     val = 5.0;
          else
          if( val > 1.0 )     val = 2.0;
          else
                              val = 1.0;

          if( exponent < 0 ) {
             for(i=exponent; i<0; i++) { val /= 10.0; }
          } else {
             for(i=0; i<exponent; i++) { val *= 10.0; }
          }
          
          return val;

      }                  


      private double log10( double a ) throws ArithmeticException {
           return Math.log(a)/2.30258509299404568401;
      }



}


