/**
 * MyDomParser.java
 * @author Thomas Rowland
 * @version 02-08-03
 */
package xml.musicCatalog;

import org.apache.xerces.parsers.DOMParser;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;

/*
 * This program demonstrates how to parse an XML file using the
 * Xerces parser implementation directly, and the DOM api.
 * Does not use JAXP.
 * Parses a MusicCatalog XML file and prints out the content.
 */

public class MyDomParser {

    private static DOMParser parser = null;
    private static PrintStream out = null;

    static final int ELEMENT_NODE = 1;
    static final int ATTR_NODE = 2;
    static final int TEXT_NODE = 3;

    /*
     *	Parse an xml file.
     */
    public static void parse(String uri, OutputStream os)
            throws Exception {

        try {
            out = new PrintStream(os);
            parser = new DOMParser();
            parser.parse(uri);
            Document domDocument = parser.getDocument();
            Element root = domDocument.getDocumentElement();
            traverse(root);
            out.flush();
        } catch (SAXException e) {
            out.flush();
            // get the wrapped exception
            Exception ex = e.getException();

            if (ex != null) {
                ex.printStackTrace();
            }
            throw new Exception(
                    "** SAXException:\n" + e.getMessage());
        } catch (IOException e) {
            out.flush();
            throw new Exception(
                    "IOException:\n" + e.getMessage());
        }
    }


    /*
     * Recursive method which traverses the DOM Document
     * and outputs the content to the specified OutputStream.
     */
    public static void traverse(Node elem) {
        try {
            // handle the attributes
            if (elem.hasAttributes()) {
                NamedNodeMap attrs = elem.getAttributes();
                int alength = attrs.getLength();
                for (int i = 0; i < alength; i++) {
                    Attr attr = (Attr) attrs.item(i);
                    out.println(attr.getName() + ":\t" + attr.getValue());
                }
            }

            // handle the child nodes, printing out <Item> elements
            if (elem.hasChildNodes()) {
                NodeList children = elem.getChildNodes();
                int length = children.getLength();

                for (int i = 0; i < length; i++) {
                    Node n = children.item(i);
                    String name = n.getNodeName();

                    if (n.getNodeType() == ELEMENT_NODE) {
                        if (name.equals("Item"))
                            out.println("\n");
                        else
                            out.print(name + ":\t");
                        traverse(n); //recurse
                    } else if (n.getNodeType() == TEXT_NODE) {
                        String txt = n.getNodeValue().trim();
                        if (!txt.equals("")) {
                            out.println(txt);
                        }
                    }
                }
                return;
            }
        } catch (DOMException e) {
            System.out.println(
                    "*** DOMException\n" + e.getMessage());
        }
    }

}//