package sound;

/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */

import gui.layouts.VerticalLayout;
import gui.run.RunButton;
import gui.run.RunRadio;
import gui.run.RunSpinnerSlider;
import gui.run.RunTextField;
import math.fourierTransforms.FFT1d;
import sound.ulaw.UlawCodec;

import javax.swing.*;
import java.awt.*;

/**
 * Created by Robert Distinti.
 * User: default
 * Date: Jul 23, 2005
 * Time: 11 aM
 */
public class PitchShifter extends JDialog {
    private OscopePanel op=new OscopePanel();
    private RunSpinnerSlider slider;
    private double[] originalData=new Oscillator(440, 1000).getSineWave();  // set default data
    private int originalLength=originalData.length;
    private double[] filteredData;
    private double[] displayData;
    private UlawCodec playData;
    private RunRadio isNormal;
    private RunRadio isPSD;
    private RunRadio isReal;
    private RunRadio isImag;
    private ButtonGroup displayGroup=new ButtonGroup();
    private RunRadio isDougs;
    private RunRadio isBobs;
    private RunRadio isNone;
    private ButtonGroup shiftGroup=new ButtonGroup();

    private double[] copy(double []a){
        double out[] = new double[a.length];
        for (int i = 0; i <a.length; i++) {
            out[i] = a[i];
        }
        return out;
    }

    private double getDouble(RunTextField src){
        double rtn=Double.valueOf(src.getText()).doubleValue();
        return rtn;
    };

    private double logBase2(int length) {
        return (Math.log(length) / Math.log(2));
    }

    private double[] makePowerOfTwo(double[] in){
        int n = (int)Math.ceil(logBase2(in.length));
        double out[] = new double[1 << n];
        for (int i = 0; i < originalLength; i++) {
            out[i] = in[i];
        }
        return out;
    }

    private void scaleData(double scale_){
        double out[] = new double[originalData.length];
        for (int i = 0; i < originalData.length; i++) {
            out[i] = originalData[i]*scale_;
        }
        originalData=out;
    }

    private void open(){
        UlawCodec ulc = new UlawCodec();
        originalData=copy(ulc.getDoubleArray());
        System.out.println("number of samples "+originalData.length);
        dataChanged();

    }

    private void play(){
        if(playData!=null){
            playData.play();
        }
    }

    private void addNoise(double d[], double amp){
        for (int i = 0; i < originalLength; i++) {
            d[i] = d[i] + amp * (Math.random() - 0.5);
        }
    }

    private void dataChanged(){
        originalLength=originalData.length;
        originalData=makePowerOfTwo(originalData);
        shiftChanged();
    }


    private void shiftChanged(){
        double real[] = new double[originalData.length];
        double imag[] = new double[originalData.length];
        double real2[] = new double[originalData.length];
        double imag2[] = new double[originalData.length];
        real=copy(originalData);
        centering(real);
        // transform
        FFT1d f=new FFT1d();
        f.computeForwardFFT(real, imag);

        // perform scaling
        double r[] = f.getRealData();
        double im[] = f.getImaginaryData();
        for (int i = 0; i < r.length; i++) {
            r[i]/=(double)originalLength;
            im[i]/=(double)originalLength;
        }

        if(isDougs.isSelected()){
            // this method assumes all data is symetical about center
            // it simply uses the data from 0 to N/2 for both halves
            int len=r.length/2;
            int shift=slider.getValue();
            int destIdxL=0;
            int destIdxH=r.length-1;
            int srcIdx=(len-1)*shift/100;
            int cnt=len-srcIdx;
            for(int i=0; i<cnt; i++){
                real2[destIdxH]=real2[destIdxL]=r[srcIdx];
                imag2[destIdxH]=imag2[destIdxL]=im[srcIdx];
                // test to see if inverting the reflected imag data would solve problem
                //imag2[destIdxH]=-im[srcIdx];
                //imag2[destIdxL]=im[srcIdx];
                // end test code 
                destIdxH--;
                destIdxL++;
                srcIdx++;
            }
        }else if(isBobs.isSelected()){
            // this does not assume data symetry
            int len=r.length/2;
            int shift=slider.getValue();
            int destIdxL=0;
            int destIdxH=r.length-1;
            int srcIdxL=(len-1)*shift/100;
            int srcIdxH=(r.length-1)-(len-1)*shift/100;
            int cnt=len-srcIdxL;
            for(int i=0; i<cnt; i++){
                real2[destIdxL]=r[srcIdxL];
                real2[destIdxH]=r[srcIdxH];
                imag2[destIdxH]=im[srcIdxH];
                imag2[destIdxL]=im[srcIdxL];
                destIdxH--;
                destIdxL++;
                srcIdxL++;
                srcIdxH--;
            }
        }else{
            // just pass through
            real2=r;
            imag2=im;
        }



        // convert back
        f.computeBackwardFFT(real2, imag2);
        centering(real2);
        filteredData=f.getRealData();

        displayChanged();
    }

    public static void centering(double r[]) {
        int s = 1;
        for (int i = 0; i < r.length; i++) {
            s = -s;
            r[i] *= s;
        }
    }

    private void displayChanged(){
        double out[] = new double[originalLength];
        double imag[] = new double[filteredData.length];
        for (int i = 0; i < originalLength; i++) {
            out[i] = filteredData[i];
        }
        displayData=out;
        if(isPSD.isSelected()){
            out=copy(filteredData);
            FFT1d f=new FFT1d();
            centering(out);
            f.computeForwardFFT(out, imag);

            double psd[]= f.getPSD() ;
            op.setData(psd);
        }else if (isImag.isSelected()){
            out=copy(filteredData);
            FFT1d f=new FFT1d();
            f.computeForwardFFT(out, imag);
            double d[]= f.getImaginaryData() ;
            // need to scale it
            for (int i = 0; i < d.length; i++) {
                d[i]/=(double)originalLength;
            }
            op.setData(d);
        }else if (isReal.isSelected()){
            out=copy(filteredData);
            FFT1d f=new FFT1d();
            f.computeForwardFFT(out, imag);
            double d[]= f.getRealData() ;
            // need to scale it
            for (int i = 0; i < d.length; i++) {
                d[i]/=(double)originalLength;
            }
            op.setData(d);
        }else{
            op.setData(displayData);
        }
        playData=new UlawCodec(displayData);
    }


    public PitchShifter(){
        setModal(false);
        setTitle("Pitch Shifter Ver2");
        JPanel jp = new JPanel();
        Container c = getContentPane();
        c.setLayout(new BorderLayout());
        c.add(jp,BorderLayout.NORTH);
        c.add(op,BorderLayout.CENTER);
        jp.add(new RunButton("[Open"){
            public void run(){
                open();
            }
        });
        jp.add(slider=new RunSpinnerSlider(new SpinnerNumberModel(0,0,100,10)){
            public void run(){
                shiftChanged();
            }
        });

        jp.add(new RunButton("[Play"){
            public void run(){
                play();
            }
        });

        JPanel jp2 = new JPanel();
        jp.add(jp2);
        jp2.setLayout(new VerticalLayout());
        jp2.add(isNormal=new RunRadio("[Normal"){
            public void run(){
                displayChanged();
            }
        });
        jp2.add(isPSD=new RunRadio("PS[D"){
            public void run(){
                displayChanged();
            }
        });
        jp2.add(isImag=new RunRadio("[Imag"){
            public void run(){
                displayChanged();
            }
        });
        jp2.add(isReal=new RunRadio("[Real"){
            public void run(){
                displayChanged();
            }
        });


        JPanel jp3 = new JPanel();
        jp.add(jp3);
        jp3.setLayout(new VerticalLayout());
        jp3.add(isNone=new RunRadio("No Shift"){
            public void run(){
                shiftChanged();
            }
        });
        jp3.add(isDougs=new RunRadio("Use Doug's"){
            public void run(){
                shiftChanged();
            }
        });
        jp3.add(isBobs=new RunRadio("Use Bob's"){
            public void run(){
                shiftChanged();
            }
        });

        originalData=new Oscillator(440, 2048).getSineWave();
        scaleData(0.5);
        dataChanged();

        displayGroup.add(isNormal);
        displayGroup.add(isPSD);
        displayGroup.add(isImag);
        displayGroup.add(isReal);
        isNormal.setSelected(true);

        shiftGroup.add(isNone);
        shiftGroup.add(isDougs);
        shiftGroup.add(isBobs);
        isNone.setSelected(true);



        setSize(600, 480);
        setVisible(true);
    }


    public static void main(String[] args) {
        new PitchShifter();
    }
}

