/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * VelocityVerlet:  A velocity Verlet method ODE solver.
 *
 * The velocity Verlet algorithm is a self-starting equivalent of the Verlet algorithm.
 * It assumes a constant acceleration to estimate the final position
 * and an average accleration to estimate the final velocity.
 * The position is first updated, the force is calcualted at the new position,
 * and then the velocity is updated.
 *
 * x(n+1) = x(n) + v(n)* dt + a(n)*dt*dt/2
 * a_est=F(x(n+1),v(n),t)/m
 * v(n+1) = v(n) + (a(n)+a_est)*dt/2
 *
 * CAUTION! The ODE interface produces an inefficient implementation of the velocity Verlet
 * algorithm since the force is evaluated twice.  The velocity Verlet algorithm is
 * usually coded directly into the model so that the force need only be evaluated
 * once.
 *
 * CAUTION! This implementation assumes that the state vector has 2*N + 1 variables.
 * These variables alternate between position and velocity with the last variable being time.
 * That is, the  state vector is ordered as follows:
 *
 * x1, d x1/dt, x2, d x2/dt, x3, d x3/dt ..... xN, d xN/dt, t
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class VelocityVerlet implements ODESolver {

  private double   stepSize = 0.1;
  private int      numEqn   = 0;
  private double[] rate;   // rate with orginal positions
  private double[] rate2;  // rate with new positions
  private ODE      ode;

  /**
   * Constructs the VelocityVerlet ODESolver for a system of ordinary  differential equations.
   *
   * @param _ode the system of differential equations.
   */
  public VelocityVerlet(ODE _ode) {
    ode = _ode;
    initialize(stepSize);
  }

  /**
   * Initializes the ODE solver.
   *
   * Two temporary rate arrays are allocated.
   * The number of differential equations is determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
  public void initialize(double _stepSize) {
    stepSize = _stepSize;
    double state[] = ode.getState();
    if(state == null) {  // state vector not defined.
      numEqn = 0;
    } else {
      numEqn = state.length;
    }
    rate  = new double[numEqn];
    rate2 = new double[numEqn];
  }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
  public double step() {
    // state[]: x1, d x1/dt, x2, d x2/dt .... xN, d xN/dt, t
    double[] state = ode.getState();
    if(state.length != numEqn) {
      initialize(stepSize);
    }
    ode.getRate(state, rate);  // get the initial rate
    double dt2 = stepSize * stepSize;  // the step size squared
    // increment the positions using the velocity and acceleration
    for(int i = 0; i < numEqn - 1; i += 2) {
      state[i] = state[i] + stepSize * rate[i] + dt2 * rate[i + 1] / 2;
    }
    ode.getRate(state, rate2);  // get rate using the new positions
    for(int i = 1; i < numEqn - 1; i += 2) {
      // increment the velocities with the average rate
      state[i] = state[i] + stepSize * (rate[i] + rate2[i]) / 2.0;
    }
    state[numEqn - 1] += rate[numEqn - 1] * stepSize;  // increment the independent parameter
    return stepSize;
  }

  /**
   * Sets the step size.
   *
   * The step size remains fixed in this algorithm.
   *
   * @param _stepSize
   */
  public void setStepSize(double _stepSize) {
    stepSize = _stepSize;
  }

  /**
   * Gets the step size.
   *
   * The stepsize is constant in this algorithm.
   *
   * @return the step size
   */
  public double getStepSize() {
    return stepSize;
  }
}
