package math.numerics;

/**
 * Class Interpolation defines simple interpolation algorithms.
 * This class cannot be subclassed or instantiated because all methods are static.
 *
 * @author Wolfgang Christian
 */

public class Interpolation {
  private Interpolation() {
  }

  /**
   * Linear interpolation at a single point x.
   * @param x double
   * @param x0 double
   * @param x1 double
   * @param y0 double
   * @param y1 double
   * @return double
   */
  static public double linear(final double x, final double x0, final double x1,
                              final double y0, final double y1) {
    if ( (x1 - x0) == 0)return (y0 + y1) / 2;
    else return y0 + (x - x0) * (y1 - y0) / (x1 - x0);
  }

  /**
   * Legendre polynomial interpolation at a single point x.
   *
   * Because Legendre polynomials tend to be ill behaved, this method should be used with care.
   *
   * A LegendrePolynomial object should be used if multiple interpolations
   * are to be performed using the same data.
   *
   * @param x double
   * @param xd double[] the x data
   * @param yd double[] the y data
   * @return double
   */
  static public double legendre(final double x, final double[] xd, final double yd[]) {
    if (xd.length != yd.length) {
      throw new IllegalArgumentException("Arrays must be of equal length.");
    }
    double sum = 0;
    for(int i=0, n=xd.length; i<n; i++){
      if(x-xd[i]==0)return x;
      double product=yd[i];
      for(int j=0;j<n; j++){
        if(i==j || (xd[i]-xd[j]==0)) continue;
        product *= (x-xd[i])/(xd[i]-xd[j]);
      }
      sum+=product;
    }
    return sum;
  }

}
